﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/hid/hid_ExternalBus.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hidbus/detail/hidbus_Log.h>

#include "hidbus_SchedulerTask.h"
#include "hidbus_ServerUtil.h"
#include "../detail/hidbus_InternalUtility.h"

namespace nn { namespace hidbus { namespace server {

namespace {

const auto TimerInterval = ::nn::TimeSpan::FromMilliSeconds(5);

} // namespace

SchedulerTask::SchedulerTask() NN_NOEXCEPT
    : m_TimerActivateMutex(false)
    , m_TimerEvent()
    , m_UnregisterFlag(false)
{
    for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
    {
        m_SteteMutex[i] = NN_OS_SDK_MUTEX_INITIALIZER();
        m_IsFocusChanged[i] = false;
    }

    // 何もしない
}

SchedulerTask::~SchedulerTask() NN_NOEXCEPT
{
    // 何もしない
}

void SchedulerTask::ActivatePeriodicTimer() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TimerActivateMutex)> locker(m_TimerActivateMutex);
    if (m_PeriodicTimerActivationCount.IsZero())
    {
        // 新規に要求された場合のみ Timer を発動
        nn::os::StartPeriodicTimerEvent(m_TimerEvent.GetBase(), 0, TimerInterval);
        NN_DETAIL_HIDBUS_INFO_V1("Start Timer\n");
    }
    // アクティブ化回数をインクリメント
    ++m_PeriodicTimerActivationCount;
    NN_DETAIL_HIDBUS_INFO_V1("PeriodicTimerActivate ++\n");
}

void SchedulerTask::DeactivatePeriodicTimer() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TimerActivateMutex)> locker(m_TimerActivateMutex);
    // アクティブ化回数をデクリメント
    --m_PeriodicTimerActivationCount;
    NN_DETAIL_HIDBUS_INFO_V1("PeriodicTimerActivate --\n");
    if (m_PeriodicTimerActivationCount.IsZero())
    {
        // 新規に要求された場合のみ Timer を発動
        nn::os::StopTimerEvent(m_TimerEvent.GetBase());
        NN_DETAIL_HIDBUS_INFO_V1("Stop Timer\n");
    }
}

::nn::Result SchedulerTask::Initialize(nn::hidbus::server::ResourceHolder* pResourceHolder) NN_NOEXCEPT
{
    // State の初期化
    for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
    {
        m_Status[i].handle._storage = 0;
        m_Status[i].state = HidbusState_NotActivated;
    }

    NN_RESULT_DO(this->ActivateThread());
    m_pResourceHolder = pResourceHolder;
    NN_RESULT_SUCCESS;
}

::nn::Result SchedulerTask::Finalize() NN_NOEXCEPT
{
    NN_RESULT_DO(this->DeactivateThread());
    NN_RESULT_SUCCESS;
}

::nn::Result SchedulerTask::Activate(BusHandle handle) NN_NOEXCEPT
{
    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    ActivatePeriodicTimer();

    m_Status[internalNumber].handle = hidHandle;

    ::std::lock_guard<decltype(m_SteteMutex[internalNumber])> locker(m_SteteMutex[internalNumber]);

    if (m_Status[internalNumber].state == HidbusState_NotActivated)
    {
        m_Status[internalNumber].state = HidbusState_Activated;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SchedulerTask::Deactivate(BusHandle handle) NN_NOEXCEPT
{
    auto hidHandle = ConvertHandleFromHidbusToHid(handle);
    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    DeactivatePeriodicTimer();

    ::std::lock_guard<decltype(m_SteteMutex[internalNumber])> locker(m_SteteMutex[internalNumber]);

    m_Status[internalNumber].handle = hidHandle;
    m_Status[internalNumber].state = HidbusState_NotActivated;

    NN_RESULT_SUCCESS;
}


void SchedulerTask::StartPollingMode(nn::hidbus::JoyPollingMode mode, BusHandle handle) NN_NOEXCEPT
{
    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);
    GetResourceHolder().GetStatusManager()->SetPollingMode(static_cast<int>(mode), internalNumber);
    ::std::lock_guard<decltype(m_SteteMutex[internalNumber])> locker(m_SteteMutex[internalNumber]);
    m_Status[internalNumber].state = HidbusState_PollingMode;
}

void SchedulerTask::StopPollingMode(BusHandle handle) NN_NOEXCEPT
{
    auto internalNumber = nn::hidbus::detail::GetInternalIndexFromHandle(handle);

    ::std::lock_guard<decltype(m_SteteMutex[internalNumber])> locker(m_SteteMutex[internalNumber]);
    if (m_Status[internalNumber].state == HidbusState_PollingMode)
    {
        m_Status[internalNumber].state = HidbusState_Attached;
    }
}

void SchedulerTask::AddFocusChangeTask(int index) NN_NOEXCEPT
{
    if (m_pResourceHolder->GetStatusManager()->IsEnabled(index))
    {
        ::std::lock_guard<decltype(m_SteteMutex[index])> locker(m_SteteMutex[index]);
        m_IsFocusChanged[index] = true;
    }
}

void SchedulerTask::ExcuteUnregister() NN_NOEXCEPT
{
    bool IsForcusChangedAny = false;
    for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
    {
        ::std::lock_guard<decltype(m_SteteMutex[i])> locker(m_SteteMutex[i]);
        if (m_IsFocusChanged[i] == true)
        {
            IsForcusChangedAny = true;
        }
    }

    if (!IsForcusChangedAny)
    {
        // 強制的に ActivationCount を 0 にしてタイマーを止める。
        m_PeriodicTimerActivationCount.SetZero();
        nn::os::StopTimerEvent(m_TimerEvent.GetBase());
        NN_DETAIL_HIDBUS_INFO("Force Stop Timer\n");

        // 登録されていた AURID を無効化し、内部ステートを NotActivated へ変更
        for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
        {
            auto hidbusHandle = ConvertHandleFromHidToHidbus(m_Status[i].handle);
            GetResourceHolder().SetAppletResourceUserId(nn::applet::AppletResourceUserId::GetInvalidId(), hidbusHandle);
            GetResourceHolder().SetInitializedAppletResourceUserId(nn::applet::AppletResourceUserId::GetInvalidId(), hidbusHandle);
            m_Status[i].state = HidbusState_NotActivated;
            GetResourceHolder().SetResourceInitialized(false, i);
        }

        // SharedMemory に作成された Status を Clear
        m_pResourceHolder->ClearSharedMemory();
        NN_DETAIL_HIDBUS_INFO("Clear SharedMemory\n");

        m_UnregisterFlag = false;
    }
}

void SchedulerTask::UnregisterApplication(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
    {
        auto internalAruid = GetResourceHolder().GetAppletResourceUserId(i);
        if (internalAruid == aruid)
        {
            AddFocusChangeTask(i);

            // 最後に強制的に Timer を止めるためのフラグを立たせる
            // TODO : 今はアプリ前提なのでここでよいが、アプリ以外も対応したら、Timer 等の処理は分岐して処理する
            NN_DETAIL_HIDBUS_INFO_V1("Set Unregister Flag\n");
            m_UnregisterFlag = true;
        }

        // Initialize だけ実行されているものに対しては m_UnregisterFlag だけ立てる
        if (GetResourceHolder().GetInitializedAppletResourceUserId(i) == aruid)
        {
            m_UnregisterFlag = true;
        }
    }

    if (m_UnregisterFlag)
    {
        // 1つも Enable 状態のタスクがない場合、下記 API 内で終了処理が即座に行われる。
        // Enable 状態のタスクがある場合、下記 API 内では何も行われず、TaskSchduler 内で終了処理が行われる。
        ExcuteUnregister();
    }
}

void SchedulerTask::DeactivateTransferMemory(BusHandle handle) NN_NOEXCEPT
{
    auto transferMemoryType = m_pResourceHolder->GetTransferMemoryType(handle);

    nn::os::UnmapTransferMemory(transferMemoryType);
    nn::os::DestroyTransferMemory(transferMemoryType);
}

void SchedulerTask::SetThreadName(
    ::nn::os::ThreadType* pThreadType) NN_NOEXCEPT
{
    ::nn::os::SetThreadName(pThreadType, NN_SYSTEM_THREAD_NAME(hidbus, SchedulerTaskThread));
}

void SchedulerTask::LinkEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    // Link順が多重待ち時の優先度となるので注意が必要
    m_TimerEvent.Link(pMultiWait);
}

void SchedulerTask::UnlinkEvent() NN_NOEXCEPT
{
    m_TimerEvent.Unlink();
}

void SchedulerTask::HandleInvalidHandleResult(int index) NN_NOEXCEPT
{
    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    ::std::lock_guard<decltype(m_SteteMutex[index])> locker(m_SteteMutex[index]);
    // Activate 状態で Invalid の場合、コントローラーの切断が行っているので、Finalize 相当の処理を行う
    // TODO : もうちょっとわかりやすくしておかないと事故りそう。
    DeactivatePeriodicTimer();
    GetResourceHolder().GetStatusManager()->SetPollingModeState(false, index);
    m_pResourceHolder->GetStatusManager()->SetEnableState(false, index);
    m_pResourceHolder->GetStatusManager()->SetAttachmentState(false, nn::hidbus::ResultInvalidHandle(), index);
    auto hidbusHandle = ConvertHandleFromHidToHidbus(m_Status[index].handle);
    GetResourceHolder().SetInitializedAppletResourceUserId(nn::applet::AppletResourceUserId::GetInvalidId(), hidbusHandle);
    GetResourceHolder().SetResourceInitialized(false, index);
    m_Status[index].state = HidbusState_NotActivated;

    // FocusChange 発生時に同時に InvalidHandle が発生すると、 m_IsFocusChanged フラグが disable にならないため、ここで落とす。
    m_IsFocusChanged[index] = false;
}

void SchedulerTask::HandleActivatedState(int index) NN_NOEXCEPT
{
    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    // 接続状態を確認する
    bool isConnected;
    auto result = nn::hid::IsExternalBusDeviceConnected(&isConnected, m_Status[index].handle);
    if (result.IsSuccess())
    {
        if (isConnected)
        {
            // 接続されたら Attached へ遷移
            ::std::lock_guard<decltype(m_SteteMutex[index])> locker(m_SteteMutex[index]);
            NN_DETAIL_HIDBUS_INFO("Got Attached\n");
            auto hidbusResult = ConvertHidResultToHidBusResult(result);
            m_pResourceHolder->GetStatusManager()->SetAttachmentState(true, hidbusResult, index);
            m_Status[index].state = HidbusState_Attached;
        }
        else
        {
            // 接続されてなくても、 Result は更新する。(ために State を更新)
            ::std::lock_guard<decltype(m_SteteMutex[index])> locker(m_SteteMutex[index]);
            auto hidbusResult = ConvertHidResultToHidBusResult(result);
            m_pResourceHolder->GetStatusManager()->SetAttachmentState(false, hidbusResult, index);
        }
    }
    else if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(result))
    {
        HandleInvalidHandleResult(index);
    }
}

void SchedulerTask::HandleAttachedState(int index, bool isForceDisable) NN_NOEXCEPT
{
    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    bool isConnected = true;
    if (!isForceDisable)
    {
        auto result = nn::hid::IsExternalBusDeviceConnected(&isConnected, m_Status[index].handle);
        if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(result))
        {
            HandleInvalidHandleResult(index);
        }
    }

    if (!isConnected || isForceDisable)
    {
        if (m_pResourceHolder->GetStatusManager()->IsEnabled(index))
        {
            // 切断されているので、 Disable にする。
            // TODO : ここで行うと、ポーリング処理がブロックされるため、別の RequestHandler に追い出す。
            // Disable 時は Required Version はチェックされない。
            auto enableResult = nn::hid::EnableExternalBusDevice(m_Status[index].handle, false, 0);
            if (enableResult.IsSuccess())
            {
                ::std::lock_guard<decltype(m_SteteMutex[index])> locker(m_SteteMutex[index]);
                m_pResourceHolder->GetStatusManager()->SetEnableState(false, index);
                NN_DETAIL_HIDBUS_INFO("Disable Power\n");
                m_Status[index].state = HidbusState_Activated;
                if (isForceDisable)
                {
                    m_IsFocusChanged[index] = false;
                }
                else
                {
                    auto hidbusResult = ConvertHidResultToHidBusResult(enableResult);
                    m_pResourceHolder->GetStatusManager()->SetAttachmentState(false, hidbusResult, index);
                }
            }
            else if (nn::hid::ResultExternalBusDeviceReadyTimeout::Includes(enableResult))
            {
                NN_DETAIL_HIDBUS_WARN("DeviceReadyTimeout occur. Retry.\n");
                // Tera のステート変更のタイムアウトでここに来る。
                // TORIAEZU : Retry するために、ステートを変更しない。
            }
            else if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(enableResult))
            {
                HandleInvalidHandleResult(index);
            }
            else if (nn::hid::ResultNoExternalBusFoundOnNpad::Includes(enableResult))
            {
                // 切断処理中に handle が有効なまま、ジョイコンが detach された場合、InvalidHandle ではなく、こちらのエラーになる。
                // Invalid 相当の処理を行う
                HandleInvalidHandleResult(index);
            }
            else
            {
                // Disable 時に他のエラーは返らない想定なので、雑に今は ABORT
                NN_ABORT_UNLESS_RESULT_SUCCESS(enableResult);
            }
        }
    }
}

void SchedulerTask::HandlePollingModeState(int index, bool isForceDisable) NN_NOEXCEPT
{
    GetResourceHolder().LockInternalStateMutex(index);
    NN_UTIL_SCOPE_EXIT
    {
        GetResourceHolder().UnlockInternalStateMutex(index);
    };

    bool isConnected = true;
    if (!isForceDisable)
    {
        // 接続状態を確認する
        auto result = nn::hid::IsExternalBusDeviceConnected(&isConnected, m_Status[index].handle);
        if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(result))
        {
            auto pTransferMemoryMutex = GetResourceHolder().GetTransferMemoryMutexPointer(index);
            ::std::lock_guard<decltype(*pTransferMemoryMutex)> locker(*pTransferMemoryMutex);

            // TransferMemory の破棄
            auto hidbusHandle = ConvertHandleFromHidToHidbus(m_Status[index].handle);
            DeactivateTransferMemory(hidbusHandle);
            HandleInvalidHandleResult(index);
            return;
        }
    }

    if (!isConnected || isForceDisable)
    {
        // PollingMode からの Detach はまず Polling Mode を Disable にする
        auto pollingModeDisableResult = nn::hid::DisablePollingReceiveModeForAttachmentDevice(m_Status[index].handle);
        if (pollingModeDisableResult.IsSuccess())
        {
            auto pTransferMemoryMutex = GetResourceHolder().GetTransferMemoryMutexPointer(index);
            ::std::lock_guard<decltype(*pTransferMemoryMutex)> locker(*pTransferMemoryMutex);

            // TransferMemory の破棄
            auto hidbusHandle = ConvertHandleFromHidToHidbus(m_Status[index].handle);
            DeactivateTransferMemory(hidbusHandle);
            GetResourceHolder().GetStatusManager()->SetPollingModeState(false, index);
            NN_DETAIL_HIDBUS_INFO("PollingMode Disable.\n");

            // Attached の ハンドリングの実施
            HandleAttachedState(index, isForceDisable);
        }
        else if (nn::hid::ResultExternalBusDeviceInvalidHandle::Includes(pollingModeDisableResult))
        {
            auto pTransferMemoryMutex = GetResourceHolder().GetTransferMemoryMutexPointer(index);
            ::std::lock_guard<decltype(*pTransferMemoryMutex)> locker(*pTransferMemoryMutex);

            // TransferMemory の破棄
            auto hidbusHandle = ConvertHandleFromHidToHidbus(m_Status[index].handle);
            DeactivateTransferMemory(hidbusHandle);
            HandleInvalidHandleResult(index);
        }
        else if (nn::hid::ResultExternalBusDeviceReadyTimeout::Includes(pollingModeDisableResult))
        {

            NN_DETAIL_HIDBUS_WARN("DeviceReadyTimeout occur. Retry.\n");
            // Tera のステート変更のタイムアウトでここに来る。
            // TORIAEZU : Retry するために、ステートを変更しない。
        }
        else
        {
            // Disable 時に他のエラーは返らない想定なので、雑に今は ABORT
            NN_ABORT_UNLESS_RESULT_SUCCESS(pollingModeDisableResult);
        }
    }
}

void SchedulerTask::ExcuteFocusChangedTask(int index) NN_NOEXCEPT
{
    NN_DETAIL_HIDBUS_INFO("ExcuteFocusChangedTask\n");
    switch (m_Status[index].state)
    {
    case HidbusState_NotActivated:
        // NotActivated 状態でこのタスクが積まれることはないはずだが、フラグが立ったままだと暴走するため、
        // ワーニングのログ表示を行い、フラグを落とす
        NN_DETAIL_HIDBUS_WARN("ExcuteFocusChangedTask occurs in NotActivated\n");
        m_IsFocusChanged[index] = false;
        break;
    case HidbusState_Activated:
        // タイミングによって、Enable 状態でタスクが積まれ、ループが回ってきたときにデバイスが抜けて Disable になることがあるので、
        // フラグだけ落として抜ける
        m_IsFocusChanged[index] = false;
        break;
    case HidbusState_Attached:
        HandleAttachedState(index, true);
        break;
    case HidbusState_PollingMode:
        HandlePollingModeState(index, true);
        break;
    default:NN_UNEXPECTED_DEFAULT;
    }
}

void SchedulerTask::HandleEvent(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    if (waitId == m_TimerEvent.GetWaitId())
    {
        nn::os::ClearTimerEvent(m_TimerEvent.GetBase());
        for (int i = 0; i < nn::hidbus::detail::MaxHidbusNumber; i++)
        {
            if (m_IsFocusChanged[i])
            {
                // Lock 中に State が変わっている可能性があるため、ここでロックをかける
                GetResourceHolder().LockInternalStateMutex(i);
                NN_UTIL_SCOPE_EXIT
                {
                    GetResourceHolder().UnlockInternalStateMutex(i);
                };

                ExcuteFocusChangedTask(i);
            }
            else
            {
                // Lock 中に State が変わっている可能性があるため、ここでロックをかける
                GetResourceHolder().LockInternalStateMutex(i);
                NN_UTIL_SCOPE_EXIT
                {
                    GetResourceHolder().UnlockInternalStateMutex(i);
                };

                // State Check
                switch (m_Status[i].state)
                {
                case HidbusState_NotActivated:
                    // 何もしない
                    break;
                case HidbusState_Activated:
                    HandleActivatedState(i);
                    break;
                case HidbusState_Attached:
                    HandleAttachedState(i, false);
                    break;
                case HidbusState_PollingMode:
                    HandlePollingModeState(i, false);
                    break;
                default:NN_UNEXPECTED_DEFAULT;
                }

                // Polling Data のコピー
                if (m_Status[i].state == HidbusState_PollingMode)
                {
                    size_t outSize = 0;
                    // NOTE : 現時点で大きい方のデータサイズ
                    uint8_t data[detail::JoyConDisableSixAxisSensorMaxPollingDataSize];

                    auto pollingDataResult = nn::hid::GetPollingDataForAttachmentDevice(&outSize, data, sizeof(data), m_Status[i].handle);
                    auto hidbusResult = ConvertHidResultToHidBusResultForCommunicatin(pollingDataResult);

                    if (!nn::hidbus::ResultExternalDeviceTimeout::Includes(hidbusResult))
                    {
                        // Timeout 以外でデータを更新
                        int pollingMode = GetResourceHolder().GetStatusManager()->GetPollingMode(i);

                        // NOTE : ジョイコン向けポーリングモードの実装になっているので、他のコントローラに対応するときには何か対応を入れる。
                        switch (static_cast<nn::hidbus::JoyPollingMode>(pollingMode))
                        {
                        case nn::hidbus::JoyPollingMode_SixAxisSensorEnable:
                            GetJoyEnableSixAxisPollingDataAccessor(i)->SetPollingData(data, static_cast<uint16_t>(outSize), hidbusResult);
                            break;

                        case nn::hidbus::JoyPollingMode_SixAxisSensorDisable:
                            GetJoyDisableSixAxisPollingDataAccessor(i)->SetPollingData(data, static_cast<uint16_t>(outSize), hidbusResult);
                            break;

                        default:NN_UNEXPECTED_DEFAULT;
                        }
                    }
                }
            }
        }

        // TODO : アプレットでも使いたい場合、常にだれが InFocus かを判定するために、Timer は止めないようにする必要がある。
        if (m_UnregisterFlag)
        {
            ExcuteUnregister();
        }
    }
}

SchedulerTask& GetSchedulerTask() NN_NOEXCEPT
{
    return nn::hidbus::detail::StaticObject<SchedulerTask>::Get();
}

}}} // namespace nn::hidbus::server
