﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/os/os_SystemEvent.h>

namespace nn { namespace hidbus { namespace server {

//!< 多重待ちイベントを扱うクラスの基底クラスです。
template<typename EventType>
class MultiWaitEventBase
{
    NN_DISALLOW_COPY(MultiWaitEventBase);
    NN_DISALLOW_MOVE(MultiWaitEventBase);

protected:
    //!< イベントオブジェクト
    EventType m_Event;

    //!< 多重待ちオブジェクトホルダー
    ::nn::os::MultiWaitHolderType m_MultiWaitHolder;

public:
    MultiWaitEventBase() NN_NOEXCEPT { /* 何もしない */ }

    virtual ~MultiWaitEventBase() NN_NOEXCEPT { /* 何もしない */ }

    //!< 多重待ちオブジェクトにイベントオブジェクトをリンクします。
    void Link(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMultiWait);
        ::nn::os::LinkMultiWaitHolder(pMultiWait, &m_MultiWaitHolder);
    }

    //!< イベントオブジェクトを全ての多重待ちオブジェクトからアンリンクします。
    void Unlink() NN_NOEXCEPT
    {
        ::nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
    }

    //!< イベントオブジェクトへのポインタを返します。
    EventType* GetBase() NN_NOEXCEPT
    {
        return &m_Event;
    }

    //!< 待ちの識別子を返します。
    const ::nn::os::MultiWaitHolderType* GetWaitId() const NN_NOEXCEPT
    {
        return &m_MultiWaitHolder;
    }
};

//!< 多重待ちイベントを扱うクラスの基底クラスです。
class MultiWaitEvent final : public MultiWaitEventBase<::nn::os::EventType>
{
public:
    MultiWaitEvent() NN_NOEXCEPT
    {
        // イベントオブジェクトを初期化
        ::nn::os::InitializeEvent(&m_Event,
                                  false,
                                  ::nn::os::EventClearMode_ManualClear);

        // 多重待ちオブジェクトホルダーを初期化
        ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);
    }

    virtual ~MultiWaitEvent() NN_NOEXCEPT NN_OVERRIDE
    {
        // 多重待ちオブジェクトホルダーを破棄
        ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

        // イベントオブジェクトを破棄
        ::nn::os::FinalizeEvent(&m_Event);
    }

    //!< 多重待ちイベントをシグナル状態にセットします。
    void Signal() NN_NOEXCEPT
    {
        ::nn::os::SignalEvent(&m_Event);
    }

    //!< 多重待ちイベントを非シグナル状態にクリアします。
    void Clear() NN_NOEXCEPT
    {
        ::nn::os::ClearEvent(&m_Event);
    }
};

//!< 多重待ちタイマーイベントを扱うクラスの基底クラスです。
class MultiWaitTimerEvent final
    : public MultiWaitEventBase<::nn::os::TimerEventType>
{
public:
    MultiWaitTimerEvent() NN_NOEXCEPT
    {
        // タイマーイベントオブジェクトを初期化
        ::nn::os::InitializeTimerEvent(&m_Event,
                                       ::nn::os::EventClearMode_ManualClear);

        // 多重待ちオブジェクトホルダーを初期化
        ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);
    }

    virtual ~MultiWaitTimerEvent() NN_NOEXCEPT NN_OVERRIDE
    {
        // 多重待ちオブジェクトホルダーを破棄
        ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

        // タイマーイベントオブジェクトを破棄
        ::nn::os::FinalizeTimerEvent(&m_Event);
    }

    //!< 多重待ちタイマーイベントを非シグナル状態にクリアします。
    void Clear() NN_NOEXCEPT
    {
        ::nn::os::ClearTimerEvent(&m_Event);
    }
};

//!< 多重待ちシステムイベントを扱うクラスの基底クラスです。
class MultiWaitSystemEvent final
    : public MultiWaitEventBase<::nn::os::SystemEventType>
{
public:
    MultiWaitSystemEvent() NN_NOEXCEPT
    {
        // システムイベントオブジェクトを初期化
        ::nn::os::CreateSystemEvent(&m_Event, ::nn::os::EventClearMode_ManualClear, false);

        // 多重待ちオブジェクトホルダーを初期化
        ::nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, &m_Event);
    }

    virtual ~MultiWaitSystemEvent() NN_NOEXCEPT NN_OVERRIDE
    {
        // 多重待ちオブジェクトホルダーを破棄
        ::nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

        // システムイベントオブジェクトを破棄
        ::nn::os::DestroySystemEvent(&m_Event);
    }

    //!< 多重待ちシステムイベントを非シグナル状態にクリアします。
    void Clear() NN_NOEXCEPT
    {
        ::nn::os::ClearSystemEvent(&m_Event);
    }
};

}}} // namespace nn::hidbus::server
