﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

namespace nn { namespace hidbus { namespace detail {

//!< リングバッファ上の要素をスレッドセーフに扱うためのクラスです。
template<typename BufferT>
class AtomicStorage final
{
    NN_DISALLOW_COPY(AtomicStorage);
    NN_DISALLOW_MOVE(AtomicStorage);

private:
    ::std::atomic<int64_t> m_SamplingNumber;
    BufferT m_State;

public:
    AtomicStorage() NN_NOEXCEPT
        : m_SamplingNumber(0)
        , m_State()
    {
        // 何もしない
    }

    ~AtomicStorage() NN_NOEXCEPT { /* 何もしない */ }

    //!< 要素をスレッドセーフに取得します。
    bool Get(BufferT* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        const int64_t samplingNumber =
            m_SamplingNumber.load(::std::memory_order_acquire);

        *pOutValue = m_State;

        // 読み出し中に値が書き換わっていなければ成功
        return (samplingNumber == m_SamplingNumber);
    }

    //!< 要素をスレッドセーフに設定します。
    void Set(const BufferT& value) NN_NOEXCEPT
    {
        m_SamplingNumber.store(
            value.samplingNumber, ::std::memory_order_release);

        ::std::atomic_thread_fence(::std::memory_order_release);

        m_State = value;
    }
};

}}} // namespace nn::hidbus::detail
