﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/hid/hid_VibrationTarget.h>
#include "detail/hid_LockableMutexType.h"
#include "detail/hid_VibrationValueQueue.h"

namespace {

//!< VibrationTarget 内の排他処理に用いるミューテックス
nn::hid::detail::LockableMutexType s_VibrationTargetMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

namespace nn { namespace hid {

VibrationTarget::VibrationTarget() NN_NOEXCEPT
    : m_IsHandleSet(false)
{
    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    m_Handle._storage = 0;
}

VibrationTarget::VibrationTarget(const VibrationDeviceHandle& handle) NN_NOEXCEPT
    : m_IsHandleSet(true), m_Handle(handle) {}

VibrationTarget::~VibrationTarget() NN_NOEXCEPT {}

void VibrationTarget::SetVibrationDeviceHandle(const VibrationDeviceHandle& handle) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    m_IsHandleSet = true;
    m_Handle = handle;
}

void VibrationTarget::UnsetVibrationDeviceHandle() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);

    //振動停止処理
    if(m_IsHandleSet)
    {
        VibrationValue value = VibrationValue::Make();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::detail::EnqueueVibrationValue(m_Handle, value));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::detail::FlushEnqueuedVibrationValues());
    }

    m_IsHandleSet = false;
    m_Handle._storage = 0;
}

bool VibrationTarget::IsVibrationDeviceHandleSet() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    return m_IsHandleSet;
}

VibrationDeviceHandle VibrationTarget::GetVibrationDeviceHandle() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    return m_Handle;
}

void VibrationTarget::GetActualVibration(VibrationValue* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    if(m_IsHandleSet)
    {
        nn::hid::GetActualVibrationValue(pOutValue, m_Handle);
    }
    else
    {
        *pOutValue = nn::hid::VibrationValue::Make();
    }
}

void VibrationTarget::GetVibrationDeviceInfo(VibrationDeviceInfo* pOutValue) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    if(m_IsHandleSet)
    {
        nn::hid::GetVibrationDeviceInfo(pOutValue, m_Handle);
    }
    else
    {
        pOutValue->deviceType = nn::hid::VibrationDeviceType_Unknown;
        pOutValue->position = nn::hid::VibrationDevicePosition_None;
    }
}

void VibrationTarget::OnNextSampleRequired(
    VibrationValue* pValue,
    VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_NOT_NULL(pInputConnections);

    std::lock_guard<decltype(s_VibrationTargetMutex)> lock(s_VibrationTargetMutex);
    if(m_IsHandleSet)
    {
        // 接続元からの振動をミックスした後、振動子に送信します (いったんキューに入れます)
        VibrationMixer::OnNextSampleRequired(pValue, pInputConnections);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::detail::EnqueueVibrationValue(m_Handle, *pValue));
    }
}

}} // namespace nn::hid
