﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_Mutex.h>
#include <nn/hid/hid_VibrationNode.h>
#include <nn/hid/hid_VibrationNodeConnection.h>
#include "detail/hid_LockableMutexType.h"
#include "detail/hid_VibrationValueQueue.h"

namespace {

//!< 有向グラフ全体で同期をとるためのサンプル番号
static uint8_t s_VibrationNodeSampleNumber;

//!< すべての振動ノードを格納するリストを取得します
typedef nn::util::IntrusiveList<
    ::nn::hid::VibrationNode,
    nn::util::IntrusiveListBaseNodeTraits<::nn::hid::VibrationNode>> VibrationNodeList;
VibrationNodeList& GetVibrationNodeList() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(VibrationNodeList, s_NodeList);
    return s_NodeList;
}

//!< VibrationNode と VibrationNodeConnection で共有するミューテックス
nn::hid::detail::LockableMutexType s_VibrationNodeMutex = { NN_OS_MUTEX_INITIALIZER(true) };

//!< すべての振動ノードを格納するリストの操作に際して利用するミューテックス
nn::hid::detail::LockableMutexType s_VibrationNodeListMutex = { NN_OS_MUTEX_INITIALIZER(true) };

}

namespace nn { namespace hid {

const nn::TimeSpan VibrationNode::DefaultVibrationSampleInterval = nn::TimeSpan::FromMilliSeconds(5);


/*
 * public
 */
VibrationNode::VibrationNode() NN_NOEXCEPT
    : m_VibrationValue(VibrationValue::Make()), m_SampleNumber(s_VibrationNodeSampleNumber)
{
    std::lock_guard<decltype(s_VibrationNodeListMutex)> lock(s_VibrationNodeListMutex);
    GetVibrationNodeList().push_back(*this);
}

VibrationNode::~VibrationNode() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeListMutex)> lock(s_VibrationNodeListMutex);
    auto it = GetVibrationNodeList().iterator_to(*this);
    GetVibrationNodeList().erase(it);
}

VibrationValue VibrationNode::GetCurrentVibration() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    return m_VibrationValue;
}

void VibrationNode::DisconnectFrom(VibrationNode* pDestination) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeListMutex)> listLock(s_VibrationNodeListMutex);
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pDestination);
    VibrationNodeConnection* pConnection = GetConnectionTo(pDestination);
    if(pConnection != nullptr)
    {
        pConnection->Disconnect();
    }
}

bool VibrationNode::IsConnectedTo(const VibrationNode* pDestination) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pDestination);
    for(auto& c : pDestination->m_InputConnections)
    {
        if(c.GetSource() == this)
        {
            return true;
        }
    }
    return false;
}

VibrationNodeConnection* VibrationNode::GetConnectionTo(const VibrationNode* pDestination) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pDestination);
    for(auto& c : pDestination->m_InputConnections)
    {
        if(c.GetSource() == this)
        {
            const VibrationNodeConnection* p = &c;
            return const_cast<VibrationNodeConnection*>(p);
        }
    }
    return nullptr;
}

void VibrationNode::SetModulationTo(const VibrationNode* pDestination, const VibrationModulation& modulation) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pDestination);
    VibrationNodeConnection* pConnection = GetConnectionTo(pDestination);
    if(pConnection != nullptr)
    {
        pConnection->SetModulation(modulation);
    }
}

VibrationModulation VibrationNode::GetModulationTo(const VibrationNode* pDestination) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pDestination);
    VibrationNodeConnection* pConnection = GetConnectionTo(pDestination);
    if(pConnection != nullptr)
    {
        return pConnection->GetModulation();
    }
    return VibrationModulation::Make();
}

/*
 * protected
 */
void VibrationNode::SetCurrentVibration(const VibrationValue& value)
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    m_VibrationValue = value;
}

/*
 * private
 */
void VibrationNode::SyncSampleNumber() NN_NOEXCEPT
{
    // 接続元の振動ノードを更新します
    for(auto& c : m_InputConnections)
    {
        c.GetSource()->SyncSampleNumber();
    }

    // 自分自身が未更新状態であった場合にのみ、自身の振動値を更新します
    if(m_SampleNumber != s_VibrationNodeSampleNumber)
    {
        VibrationValue v = GetCurrentVibration();
        OnNextSampleRequired(&v, &m_InputConnections);
        SetCurrentVibration(v);
        m_SampleNumber = s_VibrationNodeSampleNumber;
    }
}

/*
 * static public
 */
void VibrationNode::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeListMutex)> lock(s_VibrationNodeListMutex);
    s_VibrationNodeSampleNumber++;
    for(auto& node : GetVibrationNodeList())
    {
        node.SyncSampleNumber();
    }

    // キューに入った振動値を一度に送信します
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hid::detail::FlushEnqueuedVibrationValues());
}

/*
 * VibrationNodeConnection
 */

VibrationNodeConnection::VibrationNodeConnection() NN_NOEXCEPT
    : m_pSource(nullptr), m_pDestination(nullptr), m_Modulation(VibrationModulation::Make()) {}

VibrationNodeConnection::VibrationNodeConnection(VibrationNode* pSource, VibrationNode* pDestination) NN_NOEXCEPT
    : m_pSource(nullptr), m_pDestination(nullptr), m_Modulation(VibrationModulation::Make())
{
    (void)Connect(pSource, pDestination);
}

VibrationNodeConnection::~VibrationNodeConnection() NN_NOEXCEPT
{
    Disconnect();
}

bool VibrationNodeConnection::Connect(VibrationNode* pSource, VibrationNode* pDestination) NN_NOEXCEPT
{
    //すべての振動ノードを格納するリストをロックしてから、個々のノードのConnectを行う
    std::lock_guard<decltype(s_VibrationNodeListMutex)> listLock(s_VibrationNodeListMutex);
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    NN_SDK_ASSERT_NOT_NULL(pSource);
    NN_SDK_ASSERT_NOT_NULL(pDestination);

    if(IsConnected() || pSource->IsConnectedTo(pDestination))
    {
        return false;
    }
    else
    {
        m_pSource = pSource;
        m_pDestination = pDestination;
        m_pDestination->m_InputConnections.push_back(*this);
        return true;
    }
}

bool VibrationNodeConnection::IsConnected() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    return (m_pDestination != nullptr);
}

void VibrationNodeConnection::Disconnect() NN_NOEXCEPT
{
    //すべての振動ノードを格納するリストをロックしてから、個々のノードのDisconnectを行う
    std::lock_guard<decltype(s_VibrationNodeListMutex)> listLock(s_VibrationNodeListMutex);
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    if(IsConnected())
    {
        auto it = m_pDestination->m_InputConnections.iterator_to(*this);
        m_pDestination->m_InputConnections.erase(it);
        m_pSource = nullptr;
        m_pDestination = nullptr;
    }
}

VibrationNode* VibrationNodeConnection::GetSource() const NN_NOEXCEPT
{
    return m_pSource;
}

VibrationNode* VibrationNodeConnection::GetDestination() const NN_NOEXCEPT
{
    return m_pDestination;
}

VibrationModulation VibrationNodeConnection::GetModulation() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    return m_Modulation;
}

void VibrationNodeConnection::SetModulation(const VibrationModulation& modulation) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationNodeMutex)> lock(s_VibrationNodeMutex);
    m_Modulation = modulation;
}

}} // namespace nn::hid
