﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/debug/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_SystemEvent.h>

#include "detail/hid_NpadId.h"
#include "detail/hid_NpadImpl.h"
#include "detail/hid_NpadSystemImpl.h"

namespace nn { namespace hid { namespace system {

NpadDeviceTypeSet GetNpadDeviceType(const NpadIdType& id) NN_NOEXCEPT
{
    NpadDeviceTypeSet deviceType;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadDeviceType(&deviceType, id));

    return deviceType;
}

NpadPowerInfo GetNpadPowerInfo(const NpadIdType& id) NN_NOEXCEPT
{
    NpadPowerInfo powerInfo;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadPowerInfo(&powerInfo, id));

    return powerInfo;

}

void GetNpadPowerInfo(NpadPowerInfo* pOutLeftJoy, NpadPowerInfo* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadPowerInfo(pOutLeftJoy, pOutRightJoy, id));
}

void SetNpadPlayerLedBlinkingDevice(const NpadIdType& id, const system::NpadDeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::SetNpadPlayerLedBlinkingDevice(id, deviceType));
}

bool IsNpadCharging(const NpadIdType& id) NN_NOEXCEPT
{
    NN_UNUSED(id);

    return false;
}

NpadBatteryLevel GetNpadBatteryLevel(const NpadIdType& id) NN_NOEXCEPT
{
    NN_UNUSED(id);

    return nn::hid::system::NpadBatteryLevel_High;
}

void IsNpadCharging(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    NN_UNUSED(pOutLeftJoy);
    NN_UNUSED(pOutRightJoy);
}

void GetNpadBatteryLevel(NpadBatteryLevel* pOutLeftJoy, NpadBatteryLevel* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_UNUSED(id);
    NN_UNUSED(pOutLeftJoy);
    NN_UNUSED(pOutRightJoy);
}

bool IsAbxyButtonOriented(const NpadIdType& id) NN_NOEXCEPT
{
    bool isOriented;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsAbxyButtonOriented(&isOriented, id));

    return isOriented;
}

bool IsSlSrButtonOriented(const NpadIdType& id) NN_NOEXCEPT
{
    bool isOriented;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsSlSrButtonOriented(&isOriented, id));

    return isOriented;
}

void GetPlusMinusButtonCapability(bool* pOutPlusButton, bool* pOutMinusButton, const NpadIdType& id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetPlusMinusButtonCapability(pOutPlusButton, pOutMinusButton, id));
}


bool IsUnsupportedButtonPressed(const NpadIdType& id, const NpadStyleSet& style) NN_NOEXCEPT
{
    bool isPressed;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsUnsupportedButtonPressed(&isPressed, id, style));

    return isPressed;

}

bool IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT
{
    bool isPressed;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::IsHandheldButtonPressedOnConsoleMode(&isPressed));

    return isPressed;
}

void ApplyNpadSystemCommonPolicy() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ApplyNpadSystemCommonPolicy());
}

void ApplyNpadSystemCommonPolicyFull() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::ApplyNpadSystemCommonPolicyFull());
}

void GetNpadState(NpadSystemState* pOutValue, const NpadIdType& id
                  ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadState(pOutValue, id));
}

int GetNpadStates(NpadSystemState* pOutValues,
                  int count,
                  const NpadIdType& id) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetNpadStates(&gotCount, pOutValues, count, id));
    return gotCount;
}

void GetNpadState(NpadSystemExtState* pOutValue, const NpadIdType& id
                  ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadState(pOutValue, id));
}

int GetNpadStates(NpadSystemExtState* pOutValues,
                  int count,
                  const NpadIdType& id) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        detail::GetNpadStates(&gotCount, pOutValues, count, id));
    return gotCount;
}

void EnableAssigningSingleOnSlSrPress() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::EnableAssigningSingleOnSlSrPress());
}

void DisableAssigningSingleOnSlSrPress() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::DisableAssigningSingleOnSlSrPress());
}

Result GetLastActiveNpadController(NpadIdType* pOutValue) NN_NOEXCEPT
{
    auto result = detail::GetLastActiveNpadController(pOutValue);
    NN_RESULT_TRY(result);
        NN_RESULT_CATCH(::nn::hid::system::ResultNoActiveNpad)
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

void GetNpadSystemExtStyle(NpadSystemExtMainStyle* pOutMainStyle,
                           NpadSystemExtSubStyle* pOutSubStyle,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadSystemExtStyle(pOutMainStyle, pOutSubStyle, id));
}

int GetUniquePadsFromNpad(UniquePadId* pOutValues, int count, const NpadIdType& npadId) NN_NOEXCEPT
{
    int gotCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetUniquePadsFromNpad(&gotCount, pOutValues, count, npadId));
    return gotCount;
}

bool HasBattery(const NpadIdType& id) NN_NOEXCEPT
{
    bool hasBattery;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::HasBattery(&hasBattery, id));

    return hasBattery;
}

void HasBattery(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::HasBattery(pOutLeftJoy, pOutRightJoy, id));
}

InterfaceType GetNpadInterfaceType(const NpadIdType& id) NN_NOEXCEPT
{
    InterfaceType interfaceType;

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadInterfaceType(&interfaceType, id));

    return interfaceType;
}

void GetNpadInterfaceType(InterfaceType* pOutLeftJoy, InterfaceType* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetNpadInterfaceType(pOutLeftJoy, pOutRightJoy, id));
}

::nn::Result GetNpadFullKeyGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftGrip);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightGrip);

    auto result = detail::GetNpadFullKeyGripColor(pOutLeftGrip, pOutRightGrip, id);
    NN_RESULT_TRY(result);
        NN_RESULT_CATCH(::nn::hid::ResultNpadControllerNotConnected)
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_RESULT_SUCCESS;
}

NpadStyleSet GetSupportedNpadStyleSetOfCallerApplet() NN_NOEXCEPT
{
    NpadStyleSet outStyle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(detail::GetSupportedNpadStyleSetOfCallerApplet(&outStyle));
    return outStyle;
}

}}} // namespace nn::hid
