﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/xcd/xcd.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_ActivationCount.h"
#include "hid_ControllerFirmwareAccessor.h"
#include "hid_InterruptSceneNotifier.h"
#include "hid_MultiWaitEvent.h"
#include "hid_UniquePadId.h"
#include "hid_UniquePadManager.h"
#include "hid_XcdFirmwareMemory.h"

namespace nn { namespace hid { namespace detail {

// FW 更新の動作設定
struct UpdateOption
{
    typedef ::nn::util::BitFlagSet<32, UpdateOption>::Flag<0> VersionRestriction;   // バージョンによる更新要否判定を行うか
    typedef ::nn::util::BitFlagSet<32, UpdateOption>::Flag<1> PermitMcuFullUpdate;  // IAP 更新を許可するか
    typedef ::nn::util::BitFlagSet<32, UpdateOption>::Flag<2> NeedsMcuFullUpdate;   // IAP 更新が必要な状態か
    typedef ::nn::util::BitFlagSet<32, UpdateOption>::Flag<3> Reconnect;            // 更新完了後に再接続するか
    typedef ::nn::util::BitFlagSet<32, UpdateOption>::Flag<4> UseTransferMemory;    // TransferMemory を使用するか
};

// UpdateOption の集合を扱う型
typedef ::nn::util::BitFlagSet<32, UpdateOption> UpdateOptionSet;

class FileObject
{
public:
    FileObject() NN_NOEXCEPT :
        m_Handle(),
        m_IsOpened(false)
    {}

    Result Open(const char* filename) NN_NOEXCEPT
    {
        Close();

        NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, filename, nn::fs::OpenMode_Read));

        m_IsOpened = true;
        NN_RESULT_SUCCESS;
    }

    void Close() NN_NOEXCEPT
    {
        if (!m_IsOpened)
        {
            return;
        }

        nn::fs::CloseFile(m_Handle);
        m_IsOpened = false;
    }

    ::nn::fs::FileHandle GetHandle() const NN_NOEXCEPT
    {
        // オープンしていない状態でのアクセスは実装ミス
        NN_ABORT_UNLESS(m_IsOpened);
        return m_Handle;
    }

private:
    ::nn::fs::FileHandle m_Handle;
    bool                 m_IsOpened;
};

class XcdFirmwareUpdater final
{
    NN_DISALLOW_COPY(XcdFirmwareUpdater);
    NN_DISALLOW_MOVE(XcdFirmwareUpdater);

public:
    enum class FirmwareUpdateTarget
    {
        None,
        Bluetooth,
        Mcu,
        BluetoothAndMcu
    };

private:
    //!< 更新中のデバイスの Xcd ハンドル
    nn::xcd::DeviceHandle m_Handle;

    //!< 更新中のデバイスの内容
    nn::xcd::DeviceInfo m_DeviceInfo;

    //!< 更新中のデバイスの接続インターフェース
    nn::xcd::InterfaceType m_InterfaceType;

    //!< XcdFirmwareUpdater がアクティベートされた回数
    ActivationCount m_ActivationCount;

    //!< UniquePad マネージャ
    UniquePadManager* m_pUniquePadManagers[system::UniquePadIdCountMax];

    //!< 標準のシステムデータへのアクセサ
    ControllerFirmwareAccessor m_ControllerFirmwareAccessorNormal;

    //!< デバッグ/巻き戻し用のシステムデータへのアクセサ
    ControllerFirmwareAccessor m_ControllerFirmwareAccessorDebug;

    //!< FW 更新に使用するシステムデータへのアクセサ
    ControllerFirmwareAccessor* m_pActiveControllerFirmwareAccessor;

    //!< 更新する BT ファームウェアファイルに対するハンドル
    FileObject m_BluetoothFile;

    //!< 更新する Mcu ファームウェアファイルに対するハンドル
    FileObject m_McuFile;

    //!< TransferMemory で渡されたファームウェアイメージ
    XcdFirmwareMemory m_FirmwareMemory;

    //!< Bluetooth のファームウェア更新が完了した際に通知を受ける SystemEvent
    ::nn::os::SystemEventType* m_pXcdBluetoothUpdateEvent;

    //!< Mcu のファームウェア更新が完了した際に通知を受ける SystemEvent
    ::nn::os::SystemEventType* m_pXcdMcuUpdateEvent;

    //!< 新たなサンプルが受信された際に通知を受ける SystemEvent
    ::nn::os::SystemEventType* m_pSampleUpdateEvent;

    //!< コントローラーリセット時のタイムアウト
    ::nn::os::TimerEventType* m_pResetTimeoutEvent;

    //!< ファームウェアの更新対象
    FirmwareUpdateTarget m_FirmwareUpdateTarget;

    enum FirmwareUpdateState
    {
        FirmwareUpdateState_NotStarted,
        FirmwareUpdateState_BtUpdate,
        FirmwareUpdateState_Reboot1_WaitingDisconnect,
        FirmwareUpdateState_Reboot1_WaitingReconnect,
        FirmwareUpdateState_McuVersionCheck,
        FirmwareUpdateState_McuUpdate,
        FirmwareUpdateState_McuVerify,
        FirmwareUpdateState_Reboot2,
        FirmwareUpdateState_Complete,
        FirmwareUpdateState_BtVerifyError,
        FirmwareUpdateState_BtUnknownError,
        FirmwareUpdateState_McuVerifyError,
        FirmwareUpdateState_McuHardwareError,
        FirmwareUpdateState_McuUnknownError,
        FirmwareUpdateState_Disconnected,
    };
    // ファームウェアの更新状況
    FirmwareUpdateState m_FirmwareUpdateState;

    //!< ファームウェア更新のオプション
    UpdateOptionSet m_UpdateOption;

    //!< 再接続待ちか
    bool m_IsReconnectRequired;

    //!< 再接続待ち開始時の Tick
    ::nn::os::Tick m_ReconnectionStartTick;

    //!< 割り込みシーンの通知を行うクラス
    InterruptSceneNotifier* m_pInterruptSceneNotifier;

    //!< 直近の最前面アプレットの ResourceUserId
    ::nn::applet::AppletResourceUserId m_ForegroundAruid;

    //!< Hotfix 起因のファームウェア更新をスキップするためのフラグ
    bool m_IsFirmwareHotfixUpdateSkipFlagEnabled;

public:
    XcdFirmwareUpdater() NN_NOEXCEPT;
    ~XcdFirmwareUpdater() NN_NOEXCEPT;

    //!< Bluetooth のファームウェア更新が完了した際に通知を受けるためのイベントオブジェクトを受け取ります。
    void SetBluetoothUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< Mcu のファームウェア更新が完了した際に通知を受けるためのイベントオブジェクトを受け取ります。
    void SetMcuUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< 新たなサンプルが受信された際に通知を受けるためのイベントオブジェクトを受け取ります。
    void SetSampleUpdateEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

    //!< リセット処理がタイムアウトした際に通知を受けるためのイベントオブジェクトを受け取ります。
    void SetResetTimeoutEvent(nn::os::TimerEventType* pEvent) NN_NOEXCEPT;

    //!< UniquePadManager をセットします
    void SetUniquePadManagers(int index, UniquePadManager* pManager) NN_NOEXCEPT;

    //!< InterruptSceneNotifier をセットします
    void SetInterruptSceneNotifier(InterruptSceneNotifier* pNotifier) NN_NOEXCEPT;

    //!< 最前面の AppletResourceUserId 変更通知を受け取ります。
    void NotifyApplicationResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ファームウェア更新処理を有効にします。
    Result ActivateFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェア更新処理を無効にします。
    Result DeactivateFirmwareUpdate() NN_NOEXCEPT;

    //!< デバッグ/巻き戻し用システムデータのキャッシュ情報を破棄します。
    Result DiscardSystemDataCacheForRevert() NN_NOEXCEPT;

    //!< Hotfix 起因のファームウェア更新をスキップします。
    Result SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT;

    //!< ファームウェア更新処理を開始します。 (System 用)
    Result StartFirmwareUpdateForSystem(
        system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新処理を開始します。 (Debug 用)
    Result StartFirmwareUpdateForDebug(
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新処理を開始します。 (巻き戻し用)
    Result StartFirmwareUpdateForRevert(
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新を開始します。 (TransferMemory 版)
    Result StartFirmwareUpdateByTransferMemory(
        system::FirmwareUpdateDeviceHandle* pOutDeviceHandle,
        system::UniquePadId id,
        FirmwareUpdateTarget target,
        const XcdFirmwareImageInfo& imageInfo) NN_NOEXCEPT;

    //!< ファームウェア更新処理を中断します。
    Result AbortFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェアのバージョンを取得します。
    Result GetFirmwareVersion(
        system::FirmwareVersion* pOutValue,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 更新先ファームウェアのバージョンを取得します。
    Result GetDestinationFirmwareVersion(
        system::FirmwareVersion* pOutVersion,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 巻き戻し用の更新先ファームウェアのバージョンを取得します。
    Result GetDestinationFirmwareVersionForRevert(
        debug::FirmwareVersion* pOutVersion,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< 更新可能ファームウェアの有無を確認します。
    Result IsFirmwareUpdateAvailable(
        bool* pOutIsAvailable,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新が要求されているか確認します。
    Result CheckFirmwareUpdateRequired(
        system::FirmwareUpdateRequiredReason* pOutReason,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェアの更新状況を取得します。
    Result GetFirmwareUpdateState(
        system::FirmwareUpdateState* pOutState,
        system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT;

    //!< ファームウェアの更新状況を取得します。(詳細版)
    Result GetFirmwareUpdateStage(
        debug::FirmwareUpdateStage* pOutStage,
        uint8_t* pOutProgress) NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイスかどうか確認します。
    Result IsFirmwareUpdatingDevice(
        bool* pOutIsUpdating,
        system::UniquePadId id) NN_NOEXCEPT;

    //!< Xcd の Bluetooth 更新が完了した際に呼ばれる関数
    void BluetoothFirmwareUpdateCompleted() NN_NOEXCEPT;

    //!< Xcd の Mcu 更新が完了した際に呼ばれる関数
    void McuFirmwareUpdateCompleted() NN_NOEXCEPT;

    //!< Xcd の サンプル受信が完了した際に呼ばれる関数
    void SampleRecieved() NN_NOEXCEPT;

    //!< デバイスの接続状態が更新されたときに通知を受ける関数
    void DeviceUpdated() NN_NOEXCEPT;

    //!< コントローラーへの接続トリガータイムアウト通知を受ける関数
    void ConnectionTriggerTimedOut() NN_NOEXCEPT;

    //!< コントローラーリセット時のタイムアウト通知を受ける関数
    void ResetTimedOut() NN_NOEXCEPT;

private:
    //!< ファームウェア更新処理を開始します。
    Result StartFirmwareUpdateImpl(
        nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT;

    //!< ファームウェア更新処理を開始します。 (TransferMemory 版)
    Result StartFirmwareUpdateImplByTransferMemory(
        nn::xcd::DeviceHandle xcdHandle,
        FirmwareUpdateTarget target) NN_NOEXCEPT;

    //!< ファームウェア更新の終了処理を行います。
    void FinishFirmwareUpdateProcess() NN_NOEXCEPT;

    //!< ファームウェア更新をデバイス切断扱いで失敗させます。
    void FailFirmwareUpdateAsDisconnected() NN_NOEXCEPT;

    //!< ファームウェアのバージョン取得の実処理です。
    Result GetFirmwareVersionImpl(
        debug::FirmwareVersion* pOutValue,
        bool* pOutIsMcuIapCorrupted,
        ::nn::xcd::DeviceHandle xcdHandle) NN_NOEXCEPT;

    //!< 更新先ファームウェアのバージョン取得の実処理です。
    Result GetDestinationFirmwareVersionImpl(
        system::FirmwareVersion* pOutVersion,
        nn::xcd::DeviceHandle handle,
        ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT;

    //!< Bluetooth ファームウェアの更新が要求されているか確認します。
    Result CheckBluetoothFirmwareUpdateRequired(
        system::FirmwareUpdateRequiredReason* pOutReason,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    //!< Mcu ファームウェアの更新が要求されているか確認します。
    Result CheckMcuFirmwareUpdateRequired(
        system::FirmwareUpdateRequiredReason* pOutReason,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    //!< デバイスが接続されているかどうか検出します。
    bool IsDeviceConnected(nn::xcd::DeviceList& deviceList, int* pIndex) NN_NOEXCEPT;

    //!< アップデート実行中か判定します。
    bool IsUpdateRunning() NN_NOEXCEPT;

    //!< 再起動中か判定します。
    bool IsRebooting() NN_NOEXCEPT;

    //!< システムデータのマウントを解除します。
    void UnmountSystemData() NN_NOEXCEPT;

    //!< Bluetooth ファームウェア更新が必要か判定します。バージョンチェック要否フラグの影響を受けます。
    Result NeedsBluetoothUpdate(bool* pOutNeedsUpdate) NN_NOEXCEPT;

    //!< Bluetooth ファームウェア更新判定の実処理です。
    Result NeedsBluetoothUpdateImpl(
        bool* pOutNeedsUpdate,
        ::nn::xcd::DeviceHandle xcdHandle,
        ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT;

    //!< Mcu ファームウェア更新が必要か判定します。バージョンチェック要否フラグの影響を受けます。
    Result NeedsMcuUpdate(bool* pOutNeedsUpdate) NN_NOEXCEPT;

    //!< Mcu ファームウェア更新要否判定の実処理です。
    Result NeedsMcuUpdateImpl(
        bool* pOutNeedsUpdate,
        bool* pOutNeedsIapUpdate,
        ::nn::xcd::DeviceHandle xcdHandle,
        ControllerFirmwareAccessor* pAccessor) NN_NOEXCEPT;

    //!< Mcu ファームウェアのバージョン確認処理を行います。
    void ProcessMcuVersionCheck() NN_NOEXCEPT;

    //!< Mcu ファームウェアのベリファイ処理を行います
    void ProcessMcuVerify() NN_NOEXCEPT;

    //!< Bluetooth ファームウェア更新を開始します。
    Result StartBluetoothUpdate() NN_NOEXCEPT;

    //!< Mcu ファームウェア更新に必要な準備を行います。
    Result PrepareMcuUpdate() NN_NOEXCEPT;

    //!< Mcu の更新を開始します。
    Result StartMcuUpdate() NN_NOEXCEPT;

    //!< Mcu の更新を終了します。
    Result EndMcuUpdate() NN_NOEXCEPT;

    //!< UniquePadId から Xcd のデバイスハンドルを取得します。
    ::nn::xcd::DeviceHandle GetXcdDeviceHandle(system::UniquePadId id) NN_NOEXCEPT;

    //!< 更新中のデバイスを再起動します。引数に true を指定した場合は、再起動後に接続を試みます。
    void RebootUpdatingDevice(bool needsReconnect) NN_NOEXCEPT;

    //!< 再接続トリガーを発行します。
    void TriggerReconnect(bool isFirstTrigger) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
