﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <memory>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/os/os_Mutex.h>
#include <nn/vi/hid/vi_Hid.h>
#include <nn/vi/hid/vi_WindowInfo.h>

#include "hid_ActivationCount.h"
#include "hid_Point.h"
#include "hid_Rectangle.h"
#include "hid_RingLifo.h"
#include "hid_StaticObject.h"
#include "hid_WindowsProcedure-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 対象となるウィンドウハンドルを持つウィンドウの名前
const char* const WindowName = "GfxMainDisplay";

//!< Mouse のボタンのマスク表現定義です。
enum MouseButtonMask : int
{
    MouseButtonMask_Left = 1 << MouseButton::Left::Index,
    MouseButtonMask_Right = 1 << MouseButton::Right::Index,
    MouseButtonMask_Middle = 1 << MouseButton::Middle::Index,
    MouseButtonMask_Forward = 1 << MouseButton::Forward::Index,
    MouseButtonMask_Back = 1 << MouseButton::Back::Index,
};

//!< タッチ入力を表す構造体です。
struct TouchInputState
{
    int64_t samplingNumber;
    TOUCHINPUT touchInput;
};

//!< ウィンドウプロシージャの管理を行うクラスです。
class ProcedureManager final
{
    NN_DISALLOW_COPY(ProcedureManager);
    NN_DISALLOW_MOVE(ProcedureManager);

private:
    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< ウィンドウハンドル
    ::std::atomic<HWND> m_WindowHandle;

    //!< 元のウィンドウプロシージャ
    ::std::atomic<WNDPROC> m_PreviousProcedure;

    //!< マウスボタンの状態
    ::std::atomic<int> m_MouseButtonMask;

    //!< ホイールの回転差分
    ::std::atomic<int32_t> m_WheelDelta;

    //!< タッチ入力の先頭のサンプリング番号
    int64_t m_TouchInputHeadSamplingNumber;

    //!< タッチ入力の末尾のサンプリング番号
    int64_t m_TouchInputTailSamplingNumber;

    //!< タッチ入力の LIFO
    RingLifo<TouchInputState, TouchInputCountMax> m_TouchInputLifo;

    //!< タッチ入力の読み出し先
    TouchInputState m_TouchInputStates[TouchInputCountMax];

public:
    ProcedureManager() NN_NOEXCEPT
        : m_Mutex(false)
        , m_WindowHandle(0)
        , m_PreviousProcedure(0)
        , m_MouseButtonMask(0)
        , m_WheelDelta(0)
        , m_TouchInputHeadSamplingNumber(0)
        , m_TouchInputTailSamplingNumber(-1)
    {
        // 何もしない
    }

    ~ProcedureManager() NN_NOEXCEPT { /* 何もしない */ }

    //!< ウィンドウプロシージャマネージャを返します。
    static ProcedureManager& GetProcedureManager() NN_NOEXCEPT;

    //!< アクティブ化します。
    void Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    void Deactivate() NN_NOEXCEPT;

    //!< サブクラス化状態を維持します。
    bool KeepAlive() NN_NOEXCEPT;

    //!< クライアント領域のサイズを取得します。
    bool GetClientAreaSize(Rectangle* pOutValue) NN_NOEXCEPT;

    //!< マウスの座標を取得します。
    bool GetMousePosition(Point* pOutValue) NN_NOEXCEPT;

    //!< マウスボタンの状態を返します。
    MouseButtonSet GetMouseButtons() NN_NOEXCEPT;

    //!< ホイールの回転差分を破壊的に返します。
    int32_t GetWheelDelta() NN_NOEXCEPT;

    //!< タッチ入力を取得します。
    int GetTouchInputs(TOUCHINPUT* outBuffer, int count) NN_NOEXCEPT;

private:
    //!< ウィンドウプロシージャ
    static LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam,
                                    LPARAM lParam) NN_NOEXCEPT;
};

//!< ペンイベントか否かを表す値を返します。
bool IsPenEvent(DWORD dwMask) NN_NOEXCEPT
{
    return ((dwMask & 0xFFFFFF00) == 0xFF515700);
}

} // namespace

void ActivateWindowsProcedure() NN_NOEXCEPT
{
    ProcedureManager::GetProcedureManager().Activate();
}

void DeactivateWindowsProcedure() NN_NOEXCEPT
{
    ProcedureManager::GetProcedureManager().Deactivate();
}

bool KeepWindowsProcedureAlive() NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().KeepAlive();
}

bool GetWindowsProcedureClientAreaSize(Rectangle* pOutValue) NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().GetClientAreaSize(pOutValue);
}

bool GetWindowsProcedureMousePosition(Point* pOutValue) NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().GetMousePosition(pOutValue);
}

MouseButtonSet GetWindowsProcedureMouseButtons() NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().GetMouseButtons();
}

int32_t GetWindowsProcedureWheelDelta() NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().GetWheelDelta();
}

int GetWindowsProcedureTouchInputs(TOUCHINPUT* outBuffer, int count
                                   ) NN_NOEXCEPT
{
    return ProcedureManager::GetProcedureManager().GetTouchInputs(outBuffer,
                                                                  count);
}

namespace {

ProcedureManager& ProcedureManager::GetProcedureManager() NN_NOEXCEPT
{
    return StaticObject<ProcedureManager>::Get();
}

void ProcedureManager::Activate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化処理を実施

        m_MouseButtonMask = 0;
        m_WheelDelta = 0;
    }

    // アクティブ化された回数をインクリメント
    ++m_ActivationCount;
}

void ProcedureManager::Deactivate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_ASSERT(!m_ActivationCount.IsZero());

    // アクティブ化された回数をデクリメント
    --m_ActivationCount;
}

bool ProcedureManager::KeepAlive() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    if (::IsWindow(m_WindowHandle))
    {
        // サブクラス化されたウィンドウが有効ならば何もしない
        return true;
    }

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_ASSERT(!m_ActivationCount.IsZero());

    if (::IsWindow(m_WindowHandle))
    {
        // 排他された状態でサブクラス化されたウィンドウの有効性を再確認
        return true;
    }

    // ウィンドウプロシージャの観測値を初期化
    m_MouseButtonMask = 0;
    m_WheelDelta = 0;

    // サブクラス化の対象となるウィンドウを検索
    ::nn::vi::WindowInfo windowInfo = {};

    if (::nn::vi::ListWindows(&windowInfo, 1) > 0)
    {
        // ウィンドウが見つかった場合はサブクラス化を実行

        const HWND hwnd = windowInfo.windowHandle;

        auto defaultProcedure =
            reinterpret_cast<WNDPROC>(::GetClassLongPtr(hwnd, GCLP_WNDPROC));

        if (defaultProcedure != 0)
        {
            m_PreviousProcedure = defaultProcedure;

            auto previousProcedure =
                reinterpret_cast<WNDPROC>(
                    ::SetWindowLongPtr(
                        hwnd,
                        GWLP_WNDPROC,
                        reinterpret_cast<LONG_PTR>(ProcedureManager::WndProc)));

            if (previousProcedure != 0)
            {
                // サブクラス化に成功した場合はハンドルを更新
                m_PreviousProcedure = previousProcedure;
                m_WindowHandle = hwnd;

                return true;
            }
        }
    }

    // サブクラス化に失敗した場合はハンドルを無効化
    m_WindowHandle = 0;

    return false;
}

bool ProcedureManager::GetClientAreaSize(Rectangle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    HWND hwnd = m_WindowHandle;

    RECT rect = {};

    if (::GetClientRect(hwnd, &rect) != 0)
    {
        pOutValue->x = rect.left;
        pOutValue->y = rect.top;
        pOutValue->width = rect.right - rect.left;
        pOutValue->height = rect.bottom - rect.top;

        return true;
    }

    return false;
}

bool ProcedureManager::GetMousePosition(Point* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    HWND hwnd = m_WindowHandle;

    POINT point = {};

    if (hwnd == ::GetForegroundWindow() && ::GetCursorPos(&point) &&
        ::ScreenToClient(hwnd, &point))
    {
        // ウィンドウがフォアグラウンドかつ、
        // クライアント領域からの相対位置を取得できたならばカーソル位置は有効
        pOutValue->x = point.x;
        pOutValue->y = point.y;

        return true;
    }

    return false;
}

MouseButtonSet ProcedureManager::GetMouseButtons() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // この関数が呼び出された時点で、押されていないボタンの押下状態を解除し、
    // ウィンドウプロシージャが観測したマウスボタンの押下状態を確定

    // 左右のボタンのレイアウトがデフォルトか否かを判定
    const bool isDefaultLayout = (::GetSystemMetrics(SM_SWAPBUTTON) == 0);

    if (::GetAsyncKeyState(VK_LBUTTON) == 0)
    {
        m_MouseButtonMask &= isDefaultLayout ? ~MouseButtonMask_Left
                                             : ~MouseButtonMask_Right;
    }

    if (::GetAsyncKeyState(VK_RBUTTON) == 0)
    {
        m_MouseButtonMask &= isDefaultLayout ? ~MouseButtonMask_Right
                                             : ~MouseButtonMask_Left;
    }

    if (::GetAsyncKeyState(VK_MBUTTON) == 0)
    {
        m_MouseButtonMask &= ~MouseButtonMask_Middle;
    }

    if (::GetAsyncKeyState(VK_XBUTTON1) == 0)
    {
        m_MouseButtonMask &= ~MouseButtonMask_Back;
    }

    if (::GetAsyncKeyState(VK_XBUTTON2) == 0)
    {
        m_MouseButtonMask &= ~MouseButtonMask_Forward;
    }

    MouseButtonSet mouseButtons = {
        { static_cast<uint32_t>(m_MouseButtonMask) }
    };

    return mouseButtons;
}

int32_t ProcedureManager::GetWheelDelta() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    return m_WheelDelta.exchange(0);
}

int ProcedureManager::GetTouchInputs(TOUCHINPUT* outBuffer, int count
                                     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    count = ::std::max(0, ::std::min(count, TouchInputCountMax));

    count = m_TouchInputLifo.Read(m_TouchInputStates, count);

    for (int i = 0; i < count; ++i)
    {
        const TouchInputState& state = m_TouchInputStates[i];

        if (state.samplingNumber <= m_TouchInputTailSamplingNumber)
        {
            count = i;

            break;
        }

        outBuffer[i] = state.touchInput;
    }

    if (count > 0)
    {
        m_TouchInputTailSamplingNumber = m_TouchInputStates[0].samplingNumber;
    }

    return count;
}

LRESULT CALLBACK ProcedureManager::WndProc(HWND hwnd, UINT uMsg, WPARAM wParam,
                                           LPARAM lParam) NN_NOEXCEPT
{
    ProcedureManager& that = ProcedureManager::GetProcedureManager();

    switch (uMsg)
    {
    // マウスボタンの押下を記録
    case WM_LBUTTONDOWN:
        if (!IsPenEvent(static_cast<DWORD>(::GetMessageExtraInfo())))
        {
            that.m_MouseButtonMask |= MouseButtonMask_Left;
        }
        break;
    case WM_RBUTTONDOWN:
        that.m_MouseButtonMask |= MouseButtonMask_Right;
        break;
    case WM_MBUTTONDOWN:
        that.m_MouseButtonMask |= MouseButtonMask_Middle;
        break;
    case WM_XBUTTONDOWN:
        switch (GET_XBUTTON_WPARAM(wParam))
        {
        case XBUTTON1:
            that.m_MouseButtonMask |= MouseButtonMask_Back;
            break;
        case XBUTTON2:
            that.m_MouseButtonMask |= MouseButtonMask_Forward;
            break;
        default:
            // 何もしない
            break;
        }
        break;
    case WM_MOUSEWHEEL:
        that.m_WheelDelta += GET_WHEEL_DELTA_WPARAM(wParam);
        break;

    // タッチ入力を記録
    case WM_TOUCH:
        {
            auto hTouchInput = reinterpret_cast<HTOUCHINPUT>(lParam);
            UINT cInputs = LOWORD(wParam);
            ::std::unique_ptr<TOUCHINPUT[]> pInputs(new TOUCHINPUT[cInputs]());
            if (::GetTouchInputInfo(hTouchInput, cInputs, pInputs.get(),
                                    sizeof(TOUCHINPUT)))
            {
                for (UINT i = 0; i < cInputs; ++i)
                {
                    TouchInputState touchInputState = {
                        that.m_TouchInputHeadSamplingNumber,
                        pInputs[i]
                    };
                    TOUCHINPUT& touchInput = touchInputState.touchInput;
                    POINT point = {};
                    point.x = TOUCH_COORD_TO_PIXEL(touchInput.x);
                    point.y = TOUCH_COORD_TO_PIXEL(touchInput.y);
                    ::ScreenToClient(hwnd, &point);
                    touchInput.x = point.x;
                    touchInput.y = point.y;
                    touchInput.cxContact =
                        TOUCH_COORD_TO_PIXEL(touchInput.cxContact);
                    touchInput.cyContact =
                        TOUCH_COORD_TO_PIXEL(touchInput.cyContact);
                    that.m_TouchInputLifo.Append(touchInputState);
                    ++(that.m_TouchInputHeadSamplingNumber);
                }
                ::CloseTouchInputHandle(hTouchInput);
            }
        }
        break;

    default:
        // 何もしない
        break;
    }

    return ::CallWindowProc(that.m_PreviousProcedure, hwnd, uMsg, wParam,
                            lParam);
}

} // namespace

}}} // namespace nn::hid::detail
