﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace hid { namespace detail {

//!< 配列を扱うクラスです。
template<typename T, size_t N>
class Vector final
{
    NN_STATIC_ASSERT(0 < N);
    NN_DISALLOW_COPY(Vector);
    NN_DISALLOW_MOVE(Vector);

private:
    //!< 要素数
    size_t m_Count;

    //!< 要素
    T m_Elements[N];

public:
    Vector() NN_NOEXCEPT : m_Count(0) { /* 何もしない */ }

    //!< 要素数を返します。
    size_t GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

    //!< 指定された添字の要素を返します。
    const T& operator [](size_t i) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(i, m_Count);

        return m_Elements[i];
    }

    //!< 指定された添字の要素を返します。
    const T& operator [](int i) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(i, 0, static_cast<int>(m_Count));

        return m_Elements[i];
    }

    //!< 指定された要素を含むか否かを表す値を返します。
    bool Contains(const T& value) const NN_NOEXCEPT
    {
        size_t i = 0u;

        while (i < m_Count && value != m_Elements[i])
        {
            ++i;
        }

        return i < m_Count;
    }

    //!< 要素を末尾に追加します。
    void Append(const T& value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(m_Count, N);

        m_Elements[m_Count] = value;

        ++m_Count;
    }

    //!< 全ての要素を削除します。
    void Clear() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    //!< 指定された要素を削除します。
    void Remove(const T& value) NN_NOEXCEPT
    {
        size_t count = 0u;

        for (size_t i = 0u; i < m_Count; ++i)
        {
            if (value != m_Elements[i])
            {
                if (count < i)
                {
                    m_Elements[count] = m_Elements[i];
                }

                ++count;
            }
        }

        m_Count = count;
    }
};

}}} // namespace nn::hid::detail
