﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_Thread.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"
#include "hid_IUniquePadDriver.h"
#include "hid_SixAxisSensorCalibrator.h"
#include "hid_UniquePadXcdDriver.h"
#include "hid_UniquePadUtility.h"

namespace {

void GetSixAxisSensorNoiseCountRange(int32_t* pOutAccelerometerCountRange,
                                     int32_t* pOutAngularVelocityCountRange,
                                     const ::nn::xcd::AccelerometerFsr& accFsr,
                                     const ::nn::xcd::GyroscopeFsr& gyroFsr)
{
    // オフセットずれを考慮した閾値を ±0.2G とする
    // カウント数 = 0.2[G] / (FSR[G] * 2 / 2^16[count])
    switch(accFsr)
    {
    case ::nn::xcd::AccelerometerFsr_2G:
        *pOutAccelerometerCountRange = 3276;
        break;
    case ::nn::xcd::AccelerometerFsr_4G:
        *pOutAccelerometerCountRange = 1638;
        break;
    case ::nn::xcd::AccelerometerFsr_8G:
        *pOutAccelerometerCountRange = 819;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 静止ノイズは ±1.8dps http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=101846065
    // カウント数 = 1.8[dps] / (FSR[dps] * 2 / 2^16[count])
    switch(gyroFsr)
    {
    case ::nn::xcd::GyroscopeFsr_250dps:
        *pOutAngularVelocityCountRange = 236;
        break;
    case ::nn::xcd::GyroscopeFsr_500dps:
        *pOutAngularVelocityCountRange = 118;
        break;
    case ::nn::xcd::GyroscopeFsr_1000dps:
        *pOutAngularVelocityCountRange = 59;
        break;
    case ::nn::xcd::GyroscopeFsr_2000dps:
        *pOutAngularVelocityCountRange = 30;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    const float Coef = 2.5f; // 静止ノイズの Coef 倍のマージンを持たせる
    *pOutAccelerometerCountRange = static_cast<int32_t>(*pOutAccelerometerCountRange * Coef);
    *pOutAngularVelocityCountRange = static_cast<int32_t>(*pOutAngularVelocityCountRange * Coef);
}

nn::Result GetTypicalAccelerometer(nn::util::Float3* pOutCalAccelerometer,
                                   nn::xcd::DeviceHandle& handle) NN_NOEXCEPT
{
    // デバイスタイプを取得
    nn::xcd::DeviceInfo info = {};
    auto result = nn::xcd::GetDeviceInfo(&info, handle);

    if (result.IsFailure())
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // デバイス種別ごとに計算される理論値
    // 極性はセンサー基板の接地面により変わる
    const float GravityCountFsr2g = 16384.f;
    switch (info.deviceType)
    {
    case nn::xcd::DeviceType::DeviceType_FullKey:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_Left:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_Right:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = - GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_MiyabiLeft:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_MiyabiRight:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = - GravityCountFsr2g;
        break;
    default:
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorNotSupported());
    }

    NN_RESULT_SUCCESS;
}

::nn::hid::system::UniquePadSerialNumber ConvertToUniquePadSerialNumber(const ::nn::xcd::IdentificationCode& code)
{
    NN_ABORT_UNLESS_GREATER_EQUAL(::nn::hid::system::UniquePadSerialNumberLength, ::nn::xcd::IdentificationCodeLength);

    ::nn::hid::system::UniquePadSerialNumber serialNumber;
    std::memcpy(serialNumber._storage, code._code, ::nn::xcd::IdentificationCodeLength);

    return serialNumber;
}

::nn::hid::system::UniquePadSerialNumber ConvertToUniquePadSerialNumberFromBdAddr(const ::nn::bluetooth::Address& address)
{
    ::nn::hid::system::UniquePadSerialNumber serialNumber;
    std::memcpy(serialNumber._storage, address.address, sizeof(address));

    return serialNumber;
}

::nn::hid::system::UniquePadSerialNumber ConvertToUniquePadAbstractedPadId(const ::nn::hid::detail::AbstractedPadId& abstractedPadId)
{
    ::nn::hid::system::UniquePadSerialNumber serialNumber;
    std::memset(serialNumber._storage, 0xff, sizeof(serialNumber));
    std::memcpy(serialNumber._storage, &abstractedPadId, sizeof(abstractedPadId));
    return serialNumber;
}

const int ReleasePositionThreshold  = 400;    // リリース位置の範囲

} // namespace

namespace nn { namespace hid { namespace detail {

namespace {

AnalogStickState CastAnalogStickState(::nn::xcd::AnalogStickState& xcdState)
{
    AnalogStickState state = { static_cast<int32_t>(xcdState.x), static_cast<int32_t>(xcdState.y) };
    return state;
}

::nn::xcd::AnalogStickState CastAnalogStickState(AnalogStickState& xcdState)
{
    ::nn::xcd::AnalogStickState state = { static_cast<int16_t>(xcdState.x), static_cast<int16_t>(xcdState.y) };
    return state;
}

} // namespace

UniquePadXcdDriver::UniquePadXcdDriver() NN_NOEXCEPT
    : m_IsActivated(false)
    , m_Calibrator()
    , m_SixAxisSensorUserCalibrationStage(system::SixAxisSensorUserCalibrationStage_Completed)
{
    for (auto& pManager : m_pAnalogStickCalibration)
    {
        pManager = nullptr;
    }
}

UniquePadXcdDriver::~UniquePadXcdDriver() NN_NOEXCEPT
{
    // 何もしない
}

void UniquePadXcdDriver::AttachAbstractedPad(IAbstractedPad* pPad) NN_NOEXCEPT
{
    m_pPad = pPad;
}

bool UniquePadXcdDriver::IsActivated() const NN_NOEXCEPT
{
    return m_IsActivated;
}

void UniquePadXcdDriver::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPad);
    if (m_IsActivated == false)
    {
        // キャリブレーションのステージを初期化
        InitializeAnalogStickManualCalibration();
        m_SixAxisSensorUserCalibrationStage = system::SixAxisSensorUserCalibrationStage_Completed;

        for(auto& analogStick : m_pAnalogStickCalibration)
        {
            analogStick->Activate();
        }
        m_IsActivated = true;
    }
}

void UniquePadXcdDriver::Deactivate() NN_NOEXCEPT
{
    if (m_IsActivated == true)
    {
        // キャリブレーションのステージを初期化
        InitializeAnalogStickManualCalibration();
        m_SixAxisSensorUserCalibrationStage = system::SixAxisSensorUserCalibrationStage_Completed;

        for (auto& analogStick : m_pAnalogStickCalibration)
        {
            analogStick->Deactivate();
        }
        m_IsActivated = false;
    }
}

void UniquePadXcdDriver::SetAnalogStickCalibrationManager(AnalogStickCalibrationManager pManager[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_EQUAL(count, AnalogStickPositionCountMax);
    NN_SDK_REQUIRES(!m_IsActivated);
    for (int i = 0; i < count; i++)
    {
        m_pAnalogStickCalibration[i] = &pManager[i];
    }
}

IAbstractedPad* UniquePadXcdDriver::GetAbstractedPad() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsActivated == true, "UniquePadXcdDriver is not activated");
    return m_pPad;
}

bool UniquePadXcdDriver::IsSixAxisSensorUserCalibrationSupported() const NN_NOEXCEPT
{
    return this->IsSixAxisSensorUserCalibrationSupportedImpl().IsSuccess();
}

::nn::Result UniquePadXcdDriver::ResetSixAxisSensorCalibrationValues() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsSixAxisSensorUserCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        GetXcdDeviceHandle(&xcdHandle),
        system::ResultSixAxisSensorNotSupported());

    // 工程検査時にリセット
    auto result = nn::xcd::ResetSensorCalibrationValue(xcdHandle);

    if (::nn::xcd::ResultNotConnected::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }
    else if (::nn::xcd::ResultSerialFlashOnWrite::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorWriteFailure());
    }

    NN_RESULT_SUCCESS;
}

//!< 6軸センサーのユーザーキャリブレーション処理を開始します。
::nn::Result UniquePadXcdDriver::StartSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsSixAxisSensorUserCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        GetXcdDeviceHandle(&xcdHandle),
        system::ResultSixAxisSensorNotSupported());

    // モデル情報( SerialFlash に書かれた設置時の加速度カウント値)を読み出す
    nn::xcd::SensorState horizontalAccelerometerCount = {};
    auto result = nn::xcd::GetSensorHorizontalOffset(&horizontalAccelerometerCount,
                                                     xcdHandle);
    if (result.IsFailure())
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // FSR を指定する
    ::nn::xcd::AccelerometerFsr accFsr = ::nn::xcd::AccelerometerFsr::AccelerometerFsr_2G;
    ::nn::xcd::GyroscopeFsr     gyroFsr = ::nn::xcd::GyroscopeFsr::GyroscopeFsr_2000dps;

    // 現在の FSR 設定からズレの許容カウント数を取得する
    int32_t   accelerometerCountRange;
    int32_t   angularVelocityCountRange;
    GetSixAxisSensorNoiseCountRange(&accelerometerCountRange,
                                    &angularVelocityCountRange,
                                    accFsr,
                                    gyroFsr);

    // キャリブレーション値を取得
    ::nn::xcd::SensorCalibrationValue calibrationValue = {};
    result = ::nn::xcd::GetSensorCalibrationValue(&calibrationValue,
                                                  xcdHandle);

    if (::nn::xcd::ResultNotConnected::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // Typical 値を取得
    ::nn::util::Float3 typicalAccelerometer = {};
    NN_RESULT_DO(GetTypicalAccelerometer(&typicalAccelerometer,
                                         xcdHandle));

    // キャリブレーションに係るパラメータ設定を Calibrator に反映させる
    m_Calibrator.Reset();
    const int ValidSampleCountMax = 200; // 標本平均値をとるサンプル数
    m_Calibrator.SetCalibrationParameters(horizontalAccelerometerCount,
                                          accelerometerCountRange,
                                          angularVelocityCountRange,
                                          ValidSampleCountMax,
                                          calibrationValue,
                                          typicalAccelerometer);

    m_SixAxisSensorUserCalibrationStage = system::SixAxisSensorUserCalibrationStage_Measuring;
    NN_RESULT_SUCCESS;
}

//!< 6軸センサーのユーザーキャリブレーション処理をキャンセルします。
::nn::Result UniquePadXcdDriver::CancelSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    m_Calibrator.Reset();
    m_SixAxisSensorUserCalibrationStage = system::SixAxisSensorUserCalibrationStage_Completed;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::StartAnalogStickManualCalibration(system::AnalogStickPosition position) NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsAnalogStickCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        GetXcdDeviceHandle(&xcdHandle),
        system::ResultAnalogStickManualCalibrationNotSupported());

    ::nn::xcd::DeviceInfo deviceInfo;
    if (::nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsFailure())
    {
        NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
    }

    ::nn::xcd::AnalogStickDeviceParameter parameter;
    switch (position)
    {
    case system::AnalogStickPosition_Left:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Right)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::GetLeftAnalogStickDeviceParameter(&parameter, xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Left)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::GetLeftAnalogStickDeviceParameter(&parameter, xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto originMin = CastAnalogStickState(parameter.originRangeMin);
    auto originMax = CastAnalogStickState(parameter.originRangeMax);
    auto strokePositive = CastAnalogStickState(parameter.minimumStrokePositive);
    auto strokeNegative = CastAnalogStickState(parameter.minimumStrokeNegative);

    NN_RESULT_DO(
        m_pAnalogStickCalibration[GetIndexFromAnalogStickPosition(position)]->StartAnalogStickManualCalibration(
            originMin,
            originMax,
            strokePositive,
            strokeNegative,
            ReleasePositionThreshold
        )
    );

    m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = system::AnalogStickManualCalibrationStage_ReleaseFromRight;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::RetryCurrentAnalogStickManualCalibrationStage(system::AnalogStickPosition position) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAnalogStickCalibration[GetIndexFromAnalogStickPosition(position)]->RetryCurrentAnalogStickManualCalibration());
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::CancelAnalogStickManualCalibration(system::AnalogStickPosition position) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAnalogStickCalibration[GetIndexFromAnalogStickPosition(position)]->CancelAnalogStickManualCalibration());
    InitializeAnalogStickManualCalibration();
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::ResetAnalogStickManualCalibration(system::AnalogStickPosition position) NN_NOEXCEPT
{
    //Reset は Result を返さないので内部ステートだけ変更して結果は必ず NN_RESULT_SUCCESS
    if (this->ResetAnalogStickManualCalibrationImpl(position).IsSuccess())
    {
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = system::AnalogStickManualCalibrationStage_Clear;
    }
    else
    {
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = system::AnalogStickManualCalibrationStage_ClearCompleted;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::ResetAnalogStickManualCalibrationImpl(system::AnalogStickPosition position) NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsAnalogStickCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        GetXcdDeviceHandle(&xcdHandle),
        system::ResultAnalogStickManualCalibrationNotSupported());

    ::nn::xcd::DeviceInfo deviceInfo;
    if (::nn::xcd::GetDeviceInfo(&deviceInfo, xcdHandle).IsFailure())
    {
        NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
    }

    switch (position)
    {
    case system::AnalogStickPosition_Left:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Right)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::ResetLeftAnalogStickValidRange(xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        if (deviceInfo.deviceType == ::nn::xcd::DeviceType_Left)
        {
            NN_RESULT_THROW(system::ResultAnalogStickDeviceNotConnected());
        }
        ::nn::xcd::ResetRightAnalogStickValidRange(xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

void UniquePadXcdDriver::ProceedAnalogStick(AnalogStickCalibrationStateImpl* pOutState, system::AnalogStickPosition position) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    pOutState->flags.Reset();
    pOutState->stage = system::AnalogStickManualCalibrationStage_Completed;
    pOutState->state = AnalogStickState();

    if (m_IsActivated == false ||
        this->IsDeviceConnected() == false)
    {
        return;
    }

    auto padState = m_pPad->GetPadState();

    switch (position)
    {
    case system::AnalogStickPosition_Left:
        if (m_pPad->GetDeviceType().Test<system::DeviceType::JoyConRight>() ||
            m_pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyRight>())
        {
            return;
        }
        pOutState->flags.Set<AnalogStickCalibrationFlags::IsButtonPressed>(
            padState.buttons.Test<AbstractedPadButton::StickL>());
        break;
    case system::AnalogStickPosition_Right:
        if (m_pPad->GetDeviceType().Test<system::DeviceType::JoyConLeft>() ||
            m_pPad->GetDeviceType().Test<system::DeviceType::HandheldJoyLeft>())
        {
            return;
        }
        pOutState->flags.Set<AnalogStickCalibrationFlags::IsButtonPressed>(
            padState.buttons.Test<AbstractedPadButton::StickR>());
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] ==
        system::AnalogStickManualCalibrationStage_Completed ||
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] ==
        system::AnalogStickManualCalibrationStage_ClearCompleted)
    {
        // キャリブレーション中じゃないときの処理
        switch (position)
        {
        case system::AnalogStickPosition_Left:
            pOutState->state = padState.analogStickL;
            break;
        case system::AnalogStickPosition_Right:
            pOutState->state = padState.analogStickR;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        // キャリブレーション中の処理
        nn::xcd::DeviceHandle xcdHandle;
        if (GetXcdDeviceHandle(&xcdHandle) == false)
        {
            // 通常はここにはこない
            return;
        }

        ::nn::xcd::PadState xcdState;
        ::nn::xcd::AnalogStickState* pAnalogStickState;
        ::nn::xcd::GetPadState(&xcdState, xcdHandle);
        switch (position)
        {
        case system::AnalogStickPosition_Left:
            pAnalogStickState = &xcdState.analogStickL;
            break;
        case system::AnalogStickPosition_Right:
            pAnalogStickState = &xcdState.analogStickR;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        AnalogStickState convertedState;

        switch (m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)])
        {
        case system::AnalogStickManualCalibrationStage_Completed:
        case system::AnalogStickManualCalibrationStage_ClearCompleted:
            // ここには到達しない
            break;
        case system::AnalogStickManualCalibrationStage_ReleaseFromRight:
        case system::AnalogStickManualCalibrationStage_ReleaseFromBottom:
        case system::AnalogStickManualCalibrationStage_ReleaseFromLeft:
        case system::AnalogStickManualCalibrationStage_ReleaseFromTop:
        case system::AnalogStickManualCalibrationStage_Rotate:
            convertedState = CastAnalogStickState(*pAnalogStickState);
            ProceedAnalogStickManualCalibration(pOutState, convertedState, position);
            break;
        case system::AnalogStickManualCalibrationStage_Update:
        case system::AnalogStickManualCalibrationStage_Clear:
            ProceedAnalogStickCalibrationUpdate(pOutState, position);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    // アナログスティックボタンの状態を更新
    pOutState->stage = static_cast<system::AnalogStickManualCalibrationStage>(m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)]);
}

void UniquePadXcdDriver::ProceedAnalogStickManualCalibration(AnalogStickCalibrationStateImpl* pOutState,
                                                             AnalogStickState& state,
                                                             system::AnalogStickPosition position) NN_NOEXCEPT
{
    if (this->IsAnalogStickCalibrationSupportedImpl().IsFailure())
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // キャリブレーション補正処理
    m_pAnalogStickCalibration[GetIndexFromAnalogStickPosition(position)]->ProceedAnalogStickCalibration(pOutState, state);
    // キャリブレーションが完了したかどうかをチェック
    if (pOutState->stage == system::AnalogStickManualCalibrationStage_Completed)
    {
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = system::AnalogStickManualCalibrationStage_Update;

        // キャリブレーションが行われた結果の取得
        AnalogStickState origin;
        AnalogStickState circuitMin;
        AnalogStickState circuitMax;
        m_pAnalogStickCalibration[GetIndexFromAnalogStickPosition(position)]->GetAnalogStickCalibrationValue(&origin, &circuitMin, &circuitMax);

        // Xcd の型に変換
        ::nn::xcd::AnalogStickValidRange updateValue = { CastAnalogStickState(origin),
                                                         CastAnalogStickState(circuitMin),
                                                         CastAnalogStickState(circuitMax) };

        // 更新を行う
        switch (position)
        {
        case system::AnalogStickPosition_Left:
            ::nn::xcd::UpdateLeftAnalogStickValidRange(updateValue, xcdHandle);
            break;
        case system::AnalogStickPosition_Right:
            ::nn::xcd::UpdateRightAnalogStickValidRange(updateValue, xcdHandle);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
    else
    {
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = GetAnalogStickCalibrationStageFromInt(pOutState->stage);
    }
}

void UniquePadXcdDriver::ProceedAnalogStickCalibrationUpdate(AnalogStickCalibrationStateImpl* pOutState,
                                                             system::AnalogStickPosition position) NN_NOEXCEPT
{
    if (this->IsAnalogStickCalibrationSupportedImpl().IsFailure())
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // ステートの更新
    pOutState->stage = m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)];
    pOutState->state = AnalogStickState();

    ::nn::xcd::CalibrationUpdateStatus status;
    // 更新状況を取得
    switch (position)
    {
    case system::AnalogStickPosition_Left:
        ::nn::xcd::GetLeftAnalogStickValidRangeUpdateStatus(&status, xcdHandle);
        break;
    case system::AnalogStickPosition_Right:
        ::nn::xcd::GetRightAnalogStickValidRangeUpdateStatus(&status, xcdHandle);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    // 更新が行われていれば完了へ
    if (status == ::nn::xcd::CalibrationUpdateStatus_None)
    {
        m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] = (m_AnalogStickCalibrationStage[GetIndexFromAnalogStickPosition(position)] == system::AnalogStickManualCalibrationStage_Update) ?
            system::AnalogStickManualCalibrationStage_Completed :
            system::AnalogStickManualCalibrationStage_ClearCompleted;
    }
}

void UniquePadXcdDriver::ProceedSixAxisSensor() NN_NOEXCEPT
{
    if (m_SixAxisSensorUserCalibrationStage == system::SixAxisSensorUserCalibrationStage_Completed)
    {
        return;
    }

    if (this->IsDeviceConnected() == false)
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // 6軸センサーのカウント値を取得
    int count = 0;
    nn::xcd::SixAxisSensorState states[nn::xcd::SixAxisSensorSampleCountMax] = {};
    auto result = ::nn::xcd::GetSensorStates(&count,
                                             states,
                                             nn::xcd::SixAxisSensorSampleCountMax,
                                             xcdHandle);

    if(::nn::xcd::ResultNotConnected::Includes(result))
    {
        return;
    }

    // 取得したカウント値を Calibrator に注入
    for (int j = count - 1; j >= 0; j--)
    {
        m_Calibrator.Update(states[j]);
    }

    // キャリブレーション完了時
    if(m_Calibrator.IsCalibrated())
    {
        m_SixAxisSensorUserCalibrationStage = system::SixAxisSensorUserCalibrationStage_Completed;

        if (m_Calibrator.IsAtRest() && m_Calibrator.IsHorizontal())
        {
            nn::xcd::SensorCalibrationValue calibrationValue;
            result = m_Calibrator.GetUserCalibrationValue(&calibrationValue);

            if (result.IsFailure())
            {
                return;
            }

            // ユーザーキャリブレーション値を更新
            result = ::nn::xcd::UpdateSensorCalibrationValue(calibrationValue,
                                                             xcdHandle);
            if (result.IsFailure())
            {
                return;
            }
        }
    }
}

void UniquePadXcdDriver::GetSixAxisSensorUserCalibrationState(SixAxisSensorUserCalibrationState* pOutState) const NN_NOEXCEPT
{
    pOutState->stage = m_SixAxisSensorUserCalibrationStage;
    pOutState->flags.Reset();

    // Flag の設定
    if (m_Calibrator.IsAtRest() == false)
    {
        pOutState->flags.Set<SixAxisSensorUserCalibrationFlags::IsNotAtRest>();
    }

    if (m_Calibrator.IsHorizontal() == false)
    {
        pOutState->flags.Set<SixAxisSensorUserCalibrationFlags::IsNotHorizontal>();
    }
}

void UniquePadXcdDriver::GetUniquePadConfig(UniquePadConfig* pOutConfig) const NN_NOEXCEPT
{
    if (this->IsDeviceConnected() == false)
    {
        ::nn::hid::detail::UniquePadConfig config = {};
        *pOutConfig = config;
        return;
    }

    auto deviceType = m_pPad->GetDeviceType();

    ::nn::hid::detail::UniquePadConfig config = {
        ConvertToUniquePadType(deviceType),                           // type
        ConvertToUniquePadInterface(m_pPad->GetInterfaceType()),      // interface
        ::nn::hid::system::UniquePadSerialNumber(),                   // serialNumber
        m_pPad->GetControllerNumber(),                                // controllerNumber
        true,                                                         // isActive
        0,                                                            // samplingNumber
    };

    nn::xcd::DeviceHandle xcdHandle;
    if (GetXcdDeviceHandle(&xcdHandle))
    {

        if (deviceType.Test<system::DeviceType::JoyConLeft>() ||
            deviceType.Test<system::DeviceType::HandheldJoyLeft>() ||
            deviceType.Test<system::DeviceType::JoyConRight>() ||
            deviceType.Test<system::DeviceType::HandheldJoyRight>())
        {
            nn::xcd::IdentificationCode code;
            nn::xcd::GetIdentificationCode(&code, xcdHandle);
            config.serialNumber = ConvertToUniquePadSerialNumber(code);
        }
        else if(deviceType.Test<system::DeviceType::FullKeyController>())
        {
            nn::xcd::DeviceInfo info;
            nn::xcd::GetDeviceInfo(&info, xcdHandle);
            config.serialNumber = ConvertToUniquePadSerialNumberFromBdAddr(info.address);
        }
    }
    else
    {
        config.serialNumber = ConvertToUniquePadAbstractedPadId(m_pPad->GetId());
    }

    *pOutConfig = config;
}

Result UniquePadXcdDriver::GetBluetoothAddress(::nn::bluetooth::Address* pOutAddress) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        this->IsDeviceConnected(),
        system::ResultUniquePadDisconnected());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        GetXcdDeviceHandle(&xcdHandle),
        system::ResultUniquePadNoBluetoothAddress());

    ::nn::xcd::DeviceInfo info;
    NN_RESULT_THROW_UNLESS(
        ::nn::xcd::GetDeviceInfo(&info, xcdHandle).IsSuccess(),
        system::ResultUniquePadDisconnected());

    if(info.deviceType == ::nn::xcd::DeviceType_MiyabiLeft ||
       info.deviceType == ::nn::xcd::DeviceType_MiyabiRight)
    {
        return system::ResultUniquePadNoBluetoothAddress();
    }

    *pOutAddress = info.address;
    NN_RESULT_SUCCESS;
}

Result UniquePadXcdDriver::Disconnect() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        this->IsDeviceConnected(),
        system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(
        m_pPad->GetInterfaceType() == system::InterfaceType_Bluetooth || m_pPad->GetInterfaceType() == system::InterfaceType_Usb,
        system::ResultUniquePadNoWirelessConnection());
    m_pPad->Detach();
    NN_RESULT_SUCCESS;
}

bool UniquePadXcdDriver::IsUsbConnected() NN_NOEXCEPT
{
    if (this->IsDeviceConnected())
    {
        return false;
    }
    return m_pPad->IsUsbConnected();
}

void UniquePadXcdDriver::UpdateControllerColor(nn::util::Color4u8Type& mainColor, nn::util::Color4u8Type& subColor) NN_NOEXCEPT
{
    if (this->IsDeviceConnected() == false)
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    ::nn::xcd::UpdateDeviceColor(mainColor, subColor, xcdHandle);
}

void UniquePadXcdDriver::InitializeAnalogStickManualCalibration() NN_NOEXCEPT
{
    for (auto& stage : m_AnalogStickCalibrationStage)
    {
        stage = system::AnalogStickManualCalibrationStage_Completed;
    }
}

bool UniquePadXcdDriver::IsDeviceConnected() const NN_NOEXCEPT
{
    if (m_pPad == nullptr || m_IsActivated == false)
    {
        return false;
    }
    return m_pPad->IsConnected();
}

bool UniquePadXcdDriver::GetXcdDeviceHandle(nn::xcd::DeviceHandle* pOutHandle) const NN_NOEXCEPT
{
    if (m_pPad->GetType() != AbstractedPadType_Xcd)
    {
        return false;
    }

    *pOutHandle = static_cast<AbstractedPadXcd*>(m_pPad)->GetXcdDeviceHandle();
    return true;
}

::nn::Result UniquePadXcdDriver::IsSixAxisSensorUserCalibrationSupportedImpl() const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(this->IsDeviceConnected(), system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(m_pPad->GetFeatureSet().Test<AbstractedPadFeature::SixAxisSensorUserCal>(),
        system::ResultSixAxisSensorNotSupported());
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadXcdDriver::IsAnalogStickCalibrationSupportedImpl() const NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(this->IsDeviceConnected(), system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(m_pPad->GetFeatureSet().Test<AbstractedPadFeature::AnalogStickUserCal>(),
        system::ResultAnalogStickManualCalibrationNotSupported());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
