﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_SixAxisSensorCalibrator.h"
#include "hid_UniquePadId.h"
#include "hid_UniquePadSixAxisSensorDriver.h"

namespace {

void GetSixAxisSensorNoiseCountRange(int32_t* pOutAccelerometerCountRange,
                                     int32_t* pOutAngularVelocityCountRange,
                                     const ::nn::xcd::AccelerometerFsr& accFsr,
                                     const ::nn::xcd::GyroscopeFsr& gyroFsr)
{
    // オフセットずれを考慮した閾値を ±0.2G とする
    // カウント数 = 0.2[G] / (FSR[G] * 2 / 2^16[count])
    switch(accFsr)
    {
    case ::nn::xcd::AccelerometerFsr_2G:
        *pOutAccelerometerCountRange = 3276;
        break;
    case ::nn::xcd::AccelerometerFsr_4G:
        *pOutAccelerometerCountRange = 1638;
        break;
    case ::nn::xcd::AccelerometerFsr_8G:
        *pOutAccelerometerCountRange = 819;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 静止ノイズは ±1.8dps http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=101846065
    // カウント数 = 1.8[dps] / (FSR[dps] * 2 / 2^16[count])
    switch(gyroFsr)
    {
    case ::nn::xcd::GyroscopeFsr_250dps:
        *pOutAngularVelocityCountRange = 236;
        break;
    case ::nn::xcd::GyroscopeFsr_500dps:
        *pOutAngularVelocityCountRange = 118;
        break;
    case ::nn::xcd::GyroscopeFsr_1000dps:
        *pOutAngularVelocityCountRange = 59;
        break;
    case ::nn::xcd::GyroscopeFsr_2000dps:
        *pOutAngularVelocityCountRange = 30;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    const float Coef = 2.5f; // 静止ノイズの Coef 倍のマージンを持たせる
    *pOutAccelerometerCountRange = static_cast<int32_t>(*pOutAccelerometerCountRange * Coef);
    *pOutAngularVelocityCountRange = static_cast<int32_t>(*pOutAngularVelocityCountRange * Coef);
}

nn::Result GetTypicalAccelerometer(nn::util::Float3* pOutCalAccelerometer,
                                   nn::xcd::DeviceHandle& handle) NN_NOEXCEPT
{
    // デバイスタイプを取得
    nn::xcd::DeviceInfo info = {};
    auto result = nn::xcd::GetDeviceInfo(&info, handle);

    if (result.IsFailure())
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // デバイス種別ごとに計算される理論値
    // 極性はセンサー基板の接地面により変わる
    const float GravityCountFsr2g = 16384.f;
    switch (info.deviceType)
    {
    case nn::xcd::DeviceType::DeviceType_FullKey:
    case nn::xcd::DeviceType::DeviceType_Tarragon:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_Left:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_Right:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = - GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_MiyabiLeft:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = GravityCountFsr2g;
        break;
    case nn::xcd::DeviceType::DeviceType_MiyabiRight:
        pOutCalAccelerometer->x = 0.0f;
        pOutCalAccelerometer->y = 0.0f;
        pOutCalAccelerometer->z = - GravityCountFsr2g;
        break;
    default:
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorNotSupported());
    }

    NN_RESULT_SUCCESS;
}

} // namespace

namespace nn { namespace hid { namespace detail {

UniquePadSixAxisSensorDriver::UniquePadSixAxisSensorDriver() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_Calibrator()
    , m_Stage(system::SixAxisSensorUserCalibrationStage_Completed)
{
    // 何もしない
}

UniquePadSixAxisSensorDriver::~UniquePadSixAxisSensorDriver() NN_NOEXCEPT
{
    // 何もしない
}

void UniquePadSixAxisSensorDriver::SetUniquePadAbstractedPadHolder(UniquePadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    m_pAbstractedPadHolder = pHolder;
}

::nn::Result UniquePadSixAxisSensorDriver::StartSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsSixAxisSensorUserCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle),
        system::ResultSixAxisSensorNotSupported());

    // モデル情報( SerialFlash に書かれた設置時の加速度カウント値)を読み出す
    nn::xcd::SensorState horizontalAccelerometerCount = {};
    auto result = nn::xcd::GetSensorHorizontalOffset(&horizontalAccelerometerCount,
                                                     xcdHandle);
    if (result.IsFailure())
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // FSR を指定する
    ::nn::xcd::AccelerometerFsr accFsr = ::nn::xcd::AccelerometerFsr::AccelerometerFsr_2G;
    ::nn::xcd::GyroscopeFsr     gyroFsr = ::nn::xcd::GyroscopeFsr::GyroscopeFsr_2000dps;

    // 現在の FSR 設定からズレの許容カウント数を取得する
    int32_t   accelerometerCountRange;
    int32_t   angularVelocityCountRange;
    GetSixAxisSensorNoiseCountRange(&accelerometerCountRange,
                                    &angularVelocityCountRange,
                                    accFsr,
                                    gyroFsr);

    // キャリブレーション値を取得
    ::nn::xcd::SensorCalibrationValue calibrationValue = {};
    result = ::nn::xcd::GetSensorCalibrationValue(&calibrationValue,
                                                  xcdHandle);

    if (::nn::xcd::ResultNotConnected::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }

    // Typical 値を取得
    ::nn::util::Float3 typicalAccelerometer = {};
    NN_RESULT_DO(GetTypicalAccelerometer(&typicalAccelerometer,
                                         xcdHandle));

    // キャリブレーションに係るパラメータ設定を Calibrator に反映させる
    m_Calibrator.Reset();
    const int ValidSampleCountMax = 200; // 標本平均値をとるサンプル数
    m_Calibrator.SetCalibrationParameters(horizontalAccelerometerCount,
                                          accelerometerCountRange,
                                          angularVelocityCountRange,
                                          ValidSampleCountMax,
                                          calibrationValue,
                                          typicalAccelerometer);

    m_Stage = system::SixAxisSensorUserCalibrationStage_Measuring;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadSixAxisSensorDriver::CancelSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    m_Calibrator.Reset();
    m_Stage = system::SixAxisSensorUserCalibrationStage_Completed;
    NN_RESULT_SUCCESS;
}

::nn::Result UniquePadSixAxisSensorDriver::GetSixAxisSensorUserCalibrationStage(system::SixAxisSensorUserCalibrationStage* pOutValue) NN_NOEXCEPT
{
    *pOutValue = m_Stage;
    if (*pOutValue == system::SixAxisSensorUserCalibrationStage_Completed)
    {
        if (m_Calibrator.IsAtRest() == false)
        {
            NN_RESULT_THROW(system::ResultSixAxisSensorNotStable());
        }
        if (m_Calibrator.IsHorizontal() == false)
        {
            NN_RESULT_THROW(system::ResultSixAxisSensorNotHorizontal());
        }
    }
    NN_RESULT_SUCCESS;
}

bool UniquePadSixAxisSensorDriver::IsSixAxisSensorUserCalibrationSupported() const NN_NOEXCEPT
{
    return this->IsSixAxisSensorUserCalibrationSupportedImpl().IsSuccess();
}

::nn::Result UniquePadSixAxisSensorDriver::ResetSixAxisSensorCalibrationValues() NN_NOEXCEPT
{
    NN_RESULT_DO(this->IsSixAxisSensorUserCalibrationSupportedImpl());

    nn::xcd::DeviceHandle xcdHandle;
    NN_RESULT_THROW_UNLESS(
        m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle),
        system::ResultSixAxisSensorNotSupported());

    // 工程検査時にリセット
    auto result = nn::xcd::ResetSensorCalibrationValue(xcdHandle);

    if (::nn::xcd::ResultNotConnected::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorDisconnected());
    }
    else if (::nn::xcd::ResultSerialFlashOnWrite::Includes(result))
    {
        NN_RESULT_THROW(::nn::hid::system::ResultSixAxisSensorWriteFailure());
    }

    NN_RESULT_SUCCESS;
}

void UniquePadSixAxisSensorDriver::Update() NN_NOEXCEPT
{
    if (m_Stage == system::SixAxisSensorUserCalibrationStage_Completed)
    {
        return;
    }

    if (m_pAbstractedPadHolder->IsConnected() == false)
    {
        return;
    }

    nn::xcd::DeviceHandle xcdHandle;
    if (m_pAbstractedPadHolder->GetXcdDeviceHandle(&xcdHandle) == false)
    {
        return;
    }

    // 6軸センサーのカウント値を取得
    int count = 0;
    nn::xcd::SixAxisSensorState states[nn::xcd::SixAxisSensorSampleCountMax] = {};
    auto result = ::nn::xcd::GetSensorStates(&count,
                                             states,
                                             nn::xcd::SixAxisSensorSampleCountMax,
                                             xcdHandle);

    if(::nn::xcd::ResultNotConnected::Includes(result))
    {
        return;
    }

    // 取得したカウント値を Calibrator に注入
    for (int j = count - 1; j >= 0; j--)
    {
        m_Calibrator.Update(states[j]);
    }

    // キャリブレーション完了時
    if(m_Calibrator.IsCalibrated())
    {
        m_Stage = system::SixAxisSensorUserCalibrationStage_Completed;

        if (m_Calibrator.IsAtRest() && m_Calibrator.IsHorizontal())
        {
            nn::xcd::SensorCalibrationValue calibrationValue;
            result = m_Calibrator.GetUserCalibrationValue(&calibrationValue);

            if (result.IsFailure())
            {
                return;
            }

            // ユーザーキャリブレーション値を更新
            result = ::nn::xcd::UpdateSensorCalibrationValue(calibrationValue,
                                                             xcdHandle);
            if (result.IsFailure())
            {
                return;
            }
        }
    }
}

void UniquePadSixAxisSensorDriver::Reset() NN_NOEXCEPT
{
    m_Stage = system::SixAxisSensorUserCalibrationStage_Completed;
    m_Calibrator.Reset();
}

::nn::Result UniquePadSixAxisSensorDriver::IsSixAxisSensorUserCalibrationSupportedImpl() const NN_NOEXCEPT
{
    IAbstractedPad* pPad;
    NN_RESULT_THROW_UNLESS(m_pAbstractedPadHolder->GetIAbstractedPad(&pPad), system::ResultUniquePadDisconnected());
    NN_RESULT_THROW_UNLESS(pPad->GetFeatureSet().Test<AbstractedPadFeature::SixAxisSensorUserCal>(),
        system::ResultSixAxisSensorNotSupported());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
