﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_GestureLifo.h"
#include "hid_HandheldManager.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_TouchScreenLifo.h"
#include "hid_TouchScreenManager.h"
#include "hid_TouchScreenStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Gesture のアプレットリソースを初期化します。
void InitializeGestureAppletResource(
    GestureSharedMemoryFormat* address,
    GestureContext* pContext,
    GestureState* pState,
    int32_t version) NN_NOEXCEPT;

//!< Gesture のアプレットリソースを更新します。
void UpdateGestureAppletResource(
    GestureSharedMemoryFormat* address,
    GestureContext* pContext,
    GestureState* pState,
    const GestureState& masterState,
    bool enablesInput) NN_NOEXCEPT;

//!< 指定されたタッチが存在するか否かを表す値を返します。
template<typename T>
bool FindsTouch(
    int32_t fingerId, const T* head, const T* tail,
    int32_t(*getter)(const T& value) NN_NOEXCEPT) NN_NOEXCEPT;

//!< TouchScreen の入力状態を計算します。
void CalculateTouchScreenState(
    TouchScreenState<TouchStateCountMax>* pOutValue,
    TouchScreenContext* pTouchScreenContext,
    const TouchScreenState<TouchStateCountMax>& state,
    bool enablesInput) NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan TouchScreenManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(4);

TouchScreenManager::TouchScreenManager() NN_NOEXCEPT
    : m_CommonActivationCount()
    , m_GestureActivationCount()
    , m_TouchScreenActivationCount()
    , m_IsAwake(false)
    , m_SamplingNumber(0)
    , m_pTimerEvent(nullptr)
    , m_pDriver(nullptr)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
    , m_pHandheldManager(nullptr)
    , m_State()
    , m_IsAutoPilotEnabled(false)
    , m_AutoPilotState()
    , m_GestureRecognizer()
    , m_Temp()
{
    for (TouchScreenAppletResourceEntry& extraEntry : m_AppletResourceEntries)
    {
        extraEntry = TouchScreenAppletResourceEntry();
    }
}

void TouchScreenManager::SetTimerEvent(
    ::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_GestureActivationCount.IsZero());
    NN_SDK_REQUIRES(m_TouchScreenActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

void TouchScreenManager::SetDriver(ITouchScreenDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_GestureActivationCount.IsZero());
    NN_SDK_REQUIRES(m_TouchScreenActivationCount.IsZero());
    m_pDriver = pDriver;
}

void TouchScreenManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_GestureActivationCount.IsZero());
    NN_SDK_REQUIRES(m_TouchScreenActivationCount.IsZero());
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

void TouchScreenManager::SetHandheldManager(
    HandheldManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_GestureActivationCount.IsZero());
    NN_SDK_REQUIRES(m_TouchScreenActivationCount.IsZero());
    m_pHandheldManager = pManager;
}

void TouchScreenManager::SetInputDetectorManager(
    InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    NN_SDK_REQUIRES(m_CommonActivationCount.IsZero());
    NN_SDK_REQUIRES(m_GestureActivationCount.IsZero());
    NN_SDK_REQUIRES(m_TouchScreenActivationCount.IsZero());
    m_pInputDetectorManager = pManager;
    m_pInputDetectorManagerMutex = pMutex;
}

::nn::Result TouchScreenManager::ActivateForGesture() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsMax(),
                           ResultGestureActivationUpperLimitOver());
    NN_RESULT_THROW_UNLESS(!m_GestureActivationCount.IsMax(),
                           ResultGestureActivationUpperLimitOver());

    if (m_GestureActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // LIFO を初期化
        this->ProcessSharedMemory([] (
            TouchScreenManager* that,
            GestureSharedMemoryFormat* gestureAddress,
            GestureContext* pGestureContext,
            TouchScreenSharedMemoryFormat* touchScreenAddress,
            TouchScreenContext* pTouchScreenContext,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isGestureActivated,
            bool isTouchScreenActivated) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(gestureAddress);
            NN_UNUSED(that);
            NN_UNUSED(touchScreenAddress);
            NN_UNUSED(pTouchScreenContext);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(isGestureActivated);
            NN_UNUSED(isTouchScreenActivated);
            *pGestureContext = GestureContext();
            ::nn::util::Get(gestureAddress->lifo).Clear();
        });

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->ActivateCommon());
        }
    }

    ++m_GestureActivationCount;

    ++m_CommonActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::DeactivateForGesture() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_GestureActivationCount.IsZero(),
                           ResultGestureDeactivationLowerLimitOver());
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultGestureDeactivationLowerLimitOver());

    --m_CommonActivationCount;

    --m_GestureActivationCount;

    if (m_GestureActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->DeactivateCommon());
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::EnsureAppletResourceForGesture(
    ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT
{
    m_Temp.aruid = aruid;
    m_Temp.version = version;

    this->ProcessSharedMemory([] (
        TouchScreenManager* that,
        GestureSharedMemoryFormat* gestureAddress,
        GestureContext* pGestureContext,
        TouchScreenSharedMemoryFormat* touchScreenAddress,
        TouchScreenContext* pTouchScreenContext,
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput,
        bool isGestureActivated,
        bool isTouchScreenActivated) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(gestureAddress);
        NN_SDK_REQUIRES_NOT_NULL(pGestureContext);
        NN_UNUSED(touchScreenAddress);
        NN_UNUSED(pTouchScreenContext);
        NN_UNUSED(enablesInput);
        NN_UNUSED(isGestureActivated);
        NN_UNUSED(isTouchScreenActivated);

        if (that->m_Temp.aruid == aruid)
        {
            const int32_t version = that->m_Temp.version;

            GestureSharedMemoryFormat* address = gestureAddress;

            GestureLifo& lifo = ::nn::util::Get(address->lifo);

            GestureContext& context = *pGestureContext;

            if (!context.flags.Test<GestureContextFlag::IsAvailable>() ||
                 context.version != version)
            {
                lifo.Clear();
            }

            if (lifo.IsEmpty())
            {
                InitializeGestureAppletResource(
                    address, &context, &that->m_Temp.gestureState, version);
            }
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::ActivateForTouchScreen() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsMax(),
                           ResultTouchScreenActivationUpperLimitOver());
    NN_RESULT_THROW_UNLESS(!m_TouchScreenActivationCount.IsMax(),
                           ResultTouchScreenActivationUpperLimitOver());

    if (m_TouchScreenActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // LIFO を初期化
        this->ProcessSharedMemory([] (
            TouchScreenManager* that,
            GestureSharedMemoryFormat* gestureAddress,
            GestureContext* pGestureContext,
            TouchScreenSharedMemoryFormat* touchScreenAddress,
            TouchScreenContext* pTouchScreenContext,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isGestureActivated,
            bool isTouchScreenActivated) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(touchScreenAddress);
            NN_UNUSED(that);
            NN_UNUSED(gestureAddress);
            NN_UNUSED(pGestureContext);
            NN_UNUSED(aruid);
            NN_UNUSED(enablesInput);
            NN_UNUSED(isGestureActivated);
            NN_UNUSED(isTouchScreenActivated);
            *pTouchScreenContext = TouchScreenContext();
            ::nn::util::Get(touchScreenAddress->lifo).Clear();
        });

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->ActivateCommon());
        }
    }

    ++m_TouchScreenActivationCount;

    ++m_CommonActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::DeactivateForTouchScreen() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_TouchScreenActivationCount.IsZero(),
                           ResultTouchScreenDeactivationLowerLimitOver());
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultTouchScreenDeactivationLowerLimitOver());

    --m_CommonActivationCount;

    --m_TouchScreenActivationCount;

    if (m_TouchScreenActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        if (m_CommonActivationCount.IsZero())
        {
            NN_RESULT_DO(this->DeactivateCommon());
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::EnsureAppletResourceForTouchScreen(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_Temp.aruid = aruid;

    this->ProcessSharedMemory([] (
        TouchScreenManager* that,
        GestureSharedMemoryFormat* gestureAddress,
        GestureContext* pGestureContext,
        TouchScreenSharedMemoryFormat* touchScreenAddress,
        TouchScreenContext* pTouchScreenContext,
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput,
        bool isGestureActivated,
        bool isTouchScreenActivated) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(touchScreenAddress);
        NN_SDK_REQUIRES_NOT_NULL(pTouchScreenContext);
        NN_UNUSED(gestureAddress);
        NN_UNUSED(pGestureContext);
        NN_UNUSED(isGestureActivated);
        NN_UNUSED(isTouchScreenActivated);

        if (that->m_Temp.aruid == aruid)
        {
            TouchScreenLifo& lifo = ::nn::util::Get(touchScreenAddress->lifo);

            typedef TouchScreenState<TouchStateCountMax> StateType;

            StateType* pOutValue = &that->m_Temp.touchScreenState;

            const StateType& state = that->m_State;

            if (lifo.IsEmpty())
            {
                CalculateTouchScreenState(
                    pOutValue, pTouchScreenContext, state, enablesInput);

                lifo.Append(*pOutValue);
            }
        }
    });

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::SetAutoPilotState(
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        >& value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultTouchScreenNotInitialized());

    if (!m_IsAutoPilotEnabled)
    {
        m_IsAutoPilotEnabled = true;

        m_AutoPilotState =
            ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>();
    }

    UpdateTouchScreenAutoPilotState(&m_AutoPilotState, value);

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::UnsetAutoPilotState() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultTouchScreenNotInitialized());
    this->DisableAutoPilot();
    NN_RESULT_SUCCESS;
}

void TouchScreenManager::Sample() NN_NOEXCEPT
{
    if (!m_CommonActivationCount.IsZero())
    {
        this->Update();

        m_GestureRecognizer.Update(m_State.touches, m_State.count);

        this->ProcessSharedMemory([] (
            TouchScreenManager* that,
            GestureSharedMemoryFormat* gestureAddress,
            GestureContext* pGestureContext,
            TouchScreenSharedMemoryFormat* touchScreenAddress,
            TouchScreenContext* pTouchScreenContext,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isGestureActivated,
            bool isTouchScreenActivated) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(that);
            NN_SDK_REQUIRES_NOT_NULL(gestureAddress);
            NN_SDK_REQUIRES_NOT_NULL(pGestureContext);
            NN_SDK_REQUIRES_NOT_NULL(touchScreenAddress);
            NN_SDK_REQUIRES_NOT_NULL(pTouchScreenContext);
            NN_UNUSED(aruid);

            if (isGestureActivated)
            {
                UpdateGestureAppletResource(
                    gestureAddress,
                    pGestureContext,
                    &that->m_Temp.gestureState,
                    that->m_GestureRecognizer.GetState(),
                    enablesInput);
            }

            if (isTouchScreenActivated)
            {
                TouchScreenSharedMemoryFormat* address = touchScreenAddress;

                TouchScreenLifo& lifo = ::nn::util::Get(address->lifo);

                typedef TouchScreenState<TouchStateCountMax> StateType;

                StateType* pOutValue = &that->m_Temp.touchScreenState;

                const StateType& state = that->m_State;

                CalculateTouchScreenState(
                    pOutValue, pTouchScreenContext, state, enablesInput);

                lifo.Append(*pOutValue);
            }
        });
    }
}

::nn::Result TouchScreenManager::WakeDeviceUp() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pHandheldManager);

    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultTouchScreenNotInitialized());

    if (m_pHandheldManager->IsHandheldEnabled())
    {
        NN_RESULT_DO(m_pDriver->Wake());
    }

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::PutDeviceToSleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_CommonActivationCount.IsZero(),
                           ResultTouchScreenNotInitialized());

    NN_RESULT_DO(m_pDriver->Sleep());

    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

void TouchScreenManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pHandheldManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);

    // 入力変化検知用に状態を保存
    m_Temp.touchScreenState = m_State;

    if (m_IsAwake && m_pHandheldManager->IsHandheldEnabled())
    {
        // 起床中かつドックアウト状態ならばデバイスも起床
        m_pDriver->Wake();
    }
    else
    {
        // それ以外の状態ではデバイスはスリープ
        m_pDriver->Sleep();
    }

    // 入力状態を取得
    m_pDriver->GetState(&m_State);

    // サンプリング番号を設定
    m_State.samplingNumber = m_SamplingNumber++;

    for (int i = 0; i < m_State.count; ++i)
    {
        m_State.touches[i] = LimitTouch(m_State.touches[i]);
    }

    if (m_IsAutoPilotEnabled)
    {
        // 自動操作が有効な場合は自動操作状態をマージ
        if (m_State.count == 0)
        {
            const int32_t count = m_AutoPilotState.count;

            m_State.count = count;

            for (int i = 0; i < count; ++i)
            {
                m_State.touches[i] = m_AutoPilotState.touches[i];
            }

            FlushTouchScreenAutoPilotState(&m_AutoPilotState);
        }
    }

    ::std::lock_guard<decltype(*m_pInputDetectorManagerMutex)
                      > locker(*m_pInputDetectorManagerMutex);

    if (IsTouchScreenStateChanged(m_State, m_Temp.touchScreenState))
    {
        m_pInputDetectorManager->Notify(
            ::nn::hid::system::InputSourceId::TouchScreen::Mask);
    }
}

void TouchScreenManager::ProcessSharedMemory(
    void (*processor)(
        TouchScreenManager* that,
        GestureSharedMemoryFormat* gestureAddress,
        GestureContext* pGestureContext,
        TouchScreenSharedMemoryFormat* touchScreenAddress,
        TouchScreenContext* pTouchScreenContext,
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput,
        bool isGestureActivated,
        bool isTouchScreenActivated) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    const bool isGestureActivated = !m_GestureActivationCount.IsZero();

    const bool isTouchScreenActivated = !m_TouchScreenActivationCount.IsZero();

    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)
                      > locker(*m_pAppletResourceManagerMutex);

    const AppletResourceEntry (&entries)[AppletResourceEntryCountMax] =
        m_pAppletResourceManager->GetAppletResourceEntries();

    for (size_t i = 0; i < AppletResourceEntryCountMax; ++i)
    {
        const AppletResourceEntry& entry = entries[i];

        TouchScreenAppletResourceEntry& extraEntry = m_AppletResourceEntries[i];

        if (!entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            extraEntry = TouchScreenAppletResourceEntry();
        }
        else
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);

            if (entry.aruid != extraEntry.aruid)
            {
                extraEntry = TouchScreenAppletResourceEntry();

                extraEntry.aruid = entry.aruid;
            }

            processor(this,
                      &entry.address->gesture,
                      &extraEntry.gestureContext,
                      &entry.address->touchScreen,
                      &extraEntry.touchScreenContext,
                      entry.aruid,
                      entry.flags.Test<AppletResourceFlag::EnablesInput>(),
                      isGestureActivated,
                      isTouchScreenActivated);
        }
    }
}

void TouchScreenManager::DisableAutoPilot() NN_NOEXCEPT
{
    m_IsAutoPilotEnabled = false;

    m_AutoPilotState =
        ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>();
}

::nn::Result TouchScreenManager::ActivateCommon() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    // ドライバをアクティブ化
    NN_RESULT_DO(m_pDriver->Activate());

    // 起床状態に設定
    m_IsAwake = true;

    // タイマーイベントをアクティブ化
    ::nn::os::StartPeriodicTimerEvent(
        m_pTimerEvent, SamplingInterval, SamplingInterval);

    m_State = TouchScreenState<TouchStateCountMax>();

    // 入力状態を更新
    this->Update();

    // ジェスチャの状態を更新
    m_GestureRecognizer.Update(m_State.touches, m_State.count);

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenManager::DeactivateCommon() NN_NOEXCEPT
{
    // ジェスチャ認識器をリセット
    m_GestureRecognizer.Reset();

    // 自動操作を無効化
    this->DisableAutoPilot();

    // タイマーイベントを非アクティブ化
    ::nn::os::StopTimerEvent(m_pTimerEvent);

    // ドライバを非アクティブ化
    NN_RESULT_DO(m_pDriver->Deactivate());

    // 起床状態を解除
    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

namespace {

void InitializeGestureAppletResource(
    GestureSharedMemoryFormat* address,
    GestureContext* pContext,
    GestureState* pState,
    int32_t version) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES_NOT_NULL(pContext);
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureContext& context = *pContext;
    context.flags.Set<GestureContextFlag::IsAvailable>();
    context.flags.Set<GestureContextFlag::IsLocked>();
    context.version = version;
    context.eventNumber = 0;
    context.contextNumber = 0;
    context.lastEventNumber = 0;
    context.lastContextNumber = 0;
    context.lastGestureType = static_cast<int32_t>(GestureType_Idle);

    auto& state = *new(pState) GestureState();
    state.eventNumber = context.eventNumber;
    state.contextNumber = context.contextNumber;
    state._type = context.lastGestureType;

    ::nn::util::Get(address->lifo).Append(state);
}

void UpdateGestureAppletResource(
    GestureSharedMemoryFormat* address,
    GestureContext* pContext,
    GestureState* pState,
    const GestureState& masterState,
    bool enablesInput) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);
    NN_SDK_REQUIRES_NOT_NULL(pContext);
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureContext& context = *pContext;

    if (!context.flags.Test<GestureContextFlag::IsAvailable>())
    {
        return;
    }

    if (context.flags.Test<GestureContextFlag::IsLocked>())
    {
        if (enablesInput && masterState.GetGestureType() == GestureType_Idle)
        {
            context.flags.Reset<GestureContextFlag::IsLocked>();
            context.lastEventNumber = masterState.eventNumber;
            context.lastContextNumber = masterState.contextNumber;
        }
    }
    else
    {
        GestureLifo& lifo = ::nn::util::Get(address->lifo);

        if (enablesInput)
        {
            if (context.lastEventNumber != masterState.eventNumber)
            {
                context.lastEventNumber = masterState.eventNumber;
                ++context.eventNumber;

                if (context.lastContextNumber != masterState.contextNumber)
                {
                    context.lastContextNumber = masterState.contextNumber;
                    ++context.contextNumber;
                }

                context.lastGestureType =
                    static_cast<int32_t>(masterState.GetGestureType());

                GestureState& state = *pState;
                state = masterState;
                state.eventNumber = context.eventNumber;
                state.contextNumber = context.contextNumber;
                lifo.Append(state);
            }
        }
        else
        {
            auto& state = *new(pState) GestureState();

            const auto gestureType =
                static_cast<GestureType>(context.lastGestureType);

            switch (gestureType)
            {
            case GestureType_Touch:
            case GestureType_Press:
            case GestureType_Pan:
            case GestureType_Pinch:
            case GestureType_Rotate:
                state.eventNumber = ++context.eventNumber;
                state.contextNumber = context.contextNumber;
                state._type = static_cast<int32_t>(GestureType_Cancel);
                lifo.Append(state);
                break;
            case GestureType_Tap:
            case GestureType_Swipe:
                state.eventNumber = ++context.eventNumber;
                state.contextNumber = context.contextNumber;
                state._type = static_cast<int32_t>(GestureType_Complete);
                lifo.Append(state);
                break;
            default:
                break;
            }

            if (gestureType != GestureType_Idle)
            {
                state.eventNumber = ++context.eventNumber;
                state.contextNumber = ++context.contextNumber;
                state._type = static_cast<int32_t>(GestureType_Idle);
                lifo.Append(state);
            }

            context.flags.Set<GestureContextFlag::IsLocked>();
            context.lastGestureType = static_cast<int32_t>(GestureType_Idle);
        }
    }
}

template<typename T>
bool FindsTouch(
    int32_t fingerId, const T* head, const T* tail,
    int32_t(*getter)(const T& value) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(head);
    NN_SDK_REQUIRES_NOT_NULL(tail);
    NN_SDK_REQUIRES_NOT_NULL(getter);

    while (head != tail)
    {
        if (getter(*head) == fingerId)
        {
            return true;
        }

        ++head;
    }

    return false;
}

void CalculateTouchScreenState(
    TouchScreenState<TouchStateCountMax>* pOutValue,
    TouchScreenContext* pTouchScreenContext,
    const TouchScreenState<TouchStateCountMax>& state,
    bool enablesInput) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pTouchScreenContext);

    TouchScreenContext& context = *pTouchScreenContext;

    if (enablesInput)
    {
        // HID 入力の提供が行われているならばマスク対象の削除処理を実施

        int32_t count = 0;

        for (int32_t i = 0; i < context.count; ++i)
        {
            for (int32_t j = 0; j < state.count; ++j)
            {
                if (context.ids[i] == state.touches[j].fingerId)
                {
                    // アプレット遷移前から存在するタッチはマスクを継続

                    context.ids[count] = context.ids[i];

                    ++count;

                    break;
                }
            }
        }

        context.count = count;
    }
    else
    {
        // HID 入力の提供が行われているならばマスク対象の追加処理を実施

        // アプレット遷移前から存在するタッチは遷移後にマスクする必要あり
        for (int32_t i = 0; i < state.count; ++i)
        {
            context.ids[i] = state.touches[i].fingerId;
        }

        context.count = state.count;
    }

    // 余った領域はゼロクリア
    for (int32_t i = context.count; i < TouchStateCountMax; ++i)
    {
        context.ids[i] = int32_t();
    }

    {
        int32_t count = 0;

        for (int32_t i = 0; i < state.count; ++i)
        {
            if (!FindsTouch<int32_t>(state.touches[i].fingerId,
                                     context.ids, context.ids + context.count,
                                     [] (const int32_t& value) NN_NOEXCEPT
                                     {
                                         return value;
                                     }))
            {
                // マスクされていなければタッチを提供

                pOutValue->touches[count] = state.touches[i];

                ++count;
            }
        }

        // 余った領域はゼロクリア
        for (int32_t i = count; i < TouchStateCountMax; ++i)
        {
            pOutValue->touches[i] = TouchState();
        }

        pOutValue->samplingNumber = state.samplingNumber;
        pOutValue->count = count;
    }
}

} // namespace

}}} // namespace nn::hid::detail
