﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_TickApi.h>

#include "hid_SixAxisSensorFusionPolicy.h"

namespace
{

bool IsStill(float stillness)
{
    return (stillness > 0.2f);
}

} // namespace

namespace nn { namespace hid { namespace detail {

SixAxisSensorFusionPolicy::SixAxisSensorFusionPolicy() NN_NOEXCEPT
    : m_SensorFusionMode(SensorFusionMode_Idle)
    , m_Stillness(0.f)
    , m_VerticalizationStrength(1.f)
    , m_StillStartedTick()
{
    // Do nothing
}

void SixAxisSensorFusionPolicy::Reset() NN_NOEXCEPT
{
    m_SensorFusionMode = SensorFusionMode_Idle;
    m_Stillness = 0.f;
    m_VerticalizationStrength = 1.f;
    m_StillStartedTick = ::nn::os::GetSystemTick();
}

void SixAxisSensorFusionPolicy::Update(const SixAxisSensorFusionParameters& parameters) NN_NOEXCEPT
{
    const auto IsStillCurrently = IsStill(parameters.stillness);

    switch (m_SensorFusionMode)
    {
    case nn::hid::detail::SixAxisSensorFusionPolicy::SensorFusionMode_Idle:
        if (IsStillCurrently)
        {
            // Stillness is changed from false to true
            if (IsStill(m_Stillness) == false)
            {
                m_StillStartedTick = ::nn::os::GetSystemTick();
            }

            // Set to Still mode since still state was kept for a while
            if ((::nn::os::GetSystemTick() - m_StillStartedTick).ToTimeSpan().GetMilliSeconds() > 3000)
            {
                m_SensorFusionMode = SensorFusionMode_Still;
                m_VerticalizationStrength = 10.f;
            }
        }
        break;
    case nn::hid::detail::SixAxisSensorFusionPolicy::SensorFusionMode_Still:
        if (IsStillCurrently == false)
        {
            // Set to Idle state
            m_SensorFusionMode = SensorFusionMode_Idle;
            m_VerticalizationStrength = 1.f;
        }
        break;
    default:
        break;
    }

    m_Stillness = parameters.stillness;
}

}}} // namespace nn::hid::detail
