﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>

#include "hid_NpadId.h"
#include "hid_SixAxisSensor.h" // 公開ヘッダ更新後、削除予定
#include "hid_SixAxisSensorAppletSetting.h"

namespace nn { namespace hid { namespace detail {

//!< 加速度センサーの設定値
struct AccelerometerSetting
{
    nn::xcd::AccelerometerFsr fsr;          //!< フルスケールレンジ

    AccelerometerPlayMode playMode;         //!< 遊びの挙動方法
    float playRadius;                       //!< 遊び半径
    float sensitivity;                      //!< 感度

    bool  isReviseEnabled;                  //!< 加速度による姿勢補正
    float revisePower;                      //!< 姿勢値の加速度補正の重み。0 から 1 の間の値が設定されます。値が大きいほど急激に補正がかかる
    float reviseRange;                      //!< 姿勢値の加速度補正の有効範囲。重力加速度を中心にこの範囲内の加速度センサの値を補正計算に使用。

    float shiftOrigin;                      //!< ゼロ点のずらし幅[G]
    float shiftSensitivity;                 //!< 感度のずらし率[%]

    // リセット
    void Reset(bool global) NN_NOEXCEPT;

    // FSR 設定
    void SetAccelerometerFsr(const nn::xcd::AccelerometerFsr& accFsr) NN_NOEXCEPT;

    // センサフュージョン
    bool IsSixAxisSensorFusionEnabled() const NN_NOEXCEPT;

    void EnableSixAxisSensorFusion(bool enable) NN_NOEXCEPT;

    void SetSixAxisSensorFusionParameters(float revisePower,
                                          float reviseRange) NN_NOEXCEPT;

    void GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                          float* pOutReviseRange) const NN_NOEXCEPT;

    void ResetAccelerationReviseParameters() NN_NOEXCEPT;

    // 加速度センサーの遊び
    void SetAccelerometerParameters(float playRadius,
                                    float sensitivity) NN_NOEXCEPT;

    void GetAccelerometerParameters(float* pOutPlayRadius,
                                    float* pOutSensitivity) const NN_NOEXCEPT;

    void ResetPlayModeParameters() NN_NOEXCEPT;

    void SetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode mode) NN_NOEXCEPT;

    void GetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode* pOutMode) const NN_NOEXCEPT;

    void ResetPlayMode() NN_NOEXCEPT;

    // デバッグ設定
    void SetShiftCalibrationValue(float setOrigin, float setSensitivity) NN_NOEXCEPT;

    void GetShiftCalibrationValue(float* pOutOrigin, float* pOutSensitivity) const NN_NOEXCEPT;

    void ResetShiftCalibrationValue() NN_NOEXCEPT;
};

//!< ジャイロセンサーの設定値
struct GyroscopeSetting
{
    nn::xcd::GyroscopeFsr fsr;               //!< フルスケールレンジ

    bool enableZeroDrift;                    //!< ゼロ点自動補正するか
    bool enableZeroPlay;                     //!< ゼロ点に遊びを設けるか
    float zeroPlayRadius;                    //!< ゼロ点の遊び半径
    GyroscopeZeroDriftMode zeroDriftMode;    //!< ゼロ点ドリフト設定

    nn::util::Float3 magnification;          //!< 角速度を計算する際に掛ける倍率
    nn::util::Float3 directionMagnification; //!< 姿勢値を計算する際に掛ける倍率

    float shiftOrigin;                       //!< ゼロ点のずらし幅[1.0f = 360 dps]
    float shiftSensitivity;                  //!< 感度のずらし率[%]

    // リセット
    void Reset(bool global) NN_NOEXCEPT;

    // FSR 設定
    void SetGyroscopeFsr(const nn::xcd::GyroscopeFsr& gyroFsr) NN_NOEXCEPT;

    // ジャイロセンサーのゼロ点補正
    void EnableZeroDrift(bool enable) NN_NOEXCEPT;

    bool IsZeroDriftEnabled() const NN_NOEXCEPT;

    void SetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode mode) NN_NOEXCEPT;

    void GetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode* pOutMode) const NN_NOEXCEPT;

    void ResetZeroDriftMode() NN_NOEXCEPT;

    // ジャイロセンサーのゼロ点遊び
    void EnableZeroPlay(bool enable) NN_NOEXCEPT;

    bool IsZeroPlayEnabled() const NN_NOEXCEPT;

    void SetZeroPlayParameters(const float& radius) NN_NOEXCEPT;

    void GetZeroPlayParameters(float* pOutRadius) const NN_NOEXCEPT;

    void ResetZeroPlayParameters() NN_NOEXCEPT;

    // 倍率設定
    void SetMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT;

    void SetDirectionMagnification(const nn::util::Float3& magnification) NN_NOEXCEPT;

    // デバッグ設定
    void SetShiftCalibrationValue(float origin, float sensitivity) NN_NOEXCEPT;

    void GetShiftCalibrationValue(float* pOutOrigin, float* pOutSensitivity) const NN_NOEXCEPT;

    void ResetShiftCalibrationValue() NN_NOEXCEPT;
};

//!< 6軸センサーの加工処理に関する設定値
struct SixAxisSensorProcessorSetting
{
    bool isSamplingEnabled;                 //!< サンプリングが有効かどうか

    bool isBaseDirReviseEnabled;            //!< 基本姿勢による姿勢補正
    nn::hid::DirectionState baseDirection;  //!< 基本姿勢
    float baseDirectionRevisePower;         //!< 基本姿勢に近づける強さ

    void Reset(bool global) NN_NOEXCEPT;

    // サンプリングに関する設定
    void EnableSampling(bool enable) NN_NOEXCEPT;

    bool IsSamplingEnabled() const NN_NOEXCEPT;

    // 基本姿勢による姿勢補正
    void EnableBaseDirection(bool enable) NN_NOEXCEPT;

    bool IsBaseDirectionEnabled() const NN_NOEXCEPT;

    void SetBaseDirection(const nn::hid::DirectionState& direction,
                          const float& power) NN_NOEXCEPT;

    void GetBaseDirection(nn::hid::DirectionState* pOutDirection,
                          float* pOutPower) const NN_NOEXCEPT;

    void ResetBaseDirection() NN_NOEXCEPT;
};

//!< 6軸センサーの設定値
struct SixAxisSensorSetting
{
    AccelerometerSetting accelerometerSetting;
    GyroscopeSetting gyroscopeSetting;
    SixAxisSensorProcessorSetting sixAxisSensorProcessorSetting;

    void Reset(bool global) NN_NOEXCEPT;
};

//!< アプリごとに持つ 6軸センサーの操作形態ごとの設定値
struct SixAxisSensorAppletSetting
{
    SixAxisSensorSetting fullKey[NpadIdCountMax];
    SixAxisSensorSetting handheld[NpadIdCountMax];
    SixAxisSensorSetting joyDualLeft[NpadIdCountMax];
    SixAxisSensorSetting joyDualRight[NpadIdCountMax];
    SixAxisSensorSetting joyLeft[NpadIdCountMax];
    SixAxisSensorSetting joyRight[NpadIdCountMax];
    SixAxisSensorSetting dummyStyle[NpadIdCountMax];

    void Reset(bool global) NN_NOEXCEPT;

    SixAxisSensorSetting* GetSixAxisSensorSetting(const SixAxisSensorHandle& handle) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
