﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_ActivationCount.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SharedMemoryHolder.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< アプレットリソースを保持するためのクラスです。
class AppletResourceHolder final
{
    NN_DISALLOW_COPY(AppletResourceHolder);
    NN_DISALLOW_MOVE(AppletResourceHolder);

private:
    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< ARUID が有効か否か
    bool m_IsAruidValid;

    //!< ARUID
    ::nn::applet::AppletResourceUserId m_Aruid;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< アプレットリソースのセッション
    ::nn::sf::SharedPointer<IAppletResource> m_AppletResource;

    //!< 共有メモリホルダ
    SharedMemoryHolder m_SharedMemoryHolder;

    //!< 共有メモリのアドレス
    ::std::atomic<SharedMemoryFormat*> m_SharedMemoryAddress;

public:
    AppletResourceHolder() NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を設定します。
    ::nn::Result SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< アプレットリソースを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 共有メモリのアドレスを返します。
    SharedMemoryFormat* GetSharedMemoryAddress() NN_NOEXCEPT;
};

//!< アプレットリソースホルダを返します。
AppletResourceHolder& GetAppletResourceHolder() NN_NOEXCEPT;

} // namespace

SharedMemoryAccessor::SharedMemoryAccessor() NN_NOEXCEPT
    : m_ActivationCount()
    , m_Mutex(false)
    , m_ResultActivationUpperLimitOver()
    , m_ResultDeactivationLowerLimitOver()
{
    // 何もしない
}

SharedMemoryAccessor::~SharedMemoryAccessor() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result SharedMemoryAccessor::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(GetAppletResourceHolder().SetAppletResourceUserId(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedMemoryAccessor::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ResultActivationUpperLimitOver.IsFailure());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsMax(), m_ResultActivationUpperLimitOver);

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        NN_RESULT_DO(GetAppletResourceHolder().Activate());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                GetAppletResourceHolder().Deactivate();
            }
        };

        NN_RESULT_DO(
            this->Attach(GetAppletResourceHolder().GetSharedMemoryAddress()));

        needsRollback = false;
    }

    // アクティブ化された回数を増加
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedMemoryAccessor::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_ResultDeactivationLowerLimitOver.IsFailure());

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsZero(), m_ResultDeactivationLowerLimitOver);

    // アクティブ化された回数を減少
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        NN_RESULT_DO(this->Detach());

        NN_RESULT_DO(GetAppletResourceHolder().Deactivate());
    }

    NN_RESULT_SUCCESS;
}

void SharedMemoryAccessor::SetResultActivationUpperLimitOver(
    ::nn::Result value) NN_NOEXCEPT
{
    m_ResultActivationUpperLimitOver = value;
}

void SharedMemoryAccessor::SetResultDeactivationLowerLimitOver(
    ::nn::Result value) NN_NOEXCEPT
{
    m_ResultDeactivationLowerLimitOver = value;
}

namespace {

AppletResourceHolder::AppletResourceHolder() NN_NOEXCEPT
    : m_Mutex(false)
    , m_IsAruidValid(false)
    , m_Aruid()
    , m_ActivationCount()
    , m_AppletResource()
    , m_SharedMemoryHolder()
    , m_SharedMemoryAddress(nullptr)
{
    // 何もしない
}

::nn::Result AppletResourceHolder::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    if (!m_IsAruidValid)
    {
        m_Aruid = aruid;

        m_IsAruidValid = true;
    }
    else
    {
        NN_RESULT_THROW_UNLESS(aruid == m_Aruid,
                               ResultAppletResourceInvalidUserId());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceHolder::Activate() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_SDK_REQUIRES(m_IsAruidValid);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultAppletResourceActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        // アプレットリソースのセッションを作成
        NN_RESULT_DO(
            pHidServer->CreateAppletResource(&m_AppletResource, m_Aruid));

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_AppletResource.Reset();
            }
        };

        // 共有メモリハンドルを取得
        ::nn::sf::NativeHandle handle;
        NN_RESULT_DO(m_AppletResource->GetSharedMemoryHandle(&handle));

        // 共有メモリをアタッチ
        m_SharedMemoryHolder.Attach(handle.GetOsHandle());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_SharedMemoryHolder.Finalize();
            }
        };

        // 共有メモリハンドルの管理権を放棄
        handle.Detach();

        NN_RESULT_THROW_UNLESS(
            m_SharedMemoryHolder.GetAddress() != nullptr,
            ResultAppletResourceSharedMemoryAllocationFailed());

        m_SharedMemoryAddress = m_SharedMemoryHolder.GetAddress();

        needsRollback = false;
    }

    // アクティブ化された回数を増加
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceHolder::Deactivate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultAppletResourceDeactivationLowerLimitOver());

    // アクティブ化された回数を減少
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        m_SharedMemoryAddress = nullptr;

        // 共有メモリを開放
        m_SharedMemoryHolder.Finalize();

        // アプレットリソースのセッションを破棄
        m_AppletResource.Reset();
    }

    NN_RESULT_SUCCESS;
}

SharedMemoryFormat* AppletResourceHolder::GetSharedMemoryAddress() NN_NOEXCEPT
{
    return m_SharedMemoryAddress;
}

AppletResourceHolder& GetAppletResourceHolder() NN_NOEXCEPT
{
    return StaticObject<AppletResourceHolder>::Get();
}

} // namespace

}}} // namespace nn::hid::detail
