﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_ActivationCount.h"

namespace nn { namespace hid { namespace detail {

//!< セッションを扱うためのクラスです。
class Session
{
    NN_DISALLOW_COPY(Session);
    NN_DISALLOW_MOVE(Session);

private:
    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< ミューテックス
    ::nn::os::Mutex m_Mutex;

public:
    Session() NN_NOEXCEPT;

    virtual ~Session() NN_NOEXCEPT;

    //!< セッションをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< セッションを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

protected:
    //!< アクティブ化された回数が上限を超えたことを表す Result 値を返します。
    virtual ::nn::Result ThrowResultActivationUpperLimitOver(
        ) NN_NOEXCEPT = 0;

    //!< アクティブ化された回数が下限を超えたことを表す Result 値を返します。
    virtual ::nn::Result ThrowResultDeactivationLowerLimitOver(
        ) NN_NOEXCEPT = 0;

    //!< 共有メモリのアロケーションに失敗したことを表す Result 値を返します。
    virtual ::nn::Result ThrowResultAllocationFailed(
        ) NN_NOEXCEPT = 0;

    //!< サービスへのプロクシを作成します。
    virtual ::nn::Result CreateProxy() NN_NOEXCEPT = 0;

    //!< サービスへのプロクシを破棄します。
    virtual void DestroyProxy() NN_NOEXCEPT = 0;

    //!< サービスをアクティブ化します。
    virtual ::nn::Result ActivateService() NN_NOEXCEPT = 0;

    //!< サービスを非アクティブ化します。
    virtual ::nn::Result DeactivateService() NN_NOEXCEPT = 0;

    //!< 共有メモリハンドルを取得します。
    virtual ::nn::Result GetSharedMemoryHandle(::nn::sf::NativeHandle* outValue
                                               ) NN_NOEXCEPT = 0;

    //!< 共有メモリがマップ状態にあるか否かを表す値を返します。
    virtual bool IsSharedMemoryMapped() NN_NOEXCEPT = 0;

    //!< 共有メモリをアタッチします。
    virtual void AttachSharedMemory(::nn::os::NativeHandle handle,
                                    bool managed) NN_NOEXCEPT = 0;

    //!< 共有メモリを開放します。
    virtual void FinalizeSharedMemory() NN_NOEXCEPT = 0;
};

}}} // namespace nn::hid::detail
