﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_Session.h"

namespace nn { namespace hid { namespace detail {

Session::Session() NN_NOEXCEPT
    : m_Mutex(false)
{
    // 何もしない
}

Session::~Session() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result Session::Activate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           this->ThrowResultActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        // サービスへのプロクシを作成
        NN_RESULT_DO(this->CreateProxy());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                this->DestroyProxy();
            }
        };

        // サービスをアクティブ化
        NN_RESULT_DO(this->ActivateService());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                this->DeactivateService();
            }
        };

        // 共有メモリハンドルを取得
        auto handle = ::nn::sf::NativeHandle();
        NN_RESULT_DO(this->GetSharedMemoryHandle(&handle));

        // 共有メモリをアタッチ
        this->AttachSharedMemory(handle.GetOsHandle(), handle.IsManaged());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                this->FinalizeSharedMemory();
            }
        };

        // ハンドルの管理権を放棄
        handle.Detach();

        NN_RESULT_THROW_UNLESS(this->IsSharedMemoryMapped(),
                               this->ThrowResultAllocationFailed());

        needsRollback = false;
    }

    // アクティブ化された回数を増加
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result Session::Deactivate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           this->ThrowResultDeactivationLowerLimitOver());

    // アクティブ化された回数を減少
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 共有メモリを開放
        this->FinalizeSharedMemory();

        // サービスを非アクティブ化
        NN_RESULT_DO(this->DeactivateService());

        // サービスへのプロクシを破棄
        this->DestroyProxy();
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
