﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace hid { namespace detail {

//!< 非スレッドセーフに リング LIFO を扱うためのクラスです。
template<typename LifoT,
         int LifoEntryCount>
class RingLifoLight
{
    NN_STATIC_ASSERT(LifoEntryCount >= 1);
    NN_DISALLOW_COPY(RingLifoLight);
    NN_DISALLOW_MOVE(RingLifoLight);

protected:
    //!< リングバッファ上の末尾位置
    int m_Tail;

    //!< リングバッファ上の要素数
    int m_Count;

    //!< リングバッファ
    LifoT m_Buffer[LifoEntryCount + 1];

public:
    RingLifoLight() NN_NOEXCEPT
        : m_Tail(0)
        , m_Count(0)
    {
        // 何もしない
    }

    virtual ~RingLifoLight() NN_NOEXCEPT { /* 何もしない */ }

public:
    //!< リング LIFO 上の要素の型
    typedef LifoT ValueType;

public:
    //!< LIFO を空にします。
    void Clear() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    //!< LIFO に要素を追加します。
    void Append(const LifoT& value) NN_NOEXCEPT
    {
        this->AppendImpl(value);
    }

    //!< LIFO が空か否か判定します。任意のタイミングでスレッドセーフです。
    bool IsEmpty() const NN_NOEXCEPT
    {
        return (m_Count == 0);
    }

    //!< LIFO から要素を読み出します。任意のタイミングでスレッドセーフです。
    int Read(LifoT* buffer, int bufferCount) const NN_NOEXCEPT
    {
        return this->ReadImpl(buffer, bufferCount);
    }

protected:
    //!< 基準点からの相対位置を絶対位置に変換します。
    static int CalculateIndex(int index, int bias, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, bufferCount);
        NN_SDK_REQUIRES_RANGE(bias, -bufferCount, bufferCount);
        return (index + bufferCount + bias) % bufferCount;
    }

    static int CalculateNextIndex(int index, int bufferCount) NN_NOEXCEPT
    {
        if(index >= (bufferCount - 1))
        {
            return 0;
        }

        return index + 1;
    }

    //!< Append() の内部実装
    template<typename T>
    void AppendImpl(const T& value) NN_NOEXCEPT
    {
        // 次の末尾位置を計算
        int nextIndex = RingLifoLight::CalculateNextIndex(m_Tail, NN_ARRAY_SIZE(m_Buffer));

        // 末尾位置に要素を書き込み
        m_Buffer[nextIndex] = value;

        // 末尾位置を更新
        m_Tail = nextIndex;

        // リングバッファ上の要素数を更新
        if (m_Count < LifoEntryCount)
        {
            ++m_Count;
        }
    }

    //!< Read() の内部実装
    template<typename T>
    int ReadImpl(T* buffer, int bufferCount) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        int countMax = 0;

        // buffer の書き換え後に要素数が減らない様に単調増加
        countMax = ::std::min(
            bufferCount, ::std::max(countMax, m_Count));

        // 読み出し開始時点の末尾位置を待避
        const int tail = m_Tail;

        int bias = countMax - 1;

        while (bias >= 0)
        {
            // 古い値から順に要素の読み出し
            const int index = RingLifoLight::CalculateIndex(
                tail, -bias, NN_ARRAY_SIZE(m_Buffer));

            buffer[bias] = m_Buffer[index];
            --bias;
        }

        return countMax;
    }
};

}}} // namespace nn::hid::detail
