﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_Event.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_ActivationCount.h"
#include "hid_RegisteredDeviceManager.h"
#include "hid_Settings.h"
#include "hid_XcdUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

system::RegisteredDeviceOld ConvertRegisteredDevice(const system::RegisteredDevice& source)
{
    system::RegisteredDeviceOld output;
    output.address = source.address;
    output.deviceType = source.deviceType;
    std::memcpy(output.identificationCode, source.identificationCode, sizeof(output.identificationCode));
    output.interfaceType = source.interfaceType;
    output.mainColor = source.mainColor;
    output.subColor = source.subColor;
    output.subType = source.subType;
    return output;
}

}

RegisteredDeviceManager::RegisteredDeviceManager() NN_NOEXCEPT :
    m_ActivationCount(),
    m_LastRegistrationDeviceCount(std::numeric_limits<int64_t>::min()),
    m_pSender(nullptr),
    m_ConnectionTriggerTimeoutEvent(::nn::os::EventClearMode_ManualClear, true),
    m_Mutex(false),
    m_DeviceRegisteredEvent(::nn::os::EventClearMode_ManualClear, true),
    m_PlayReportUpdateEvent(::nn::os::EventClearMode_ManualClear, true),
    m_AbstractedPadCount(0)
{
    // 何もしない
}

::nn::Result RegisteredDeviceManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result RegisteredDeviceManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
    }

    NN_RESULT_SUCCESS;
}

void RegisteredDeviceManager::AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAbstractedPads[m_AbstractedPadCount] = pPads;
    m_AbstractedPadCount++;
}

void RegisteredDeviceManager::SetOvlnSenderManager(OvlnSenderManager* pSender) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSender);

    m_pSender = pSender;
}

void RegisteredDeviceManager::BindXcdRegistrationEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    ::nn::xcd::BindPairingCompletedEvent(pSystemEvent);
}

void RegisteredDeviceManager::UnbindXcdRegistrationEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    ::nn::xcd::UnbindPairingCompletedEvent(pSystemEvent);
}

void RegisteredDeviceManager::BindXcdConnectionTriggerTimeoutEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    ::nn::xcd::SetConnectionTriggerTimeoutEvent(pSystemEvent);
}

void RegisteredDeviceManager::UnindXcdConnectionTriggerTimeoutEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    NN_UNUSED(pSystemEvent);

    // 何もしない
}

void RegisteredDeviceManager::SignalConnectionTriggerTimeout() NN_NOEXCEPT
{
    m_ConnectionTriggerTimeoutEvent.Signal();
}

void RegisteredDeviceManager::UpdateRegisteredDevice() NN_NOEXCEPT
{
    NN_STATIC_ASSERT(system::RegisteredDeviceCountMax >= ::nn::xcd::RegisteredDeviceCountMax);
    ::nn::xcd::RegisteredDeviceInfo xcdRegisteredDevices[nn::xcd::RegisteredDeviceCountMax];
    auto xcdDeviceCount = static_cast<int>(::nn::xcd::GetRegisteredDevices(xcdRegisteredDevices, NN_ARRAY_SIZE(xcdRegisteredDevices)));
    NN_ABORT_UNLESS_LESS_EQUAL(m_RegisteredDeviceCount, static_cast<int>(system::RegisteredDeviceCountMax));

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    m_RegisteredDeviceCount = 0;
    for (int i = 0; i < xcdDeviceCount; i++)
    {
        if (xcdRegisteredDevices[i].isIncomplete == false)
        {
            auto& hidInfo = m_RegisteredDevices[m_RegisteredDeviceCount];
            auto& xcdInfo = xcdRegisteredDevices[i];
            hidInfo.address = xcdInfo.bluetooth.address;
            memcpy(hidInfo.bdName.raw, xcdInfo.bluetooth.bdName.name, sizeof(hidInfo.bdName.raw));
            memcpy(hidInfo.classOfDevice.raw, xcdInfo.bluetooth.classOfDevice.cod, sizeof(hidInfo.classOfDevice.raw));
            hidInfo.featureSet = xcdInfo.bluetooth.featureSet;
            hidInfo.hid.pid = xcdInfo.bluetooth.hid.pid;
            hidInfo.hid.vid = xcdInfo.bluetooth.hid.vid;
            memcpy(hidInfo.linkKey.raw, xcdInfo.bluetooth.linkKey.key, sizeof(hidInfo.linkKey.raw));
            memcpy(hidInfo.identificationCode, xcdInfo.identificationCode._code, sizeof(hidInfo.identificationCode));
            hidInfo.mainColor = xcdInfo.color.color[0];
            hidInfo.subColor = xcdInfo.color.color[1];
            hidInfo.thirdColor = xcdInfo.color.color[2];
            hidInfo.forthColor = xcdInfo.color.color[3];
            hidInfo.deviceType = ConvertDeviceType(xcdInfo.deviceType, true);
            hidInfo.subType = xcdInfo.subType;
            hidInfo.interfaceType = ConvertInterfaceType(xcdInfo.interfaceType);

            // 前回からの情報が更新されたデバイスかどうかをチェック
            if (m_LastRegistrationDeviceCount < xcdInfo.registerationCount)
            {
                NpadControllerColor color;
                color.main = xcdInfo.color.color[0];
                color.sub = xcdInfo.color.color[1];
                NotifyPairingComplete(hidInfo.deviceType, color, xcdInfo.interfaceType);

                // 新規登録されたデバイスの場合は、コントローラーサポートアプレットに通知を発行
                if (xcdInfo.isUpdated == false)
                {
                    m_DeviceRegisteredEvent.Signal();
                }

                m_LastRegistrationDeviceCount = xcdInfo.registerationCount;


                // プレイレポート用に変化を通知
                m_PlayReportUpdateEvent.Signal();
            }

            ++m_RegisteredDeviceCount;
        }
    }
}

int RegisteredDeviceManager::GetRegisteredDevices(system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    int returnCount = std::min(m_RegisteredDeviceCount, count);
    for (int i = 0; i < returnCount; ++i)
    {
        pOutValues[i] = ConvertRegisteredDevice(m_RegisteredDevices[i]);
        // 未知のデバイスエミュレーション機能が有効の場合は Unknown で上書き
        if (::nn::hid::detail::FirmwareDebugSettings::EmulateFutureDevice())
        {
            pOutValues[i].deviceType = system::DeviceType::Unknown::Mask;
        }
    }
    return returnCount;
}

int RegisteredDeviceManager::GetRegisteredDevices(system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_Mutex)> locker(m_Mutex);
    int returnCount = std::min(m_RegisteredDeviceCount, count);
    for (int i = 0; i < returnCount; ++i)
    {
        pOutValues[i] = m_RegisteredDevices[i];
    }
    return returnCount;
}

void RegisteredDeviceManager::AddRegisteredDevice(const system::RegisteredDevice& device) NN_NOEXCEPT
{
    auto xcdInfo = nn::xcd::RegisteredDeviceInfo();
    xcdInfo.bluetooth.address = device.address;
    memcpy(xcdInfo.bluetooth.bdName.name, device.bdName.raw, sizeof(xcdInfo.bluetooth.bdName.name));
    memcpy(xcdInfo.bluetooth.classOfDevice.cod, device.classOfDevice.raw, sizeof(xcdInfo.bluetooth.classOfDevice.cod));
    xcdInfo.bluetooth.featureSet = device.featureSet;
    xcdInfo.bluetooth.hid.pid = device.hid.pid;
    xcdInfo.bluetooth.hid.vid = device.hid.vid;
    memcpy(xcdInfo.bluetooth.linkKey.key, device.linkKey.raw, sizeof(xcdInfo.bluetooth.linkKey.key));
    memcpy(xcdInfo.identificationCode._code, device.identificationCode, 16);
    xcdInfo.color.color[0] = device.mainColor;
    xcdInfo.color.color[1] = device.subColor;
    xcdInfo.color.color[2] = device.thirdColor;
    xcdInfo.color.color[3] = device.forthColor;
    xcdInfo.deviceType = ConvertDeviceType(device.deviceType);
    xcdInfo.subType = device.subType;
    xcdInfo.interfaceType = ConvertInterfaceType(device.interfaceType);
    nn::xcd::AddRegisteredDeviceInfo(xcdInfo);
}


void RegisteredDeviceManager::NotifyPairingComplete(system::NpadDeviceTypeSet device,
                                                    const NpadControllerColor& color,
                                                    ::nn::xcd::InterfaceType interfaceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSender);

    nn::ovln::format::ControllerPairingCompletedData data;

    data.deviceType = device;
    // コントローラーの色情報が無効なデバイスは現状存在しない
    data.color.isValid = true;
    data.color.color = color;

    switch (interfaceType)
    {
    case ::nn::xcd::InterfaceType_Bluetooth:
        data.pairingMethod = ::nn::ovln::format::ControllerPairingMethod_Button;
        break;
    case ::nn::xcd::InterfaceType_Usb:
        data.pairingMethod = ::nn::ovln::format::ControllerPairingMethod_USB;
        break;
    case ::nn::xcd::InterfaceType_Uart:
        data.pairingMethod = ::nn::ovln::format::ControllerPairingMethod_Joint;
        break;
    default:
        break;
    }

    m_pSender->Send(data);
}

::nn::Result RegisteredDeviceManager::AcquireConnectionTriggerTimeoutEvent(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_ConnectionTriggerTimeoutEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

::nn::Result RegisteredDeviceManager::SendConnectionTrigger(nn::bluetooth::Address& address) NN_NOEXCEPT
{
    // デバイスがすでに物理リンクされているかどうかを探索
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->IsAttached() == false)
        {
            continue;
        }

        if (m_pAbstractedPads[i]->GetType() == AbstractedPadType_Xcd)
        {
            if (static_cast<AbstractedPadXcd*>(m_pAbstractedPads[i])->GetAddress() == address)
            {
                // リンクはされているけど接続状態にない、かつ USB 接続の場合は接続状態に変更
                if (m_pAbstractedPads[i]->IsConnected() == false &&
                    m_pAbstractedPads[i]->GetInterfaceType() == system::InterfaceType_Usb)
                {
                    m_pAbstractedPads[i]->Connect();
                    NN_RESULT_SUCCESS;
                }

                // コントローラーが接続状態にあるので、Result を返す
                NN_RESULT_THROW(system::ResultBluetoothAlreadyConnected());
            }
        }
    }

    auto result = ::nn::xcd::SendConnectionTrigger(address);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultBluetoothDeviceNotRegistered, system::ResultBluetoothDeviceNotRegistered());
        NN_RESULT_CATCH_CONVERT(::nn::xcd::ResultBluetoothOtherTriggerEventOnProgress, system::ResultBluetoothOtherTriggerEventOnProgreess());
        // このエラーが返ってくる場合は、Xcd/Bluetooth としては接続状態にあり Hid としてはまだ接続を認識していない状態なのでそのうち接続されるはず
        NN_RESULT_CATCH_CONVERT(nn::xcd::ResultBluetoothAlreadyConnected, ResultSuccess());
    NN_RESULT_END_TRY;
    NN_RESULT_DO(result);
    NN_RESULT_SUCCESS;
}

::nn::Result RegisteredDeviceManager::AcquireDeviceRegisteredEventForControllerSupport(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_DeviceRegisteredEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

Result RegisteredDeviceManager::AcquireUpdateForPlayReportEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = m_PlayReportUpdateEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
