﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/hid/hid_ResultPalma.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/xcd/xcd_Ble.h>

#include "hid_PalmaResourceManager.h"

namespace nn { namespace hid { namespace detail {

namespace
{
    struct PalmaConnectionHandleField final
    {
        uint32_t Index;
        uint32_t XcdHandle;
    };

    bool IsValidPalmaConnectionHandle(const PalmaConnectionHandle& handle)
    {
        auto* pHandleField = reinterpret_cast<const PalmaConnectionHandleField*>(&handle);
        return (pHandleField->Index < ::nn::xcd::BleDeviceCountMax) && (pHandleField->XcdHandle != ::nn::xcd::BleInvalidConnectionHandle);
    }

    uint32_t GetIndex(const PalmaConnectionHandle& handle)
    {
        auto* pHandleField = reinterpret_cast<const PalmaConnectionHandleField*>(&handle);
        return pHandleField->Index;
    }

    PalmaManager& GetPalmaManager(PalmaManagerArray& array, const PalmaConnectionHandle& handle)
    {
        return array[GetIndex(handle)];
    }
}

PalmaResourceManager::PalmaResourceManager() NN_NOEXCEPT
    : m_PalmaMutex(false)
    , m_pAppletResourceManager(nullptr)
    , m_pAppletResourceManagerMutex(nullptr)
{
    // 何もしない
}

PalmaResourceManager::~PalmaResourceManager() NN_NOEXCEPT
{
    // 何もしない
}

void PalmaResourceManager::SetAppletResourceManager(
    AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    NN_SDK_REQUIRES_NOT_NULL(pMutex);
    m_pAppletResourceManager = pManager;
    m_pAppletResourceManagerMutex = pMutex;
}

::nn::Result PalmaResourceManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(), ResultGamePadDriverActivationUpperLimitOver());
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(), ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void PalmaResourceManager::UpdateOnAruidSwitch() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> palmaLocker(m_PalmaMutex);
    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)> appletResourceLlocker(*m_pAppletResourceManagerMutex);

    bool isScanEnabled = false;
    for (const AppletResourceEntry& entry :
             m_pAppletResourceManager->GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == m_pAppletResourceManager->GetAppletResourceUserId())
        {
            isScanEnabled = entry.flags.Test<AppletResourceFlag::EnablesPalmaScanAll>();
            break;
        }
    }

    nn::xcd::SetIsPalmaAllConnectable(isScanEnabled);
}

void PalmaResourceManager::UnregisterAppletResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    for (auto& palma : m_PalmaManagers)
    {
        palma.ResetTransferMemory(aruid);
    }
}

void PalmaResourceManager::Attach(const nn::hid::NpadIdType& npadId, nn::xcd::BleConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    for (auto& palma : m_PalmaManagers)
    {
        if (!palma.IsValidHandle())
        {
            palma.SetXcdHandle(handle);
            palma.SetNpadId(npadId);
            return;
        }
    }
}

void PalmaResourceManager::Detach(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    for (auto& palma : m_PalmaManagers)
    {
        ::nn::hid::NpadIdType id;
        if (palma.GetNpadId(&id).IsSuccess() && npadId == id)
        {
            palma.InvalidateXcdHandle();
        }
    }
}

::nn::Result PalmaResourceManager::GetTransferMemoryType(::nn::os::TransferMemoryType** pTransferMemory, PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).GetTransferMemoryType(pTransferMemory));

    NN_RESULT_SUCCESS;
}

void PalmaResourceManager::SetTransferMemory(PalmaConnectionHandle handle, void* pMappedAddress, uint64_t size) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)> appletResourceLlocker(*m_pAppletResourceManagerMutex);

    NN_SDK_REQUIRES(IsValidPalmaConnectionHandle(handle));

    GetPalmaManager(m_PalmaManagers, handle).SetTransferMemory(pMappedAddress, size, m_pAppletResourceManager->GetAppletResourceUserId());
}

::nn::Result PalmaResourceManager::GetPalmaConnectionHandle(PalmaConnectionHandle* pOutValue, NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    auto index = 0;
    for (auto& palma : m_PalmaManagers)
    {
        ::nn::hid::NpadIdType id;
        if (palma.GetNpadId(&id).IsSuccess() && npadId == id)
        {
            ::nn::xcd::BleConnectionHandle xcdHandle;
            palma.GetXcdHandle(&xcdHandle);

            auto* pHandleField = reinterpret_cast<PalmaConnectionHandleField*>(pOutValue);
            pHandleField->Index = index;
            pHandleField->XcdHandle = xcdHandle;
            NN_RESULT_SUCCESS;
        }
        index++;
    }
    NN_RESULT_THROW(ResultPalmaNotConnected());
}

::nn::Result PalmaResourceManager::Initialize(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).Initialize());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::AcquireOperationCompleteEventHandle(::nn::os::NativeHandle* pOutValue, PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).AcquireOperationCompleteEventHandle(pOutValue));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::GetPalmaOperationInfo(PalmaOperationType* pOutValue, const nn::sf::OutBuffer& outBuffer, PalmaConnectionHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).GetPalmaOperationInfo(pOutValue, outBuffer.GetPointerUnsafe(), outBuffer.GetSize()));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::GetPalmaOperationResult(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).GetPalmaOperationResult());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::PlayPalmaActivity(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).PlayPalmaActivity(index));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::SetFrModeType(PalmaConnectionHandle handle, uint64_t type) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).SetFrModeType(type));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ReadStep(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ReadStep());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::EnableStepCounter(PalmaConnectionHandle handle, bool isEnabled) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).EnableStepCounter(isEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ResetStep(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ResetStep());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ReadApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ReadApplicationSection(address, size));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::WriteApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size, PalmaApplicationSectionAccessBuffer buffer) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).WriteApplicationSection(address, size, buffer));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ReadPalmaUniqueCode(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ReadPalmaUniqueCode());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::SetPalmaUniqueCodeInvalid(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).SetPalmaUniqueCodeInvalid());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::WritePalmaActivityEntry(PalmaConnectionHandle handle, uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).WritePalmaActivityEntry(index, ledIndex, waveSet, waveIndex));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::WritePalmaRgbLedPatternEntry(PalmaConnectionHandle handle, uint64_t index, const char* buffer, size_t size) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).WritePalmaRgbLedPatternEntry(index, buffer, size));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::WritePalmaWaveEntry(PalmaConnectionHandle handle, uint64_t waveSet, uint64_t index, size_t size) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).WritePalmaWaveEntry(waveSet, index, size));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::SetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle, int32_t version) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).SetPalmaDataBaseIdentificationVersion(version));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::GetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).GetPalmaDataBaseIdentificationVersion());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::SuspendPalmaFeature(PalmaConnectionHandle handle, PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).SuspendPalmaFeature(suspendFeatureSet));

    NN_RESULT_SUCCESS;
}

void PalmaResourceManager::SetIsPalmaAllConnectable(const ::nn::applet::AppletResourceUserId& aruid, bool connectable) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    ::std::lock_guard<decltype(*m_pAppletResourceManagerMutex)> appletResourceLocker(*m_pAppletResourceManagerMutex);
    m_pAppletResourceManager->EnableAppletResourcePalmaScanAll(aruid, connectable);

    if (m_pAppletResourceManager->GetAppletResourceUserId() == aruid ||
        aruid == nn::applet::AppletResourceUserId::GetInvalidId())
    {
        nn::xcd::SetIsPalmaAllConnectable(connectable);
    }
}

void PalmaResourceManager::SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);
    nn::xcd::SetIsPalmaPairedConnectable(connectable);
}

::nn::Result PalmaResourceManager::PairPalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).PairPalma());

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ReadPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ReadPalmaPlayLog(index));

    NN_RESULT_SUCCESS;
}

::nn::Result PalmaResourceManager::ResetPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_PalmaMutex)> locker(m_PalmaMutex);

    NN_RESULT_THROW_UNLESS(IsValidPalmaConnectionHandle(handle), nn::hid::ResultPalmaInvalidHandle());

    NN_RESULT_DO(GetPalmaManager(m_PalmaManagers, handle).ResetPalmaPlayLog(index));

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
