﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_VibratorGcDriver.h"
#include "hid_VibratorXcdDriver.h"

namespace nn { namespace hid { namespace detail {

//! プレイヤ 1 人あたりの振動子の最大数。最大で左右の 2 個。
const int NpadVibratorCountPerPlayerMax = 2;

//!< 振動デバイスの管理を担うクラスです。
class NpadVibrationController final
{
    NN_DISALLOW_COPY(NpadVibrationController);
    NN_DISALLOW_MOVE(NpadVibrationController);

private:
    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< Vibrator XCD Driver
    VibratorXcdDriver* m_pVibratorXcdDrivers[NpadVibratorCountPerPlayerMax];

    //!< Vibrator Gc Driver
    VibratorGcDriver* m_pVibratorGcDriver;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

public:
    NpadVibrationController() NN_NOEXCEPT;

    ~NpadVibrationController() NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< VibratorXcdDriver をに設定します。
    void SetVibratorXcdDriver(VibratorXcdDriver vibratorArray[], int vibratorCount) NN_NOEXCEPT;

    //!< VibratorGcDriver をに設定します。
    void SetVibratorGcDriver(VibratorGcDriver* pVibrator) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< 振動デバイスの割り当て状態を更新します。
    void UpdateDeviceStates() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
