﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadColor.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>
#include <nn/util/util_MathTypes.h>

#include "hid_CommonStateUtility.h"
#include "hid_NpadUtilTypes.h"

namespace nn { namespace hid { namespace detail {

const AbstractedPadButtonSet ButtonMask_UnSupportedOnJoyLeftHorizontal =
    AbstractedPadButton::L::Mask |
    AbstractedPadButton::ZL::Mask;

const AbstractedPadButtonSet ButtonMask_UnSupportedOnJoyLeftVertical =
    AbstractedPadButton::SL::Mask |
    AbstractedPadButton::SR::Mask;

const AbstractedPadButtonSet ButtonMask_UnSupportedOnJoyRightHorizontal =
    AbstractedPadButton::R::Mask |
    AbstractedPadButton::ZR::Mask;

const AbstractedPadButtonSet ButtonMask_UnSupportedOnJoyRightVertical =
    AbstractedPadButton::SL::Mask |
    AbstractedPadButton::SR::Mask;

const AbstractedPadButtonSet ButtonMask_StickEmulationButtonMask =
                                        AbstractedPadButton::StickLUp::Mask |
                                        AbstractedPadButton::StickLRight::Mask |
                                        AbstractedPadButton::StickLDown::Mask |
                                        AbstractedPadButton::StickLLeft::Mask |
                                        AbstractedPadButton::StickRUp::Mask |
                                        AbstractedPadButton::StickRRight::Mask |
                                        AbstractedPadButton::StickRDown::Mask |
                                        AbstractedPadButton::StickRLeft::Mask;

const AbstractedPadButtonSet ButtonMask_SystemButtonMask =
                                        AbstractedPadButton::Home::Mask |
                                        AbstractedPadButton::Shot::Mask;

//!< 基本的な構成を持つ Xpad のデジタルボタンに禁則処理を適応します。
inline NpadButtonSet RestrictNpadButtons(const NpadButtonSet& buttons,
                                         ControlPadRestrictionType mode,
                                         bool enabled
                                              ) NN_NOEXCEPT
{
    return RestrictButtonsIncludingDpad(buttons, mode, enabled);
}

//!< 十字キーエミュレーションボタンを有効にします。
inline void SetCrossStickEmulationButtons(NpadButtonSet* pOutButtons,
                                          const ::nn::hid::AnalogStickState& stick,
                                          bool isLeft)
{
    // TORIAEZU: 判定領域は[上][下][左][右]それぞれ xy_deg = 45度。
    //           余った領域は両隣の領域の論理和となります。
    //           判定領域の回転角は rot_deg = 0度。

    // [右][右上]境界ベクトル
    // ( cos((xy_deg / 2 + rot_deg) * (pi/180)), sin((xy_deg / 2 + rot_deg) * (pi/180)))
    const ::nn::util::Float2 BoundaryRightAndRightUp = {{{0.92387953251f, 0.38268343235f}}};
    // [右上][上]境界ベクトル
    // ( cos((90 - xy_deg / 2 + rot_deg) * (pi/180)), sin((90 - xy_deg / 2 + rot_deg) * (pi/180)))
    const ::nn::util::Float2 BoundaryRightUpAndUp = {{{0.38268343236f, 0.92387953251f}}};

    // クランプ済のスティック入力がデッドゾーン範囲内の場合は何もしない
    int32_t SquaredRadius = stick.x * stick.x + stick.y * stick.y;

    if(SquaredRadius < CrossStickEmulationDeadZoneRadiusSquaredCount)
    {
        return;
    }

    // [上][左上]境界ベクトル
    const ::nn::util::Float2 BoundaryUpAndLeftUp = {{{-BoundaryRightAndRightUp.y,
                                                      BoundaryRightAndRightUp.x}}};
    // [左上][左]境界ベクトル
    const ::nn::util::Float2 BoundaryLeftUpAndLeft = {{{-BoundaryRightUpAndUp.y,
                                                        BoundaryRightUpAndUp.x}}};
    // 入力ベクトル
    const ::nn::util::Float2 StickInput = {{{static_cast<float>(stick.x),
                                             static_cast<float>(stick.y)}}};

    // 境界ベクトルと入力ベクトルとの内積結果から領域判定を行います。

    // [右][右上]境界ベクトルとの内積
    if(Dot(StickInput, BoundaryRightAndRightUp) < 0.0f)
    {
        // [上][左上]境界ベクトルとの内積
        if(Dot(StickInput, BoundaryUpAndLeftUp) < 0.0f)
        {
            // [左上][左]境界ベクトルとの内積
            if(Dot(StickInput, BoundaryLeftUpAndLeft) < 0.0f)
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLDown>()
                       : pOutButtons->Set<NpadButton::StickRDown>();
            }
            else
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLLeft>()
                       : pOutButtons->Set<NpadButton::StickRLeft>();

                isLeft ? pOutButtons->Set<NpadButton::StickLDown>()
                       : pOutButtons->Set<NpadButton::StickRDown>();
            }
        }
        else
        {
            // [右上][上]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryRightUpAndUp) < 0.0f)
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLLeft>()
                       : pOutButtons->Set<NpadButton::StickRLeft>();
            }
            else
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLLeft>()
                       : pOutButtons->Set<NpadButton::StickRLeft>();

                isLeft ? pOutButtons->Set<NpadButton::StickLUp>()
                       : pOutButtons->Set<NpadButton::StickRUp>();
            }
        }
    }
    else
    {
        // [上][左上]境界ベクトルとの内積
        if(Dot(StickInput, BoundaryUpAndLeftUp) < 0.0f)
        {
            // [右上][上]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryRightUpAndUp) < 0.0f)
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLRight>()
                       : pOutButtons->Set<NpadButton::StickRRight>();

                isLeft ? pOutButtons->Set<NpadButton::StickLDown>()
                       : pOutButtons->Set<NpadButton::StickRDown>();
            }
            else
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLRight>()
                       : pOutButtons->Set<NpadButton::StickRRight>();
            }
        }
        else
        {
            // [左上][左]境界ベクトルとの内積
            if (Dot(StickInput, BoundaryLeftUpAndLeft) < 0.0f)
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLRight>()
                       : pOutButtons->Set<NpadButton::StickRRight>();

                isLeft ? pOutButtons->Set<NpadButton::StickLUp>()
                       : pOutButtons->Set<NpadButton::StickRUp>();
            }
            else
            {
                isLeft ? pOutButtons->Set<NpadButton::StickLUp>()
                       : pOutButtons->Set<NpadButton::StickRUp>();
            }
        }
    }
}

inline void SetCrossStickEmulationButtonsToUdlr(NpadButtonSet* pOutButtons,
    const ::nn::hid::AnalogStickState& stick,
    bool isLeft)
{
    NpadButtonSet buttons = NpadButtonSet();
    SetCrossStickEmulationButtons(&buttons, stick, isLeft);

    if (isLeft)
    {
        pOutButtons->Set<NpadButton::Left>(buttons.Test<NpadButton::StickLLeft>());
        pOutButtons->Set<NpadButton::Up>(buttons.Test<NpadButton::StickLUp>());
        pOutButtons->Set<NpadButton::Right>(buttons.Test<NpadButton::StickLRight>());
        pOutButtons->Set<NpadButton::Down>(buttons.Test<NpadButton::StickLDown>());
    }
    else
    {
        pOutButtons->Set<NpadButton::Left>(buttons.Test<NpadButton::StickRLeft>());
        pOutButtons->Set<NpadButton::Up>(buttons.Test<NpadButton::StickRUp>());
        pOutButtons->Set<NpadButton::Right>(buttons.Test<NpadButton::StickRRight>());
        pOutButtons->Set<NpadButton::Down>(buttons.Test<NpadButton::StickRDown>());
    }
}


//!< BasicXpad のデジタルボタンの押下状態を Npad のものに変換します。(自動操作用)
inline NpadButtonSet ConvertBasicXpadButtonSet(const BasicXpadButtonSet& xpadButton
                                         ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::A>(xpadButton.Test<BasicXpadButton::A>());
    buttons.Set<NpadButton::B>(xpadButton.Test<BasicXpadButton::B>());
    buttons.Set<NpadButton::X>(xpadButton.Test<BasicXpadButton::X>());
    buttons.Set<NpadButton::Y>(xpadButton.Test<BasicXpadButton::Y>());
    buttons.Set<NpadButton::L>(xpadButton.Test<BasicXpadButton::L>());
    buttons.Set<NpadButton::R>(xpadButton.Test<BasicXpadButton::R>());
    buttons.Set<NpadButton::ZL>(xpadButton.Test<BasicXpadButton::ZL>());
    buttons.Set<NpadButton::ZR>(xpadButton.Test<BasicXpadButton::ZR>());
    buttons.Set<NpadButton::Plus>(xpadButton.Test<BasicXpadButton::Start>());
    buttons.Set<NpadButton::Minus>(xpadButton.Test<BasicXpadButton::Select>());
    buttons.Set<NpadButton::Left>(xpadButton.Test<BasicXpadButton::Left>());
    buttons.Set<NpadButton::Up>(xpadButton.Test<BasicXpadButton::Up>());
    buttons.Set<NpadButton::Right>(xpadButton.Test<BasicXpadButton::Right>());
    buttons.Set<NpadButton::Down>(xpadButton.Test<BasicXpadButton::Down>());
    buttons.Set<NpadButton::StickL>(xpadButton.Test<BasicXpadButton::StickL>());
    buttons.Set<NpadButton::StickR>(xpadButton.Test<BasicXpadButton::StickR>());

    return buttons;
}

//!< NpadId との関連を考慮して有効な操作形態を取得します
inline NpadStyleSet MaskNpadStyleSetWithId(const NpadStyleSet& styleSet, const NpadIdType& id) NN_NOEXCEPT
{
    auto outStyle = styleSet;

    // ID毎に有効な操作形態が異なる
    switch (id)
    {
    case NpadId::No1:
    case NpadId::No2:
    case NpadId::No3:
    case NpadId::No4:
    case NpadId::No5:
    case NpadId::No6:
    case NpadId::No7:
    case NpadId::No8:
        outStyle &=  (NpadStyleFullKey::Mask |
                      NpadStyleJoyDual::Mask |
                      NpadStyleJoyLeft::Mask |
                      NpadStyleJoyRight::Mask |
                      NpadStyleGc::Mask |
                      NpadStylePalma::Mask |
                      system::NpadStyleSystem::Mask |
                      system::NpadStyleSystemExt::Mask);
        break;

    case NpadId::Handheld:
        outStyle &= (NpadStyleHandheld::Mask |
                     system::NpadStyleSystem::Mask |
                     system::NpadStyleSystemExt::Mask);
        break;

    case system::NpadSystemId::Other:
        outStyle &= (system::NpadStyleSystem::Mask |
                     system::NpadStyleSystemExt::Mask);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return outStyle;
}

//!< Joy-Con 2本持ちかどうかを判定します
inline bool IsJoyDual(const system::DeviceTypeSet& deviceType)
{
    return (deviceType & DeviceTypeMask_JoyCons) == DeviceTypeMask_JoyCons;
}

//!< レール接続状態の Joy-Con 2本持ちかどうかを判定します
inline bool IsHandheldJoyDual(const system::DeviceTypeSet& deviceType)
{
    return (deviceType & DeviceTypeMask_HandheldJoyCons) == DeviceTypeMask_HandheldJoyCons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Npad のものに変換します。
inline NpadButtonSet ConvertButtonSet(const AbstractedPadButtonSet& input,
                                      const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::A>());
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::B>());
    buttons.Set<NpadButton::X>(input.Test<AbstractedPadButton::X>());
    buttons.Set<NpadButton::Y>(input.Test<AbstractedPadButton::Y>());
    buttons.Set<NpadButton::L>(input.Test<AbstractedPadButton::L>());
    buttons.Set<NpadButton::R>(input.Test<AbstractedPadButton::R>());
    buttons.Set<NpadButton::ZL>(input.Test<AbstractedPadButton::ZL>());
    buttons.Set<NpadButton::ZR>(input.Test<AbstractedPadButton::ZR>());
    buttons.Set<NpadButton::Plus>(input.Test<AbstractedPadButton::Start>());
    buttons.Set<NpadButton::Minus>(input.Test<AbstractedPadButton::Select>());
    buttons.Set<NpadButton::Left>(input.Test<AbstractedPadButton::Left>());
    buttons.Set<NpadButton::Up>(input.Test<AbstractedPadButton::Up>());
    buttons.Set<NpadButton::Right>(input.Test<AbstractedPadButton::Right>());
    buttons.Set<NpadButton::Down>(input.Test<AbstractedPadButton::Down>());
    buttons.Set<NpadButton::StickL>(input.Test<AbstractedPadButton::StickL>());
    buttons.Set<NpadButton::StickR>(input.Test<AbstractedPadButton::StickR>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickLLeft>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickLUp>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickLRight>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickLDown>());
    buttons.Set<NpadButton::StickRLeft>(input.Test<AbstractedPadButton::StickRLeft>());
    buttons.Set<NpadButton::StickRUp>(input.Test<AbstractedPadButton::StickRUp>());
    buttons.Set<NpadButton::StickRRight>(input.Test<AbstractedPadButton::StickRRight>());
    buttons.Set<NpadButton::StickRDown>(input.Test<AbstractedPadButton::StickRDown>());
    if (deviceType.Test<system::DeviceType::JoyConLeft>())
    {
        buttons.Set<NpadJoyButton::LeftSL>(input.Test<AbstractedPadButton::SL>());
        buttons.Set<NpadJoyButton::LeftSR>(input.Test<AbstractedPadButton::SR>());
    }
    else if(deviceType.Test<system::DeviceType::JoyConRight>())
    {
        buttons.Set<NpadJoyButton::RightSL>(input.Test<AbstractedPadButton::SL>());
        buttons.Set<NpadJoyButton::RightSR>(input.Test<AbstractedPadButton::SR>());
    }
    buttons.Set<NpadPalmaButton::Palma>(input.Test<AbstractedPadButton::Palma>());

    return buttons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Joy-Con(L) 横持ちとみなして NpadSystem(Ext)State のものに変換します。
inline NpadButtonSet ConvertButtonSetForLeftSingleHorizontal(const AbstractedPadButtonSet& input
                                                            ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::Minus>(input.Test<AbstractedPadButton::Select>());
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::Left>());
    buttons.Set<NpadButton::Y>(input.Test<AbstractedPadButton::Up>());
    buttons.Set<NpadButton::X>(input.Test<AbstractedPadButton::Right>());
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::Down>());
    buttons.Set<NpadJoyButton::L>(input.Test<AbstractedPadButton::SL>());
    buttons.Set<NpadJoyButton::R>(input.Test<AbstractedPadButton::SR>());
    buttons.Set<NpadButton::StickL>(input.Test<AbstractedPadButton::StickL>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickLLeft>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickLUp>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickLRight>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickLDown>());

    return buttons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Joy-Con(L) 縦持ちとみなして NpadSystem(Ext)State のものに変換します。
inline NpadButtonSet ConvertButtonSetForLeftSingleVertical(const AbstractedPadButtonSet& input
                                                           ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::Minus>(input.Test<AbstractedPadButton::Select>());
    buttons.Set<NpadButton::Y>(input.Test<AbstractedPadButton::Left>());
    buttons.Set<NpadButton::X>(input.Test<AbstractedPadButton::Up>());
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::Right>());
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::Down>());
    buttons.Set<NpadButton::L>(input.Test<AbstractedPadButton::L>());
    buttons.Set<NpadButton::ZL>(input.Test<AbstractedPadButton::ZL>());
    buttons.Set<NpadButton::StickL>(input.Test<AbstractedPadButton::StickL>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickLLeft>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickLUp>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickLRight>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickLDown>());

    return buttons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Joy-Con(R) 横持ちとみなして NpadSystem(Ext)State のものに変換します。
inline NpadButtonSet ConvertButtonSetForRightSingleHorizontal(const AbstractedPadButtonSet& input
                                                             ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::A>());
    buttons.Set<NpadButton::Y>(input.Test<AbstractedPadButton::B>());
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::X>());
    buttons.Set<NpadButton::X>(input.Test<AbstractedPadButton::Y>());
    buttons.Set<NpadButton::Plus>(input.Test<AbstractedPadButton::Start>());
    buttons.Set<NpadJoyButton::L>(input.Test<AbstractedPadButton::SL>());
    buttons.Set<NpadJoyButton::R>(input.Test<AbstractedPadButton::SR>());
    buttons.Set<NpadButton::StickL>(input.Test<AbstractedPadButton::StickR>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickRLeft>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickRUp>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickRRight>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickRDown>());

    return buttons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Joy-Con(R) 縦持ちとみなして NpadSystem(Ext)State のものに変換します。
inline NpadButtonSet ConvertButtonSetForRightSingleVertical(const AbstractedPadButtonSet& input
                                                           ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::A>());
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::B>());
    buttons.Set<NpadButton::X>(input.Test<AbstractedPadButton::X>());
    buttons.Set<NpadButton::Y>(input.Test<AbstractedPadButton::Y>());
    buttons.Set<NpadButton::R>(input.Test<AbstractedPadButton::R>());
    buttons.Set<NpadButton::ZR>(input.Test<AbstractedPadButton::ZR>());
    buttons.Set<NpadButton::Plus>(input.Test<AbstractedPadButton::Start>());
    buttons.Set<NpadButton::StickL>(input.Test<AbstractedPadButton::StickR>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickRLeft>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickRUp>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickRRight>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickRDown>());

    return buttons;
}

//!< AbstractedPad のデジタルボタンの押下状態を Palma とみなして NpadSystem(Ext)State のものに変換します。
inline NpadButtonSet ConvertButtonSetForPalma(const AbstractedPadButtonSet& input
                                             ) NN_NOEXCEPT
{
    NpadButtonSet buttons;
    buttons.Reset();
    buttons.Set<NpadButton::A>(input.Test<AbstractedPadButton::StickL>());
    buttons.Set<NpadButton::B>(input.Test<AbstractedPadButton::Palma>());
    buttons.Set<NpadButton::StickLLeft>(input.Test<AbstractedPadButton::StickLLeft>());
    buttons.Set<NpadButton::StickLUp>(input.Test<AbstractedPadButton::StickLUp>());
    buttons.Set<NpadButton::StickLRight>(input.Test<AbstractedPadButton::StickLRight>());
    buttons.Set<NpadButton::StickLDown>(input.Test<AbstractedPadButton::StickLDown>());

    return buttons;
}

//!< アナログスティックの入力状態が有効であれば代入します。
inline void OverwriteAnalogStickStateIfAvailable(AnalogStickState* pOutLeftState,
                                                 AnalogStickState* pOutRightState,
                                                 const AnalogStickState& leftInput,
                                                 const AnalogStickState& rightInput,
                                                 const system::DeviceTypeSet& deviceType)
{
    const auto leftStickDevice =  system::DeviceType::FullKeyController::Mask |
                                  system::DeviceType::JoyConLeft::Mask |
                                  system::DeviceType::HandheldJoyLeft::Mask |
                                  system::DeviceType::Palma::Mask;
    const auto rightStickDevice = system::DeviceType::FullKeyController::Mask |
                                  system::DeviceType::JoyConRight::Mask |
                                  system::DeviceType::HandheldJoyRight::Mask;
    if ((deviceType & leftStickDevice).IsAnyOn())
    {
        *pOutLeftState = leftInput;
    }
    if ((deviceType & rightStickDevice).IsAnyOn())
    {
        *pOutRightState = rightInput;
    }
}

//!< Npad の属性値を取得します。コントローラーが接続されている前提で処理を行います。
inline NpadAttributesSet GetNpadAttributesSet(bool isWired,
                                              const system::DeviceTypeSet& deviceType)
{
    NpadAttributesSet attributes = NpadAttributesSet();
    if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
    {
        attributes.Set<NpadJoyAttribute::IsLeftConnected>();
        attributes.Set<NpadJoyAttribute::IsLeftWired>(isWired);
    }
    if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
    {
        attributes.Set<NpadJoyAttribute::IsRightConnected>();
        attributes.Set<NpadJoyAttribute::IsRightWired>(isWired);
    }
    attributes.Set<NpadAttribute::IsConnected>();
    attributes.Set<NpadAttribute::IsWired>(isWired);
    return attributes;
}

//!< 電源状態を比較して一致するかどうかを確認します。
inline bool IsPowerInfoSame(system::NpadPowerInfo& first, system::NpadPowerInfo& second)
{
    return (first.batteryLevel == second.batteryLevel &&
            first.isCharging   == second.isCharging &&
            first.isPowered    == second.isPowered);
}

//!< 色情報を変換する
inline NpadControllerColor ConvertControllerColor(system::ControllerColor& input)
{
    return{ input.main, input.sub };
}

template <typename T>
inline bool IsChanged(const T& state1, const T& state2, const T& threshold)
{
    return (::std::abs(state1 - state2) > threshold);
}


//!< ディジタルボタンに変化があるか否かを返します
inline bool IsButtonSetChanged(const AbstractedPadButtonSet& state1,
                               const AbstractedPadButtonSet& state2) NN_NOEXCEPT
{
    // アナログスティックの十字キーエミュレーション入力は検知対象外
    const auto IgnoredButtons =
        AbstractedPadButton::StickLLeft::Mask  |
        AbstractedPadButton::StickLUp::Mask    |
        AbstractedPadButton::StickLRight::Mask |
        AbstractedPadButton::StickLDown::Mask  |
        AbstractedPadButton::StickRLeft::Mask  |
        AbstractedPadButton::StickRUp::Mask    |
        AbstractedPadButton::StickRRight::Mask |
        AbstractedPadButton::StickRDown::Mask;

    return ((state1 ^ state2) & ~IgnoredButtons).IsAnyOn();
}

//!< NpadState に変化があるか否かを返します
template <typename stateT>
inline bool IsAnalogStickStateChanged(const stateT& state1,
                                      const stateT& state2) NN_NOEXCEPT
{
    const bool IsLeftAnalogStickStateChanged = (
        IsChanged(state1.analogStickL.x, state2.analogStickL.x, AnalogStickThreshold) ||
        IsChanged(state1.analogStickL.y, state2.analogStickL.y, AnalogStickThreshold)
    );

    const bool IsRightAnalogStickStateChanged = (
        IsChanged(state1.analogStickR.x, state2.analogStickR.x, AnalogStickThreshold) ||
        IsChanged(state1.analogStickR.y, state2.analogStickR.y, AnalogStickThreshold)
    );

    return (IsLeftAnalogStickStateChanged || IsRightAnalogStickStateChanged);
}

//!< NpadState に変化があるか否かを返します
template <typename stateT>
inline bool IsDualAnalogStickStateChanged(const stateT& analogStick1, const stateT& analogStick2) NN_NOEXCEPT
{
    const bool IsLeftAnalogStickStateChanged = (
        IsChanged(analogStick1.left.x, analogStick2.left.x, AnalogStickThreshold) ||
        IsChanged(analogStick1.left.y, analogStick2.left.y, AnalogStickThreshold)
    );

    const bool IsRightAnalogStickStateChanged = (
        IsChanged(analogStick1.right.x, analogStick2.right.x, AnalogStickThreshold) ||
        IsChanged(analogStick1.right.y, analogStick2.right.y, AnalogStickThreshold)
    );

    return (IsLeftAnalogStickStateChanged || IsRightAnalogStickStateChanged);
}


//!< 6軸センサー入力に変化があるか否かを返します
inline bool IsSixAxisSensorStateChanged(const SixAxisSensorState& state) NN_NOEXCEPT
{
    const int AxisCountMax = 3; //!< 軸の数
    const float AngularVelocityThreshold = 0.3f; //!< 実験に基づく無入力判定閾値

    for (int i = 0; i < AxisCountMax; i++)
    {
        if(::std::abs(state.angularVelocity.v[i]) > AngularVelocityThreshold)
        {
            return true;
        }
    }
    return false;
}

}}} // namespace nn::hid::detail
