﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_NpadRailAttachmentTypes.h"

namespace nn { namespace hid { namespace detail {

struct RailAttachmentEntry
{
    //!< Attachment の状態
    NpadJoyConRailAttachmentState railAttachmentState;

    //!< レールアタッチメント機能が有効なデバイスの Xcd のデバイスハンドル
    ::nn::xcd::DeviceHandle xcdDeviceHandle;
};

//!< レールアタッチメントの状態の制御を担うクラスです。
class NpadRailAttachmentController final
{
    NN_DISALLOW_COPY(NpadRailAttachmentController);
    NN_DISALLOW_MOVE(NpadRailAttachmentController);

private:
    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< レール接続デバイスのエントリーです
    RailAttachmentEntry m_Entries[JoyConRailAttachmentCountMaxPerNpad];

public:
    NpadRailAttachmentController() NN_NOEXCEPT;

    ~NpadRailAttachmentController() NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< レール接続デバイスの状態を更新します。
    void UpdateDeviceStates() NN_NOEXCEPT;

    //!< レール接続デバイスのアクティブ状態を制御します。
    Result ActivateRailAttachment(bool activate, NpadJoyConRailIndex index) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている レール接続デバイスに割り当てられているの xcd のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithRailAttachment(::nn::xcd::DeviceHandle* pOutHandle, NpadJoyConRailIndex index) NN_NOEXCEPT;

    //!< レール接続デバイスの状態を取得します。
    NpadJoyConRailAttachmentState GetNpadJoyConRailAttachmentState(NpadJoyConRailIndex index) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
