﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadRailAttachmentController.h"
#include "hid_NpadStateUtility.h"

namespace nn { namespace hid { namespace detail {

NpadRailAttachmentController::NpadRailAttachmentController() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
{
    for (auto& entry : m_Entries)
    {
        entry.xcdDeviceHandle = nn::xcd::DeviceHandle();
        entry.railAttachmentState = NpadJoyConRailAttachmentState_NoDevice;
    }
}

NpadRailAttachmentController::~NpadRailAttachmentController() NN_NOEXCEPT { /* 何もしない */ }

void NpadRailAttachmentController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadRailAttachmentController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

::nn::Result NpadRailAttachmentController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadRailAttachmentController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadRailAttachmentController::UpdateDeviceStates() NN_NOEXCEPT
{
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    if (count == 0)
    {
        // 何も接続されていない
        for (auto& entry : m_Entries)
        {
            entry.xcdDeviceHandle = nn::xcd::DeviceHandle();
            entry.railAttachmentState = NpadJoyConRailAttachmentState_NoDevice;
        }
    }
    else
    {
        for (int i = 0; i < count; i++)
        {
            if (pPads[i]->IsConnected() &&
                pPads[i]->GetType() == AbstractedPadType_Xcd)
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    NpadJoyConRailIndex index;
                    if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::RailAttachmentLeft>())
                    {
                        index = NpadJoyConRailIndex_Left;
                    }
                    else if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::RailAttachmentRight>())
                    {
                        index = NpadJoyConRailIndex_Right;
                    }
                    else
                    {
                        // レールアタッチメントが無効なデバイス
                        continue;
                    }

                    // Rail が有効なデバイスが接続されている際の処理
                    if (m_Entries[index].railAttachmentState != NpadJoyConRailAttachmentState_Activated)
                    {
                        m_Entries[index].railAttachmentState = NpadJoyConRailAttachmentState_Deactivated;
                    }
                    m_Entries[index].xcdDeviceHandle = reinterpret_cast<AbstractedPadXcd*>(pPads[i])->GetXcdDeviceHandle();
                }
            }
        }

    }
}

::nn::Result NpadRailAttachmentController::GetXcdHandleForNpadWithRailAttachment(::nn::xcd::DeviceHandle* pOutHandle, NpadJoyConRailIndex index) NN_NOEXCEPT
{
    if (m_Entries[index].railAttachmentState == NpadJoyConRailAttachmentState_NoDevice)
    {
        NN_RESULT_THROW(ResultNoExternalBusFoundOnNpad());
    }

    *pOutHandle = m_Entries[index].xcdDeviceHandle;
    NN_RESULT_SUCCESS;
}

NpadJoyConRailAttachmentState NpadRailAttachmentController::GetNpadJoyConRailAttachmentState(NpadJoyConRailIndex index) NN_NOEXCEPT
{
    return m_Entries[index].railAttachmentState;
}

Result NpadRailAttachmentController::ActivateRailAttachment(bool activate, NpadJoyConRailIndex index) NN_NOEXCEPT
{
    auto& state = m_Entries[index].railAttachmentState;
    NN_RESULT_THROW_UNLESS(state != NpadJoyConRailAttachmentState_NoDevice, ResultNoExternalBusFoundOnNpad());

    if (activate == true && state == NpadJoyConRailAttachmentState_Deactivated)
    {
        m_Entries[index].railAttachmentState = NpadJoyConRailAttachmentState_Activated;
    }
    else if (activate == false && state == NpadJoyConRailAttachmentState_Activated)
    {
        m_Entries[index].railAttachmentState = NpadJoyConRailAttachmentState_Deactivated;
    }

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
