﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AbstractedGcAdapterPad.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadId.h"
#include "hid_NpadIndicatorPattern.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadInputStateUpdater.h"
#include "hid_NpadStateUtility.h"
#include "hid_NpadStyleDefinitions.h"
#include "hid_Settings.h"

// NpadSystemStyle の共有メモリ更新の処理を有効にする場合
// #define ENABLE_NPADSYSTEMSTYLE

namespace nn { namespace hid { namespace detail {

namespace
{
//!< AbstractedPad の入力状態を指定された NpadState にマージします
void MergeNpadState(NpadCommonStateImpl* pOutState,
                    const AbstractedPadState& inputState,
                    const system::DeviceTypeSet& deviceType,
                    bool isWired,
                    const NpadStyleDefinition& definition) NN_NOEXCEPT;

//!< AbstractedPad の入力状態を指定された本体機能向けの操作スタイルの NpadState に対して入力状態を更新します
void MergeNpadSystemState(NpadCommonStateImpl* pOutState,
    bool* pOutUnusedButtonState,
    const AbstractedPadState& inputState,
    const system::DeviceTypeSet& deviceType,
    bool isWired,
    bool isSingle,
    bool isHorizontal) NN_NOEXCEPT;

//!< AbstractedPad の入力状態からシステムボタンの入力状態を更新します
void MergeSystemButtonState(bool* pOutHomeButton,
                            bool* pOutCaptureButton,
                            const AbstractedPadState& inputState,
                            const system::DeviceTypeSet& deviceType) NN_NOEXCEPT;

//!< AbstractedPad の入力状態をマージします。
void MergeAbstractedPadState(AbstractedPadState* pOutState,
                    const AbstractedPadState& inputState,
                    const system::DeviceTypeSet& deviceType) NN_NOEXCEPT;

bool IsAnalogStickUsed(const AnalogStickState& state) NN_NOEXCEPT
{
    return state.x != 0 && state.y != 0;
}

} // namespace

NpadInputStateUpdater::NpadInputStateUpdater() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
    , m_IsHomeButtonDown(false)
    , m_IgnoreHomeButton(false)
    , m_IsCaptureButtonDown(false)
    , m_UnsupportedButtonStateForNpadSystem(false)
    , m_UnsupportedButtonStateForNpadSystemExt(false)
    , m_IsHandheldButtonPressedOnConsoleMode(false)
    , m_IntegratedStateForInputDetect()
{
    for (auto& commonState : m_CommonStates)
    {
        commonState = NpadCommonStateImpl();
    }
}

NpadInputStateUpdater::~NpadInputStateUpdater() NN_NOEXCEPT { /* 何もしない */ }

void NpadInputStateUpdater::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadInputStateUpdater::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadInputStateUpdater::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());

    m_pDeviceManager = pHolder;
}

void NpadInputStateUpdater::SetCaptureButtonNotifier(SystemButtonNotifier value
                                             ) NN_NOEXCEPT
{
    m_CaptureButtonNotifier = value;
}

void NpadInputStateUpdater::SetHomeButtonNotifier(SystemButtonNotifier value
                                          ) NN_NOEXCEPT
{
    m_HomeButtonNotifier = value;
}

::nn::Result NpadInputStateUpdater::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadInputStateUpdater::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadInputStateUpdater::EnsureNpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                               aruid,
                               [](NpadInputStateUpdater* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            true,
            false);

        that->UpdateSharedMemoryForHomeButtonEnabled(
            &(::nn::util::Get(address->internalState)),
            aruid);
    });

    NN_RESULT_SUCCESS;
}

void NpadInputStateUpdater::UpdateInputStates(bool force) NN_NOEXCEPT
{
    if (!m_NpadActivationCount.IsZero())
    {
        UpdateStates(force);
    }

    this->NotifySystemButtons();
}

void NpadInputStateUpdater::UpdateSharedMemory() NN_NOEXCEPT
{
    this->ProcessSharedMemory([](NpadInputStateUpdater* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            false,
            false);
    });
}

void NpadInputStateUpdater::UpdateSharedMemoryForced() NN_NOEXCEPT
{
    this->ProcessSharedMemory([](NpadInputStateUpdater* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            false,
            true);
    });
}

void NpadInputStateUpdater::UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadInputStateUpdater* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(enablesInput);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            false,
            true);
    });
}

::nn::Result NpadInputStateUpdater::SetHomeButtonEnabledWithSharedMemoryUpdate(bool enabled,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pCommonResourceHolder->GetNpadAppletPolicyManager().SetNpadHomeButtonEnabled(aruid, m_pDeviceManager->GetNpadIdType(), enabled));
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadInputStateUpdater* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(enablesInput);

        that->UpdateSharedMemoryForHomeButtonEnabled(
            &(::nn::util::Get(address->internalState)),
            aruid);
    });
    NN_RESULT_SUCCESS;
}

bool NpadInputStateUpdater::IsNpadSystemExtInputEnable() NN_NOEXCEPT
{
    // 据置モードかつ NpadId::Handheld の場合は常時無操作状態
    if (m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false &&
        m_pDeviceManager->GetNpadIdType() == NpadId::Handheld)
    {
        return false;
    }

    auto& systemState = m_CommonStates[NpadStyleDefinitionIndex_SystemExt];
    return systemState.buttons.IsAnyOn() ||
           m_IsHomeButtonDown ||
           m_IsCaptureButtonDown ||
           IsAnalogStickUsed(systemState.analogStickL) ||
           IsAnalogStickUsed(systemState.analogStickR);
}

bool NpadInputStateUpdater::IsHandheldButtonPressedOnConsoleMode() NN_NOEXCEPT
{
    return m_IsHandheldButtonPressedOnConsoleMode;
}

void NpadInputStateUpdater::UpdateStates(bool forceUpdate) NN_NOEXCEPT
{
    //
    // Pre Process
    //
    for (auto& state : m_CommonStates)
    {
        // State の更新の初期状態をセット
        state.isUpdated = forceUpdate;

        state.buttons.Reset();
        state.attributes.Reset();
        state.analogStickL = AnalogStickState();
        state.analogStickR = AnalogStickState();
    }

    m_GcTriggerState.triggerL = 0;
    m_GcTriggerState.triggerR = 0;

    m_UnsupportedButtonStateForNpadSystemExt = false;
    m_IsHandheldButtonPressedOnConsoleMode = false;
    m_IsHomeButtonDown = false;
    m_IsCaptureButtonDown = false;
    auto oldState = m_IntegratedStateForInputDetect;
    m_IntegratedStateForInputDetect = AbstractedPadState();

    //
    // Processing
    //
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));
    if (count == 0)
    {
        return;
    }

    // 1本持ちかどうか
    bool isSingle = (m_pAbstractedPadHolder->GetJoyAssignmentMode() == NpadJoyAssignmentMode_Single);

    // 横持ちかどうか
    bool isHorizontal = (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal) &&
                        (m_pDeviceManager->GetNpadStyleSet().Test<NpadStyleJoyRight>() || m_pDeviceManager->GetNpadStyleSet().Test<NpadStyleJoyLeft>());

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            auto isUpdated = pPads[i]->ReadAndClearIsSampleReceived();
            auto state = pPads[i]->GetPadState();
            auto deviceType = pPads[i]->GetDeviceType();
            auto isWired = pPads[i]->IsWired();
            if ((deviceType & DeviceTypeMask_HandheldJoyCons).IsAnyOn() &&
                m_pCommonResourceHolder->GetHandheldManager().IsHandheldEnabled() == false)
            {
                // 携帯機操作かつ据置モードのときは無効ボタンの検出のみ行う
                m_UnsupportedButtonStateForNpadSystemExt |= (state.buttons & (~ButtonMask_SystemButtonMask)).IsAnyOn();
                m_IsHandheldButtonPressedOnConsoleMode |= (state.buttons & (~ButtonMask_SystemButtonMask)).IsAnyOn();
            }
            else
            {
                for (int styleIndex = 0; styleIndex < NpadStyleCountMax; ++styleIndex)
                {
                    if (NpadStyleDefinitions[styleIndex].isSystem == true)
                    {
                        MergeNpadSystemState(&m_CommonStates[styleIndex], &m_UnsupportedButtonStateForNpadSystemExt, state, deviceType, isWired, isSingle, isHorizontal);
                    }
                    else
                    {
                        MergeNpadState(&m_CommonStates[styleIndex], state, deviceType, isWired, NpadStyleDefinitions[styleIndex]);
                    }
                    m_CommonStates[styleIndex].isUpdated |= isUpdated;
                    // SIGLO-77779 対応のため、新しいサンプルの受信の有無にかかわらず状態を毎回更新する
                    m_CommonStates[styleIndex].isUpdated = true;
                }
            }

            if (pPads[i]->GetType() == AbstractedPadType_GcAdapter)
            {
                reinterpret_cast<AbstractedGcAdapterPad*>(pPads[i])->GetTrigger(&m_GcTriggerState.triggerL, &m_GcTriggerState.triggerR);
            }

            MergeSystemButtonState(&m_IsHomeButtonDown, &m_IsCaptureButtonDown, state, deviceType);
            MergeAbstractedPadState(&m_IntegratedStateForInputDetect, state, deviceType);
        }
    }

    //
    // Post Process
    //

    // 入力変化の検知
    ::std::lock_guard<decltype(m_pCommonResourceHolder->GetInputDetectorManagerMutex())
                      > locker(m_pCommonResourceHolder->GetInputDetectorManagerMutex());

    if(IsButtonSetChanged(m_IntegratedStateForInputDetect.buttons, oldState.buttons))
    {
        m_pCommonResourceHolder->GetInputDetectorManager().Notify(::nn::hid::system::InputSourceId::Pad::Mask);
    }
    if (IsAnalogStickStateChanged(m_IntegratedStateForInputDetect, oldState))
    {
        m_pCommonResourceHolder->GetInputDetectorManager().Notify(::nn::hid::system::InputSourceId::AnalogStick::Mask);
    }
}

void NpadInputStateUpdater::ProcessSharedMemory(
    void (*processor)(NpadInputStateUpdater* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid,
                entry.flags.Test<AppletResourceFlag::EnablesInput>());
        }
    }
}

void NpadInputStateUpdater::ProcessSharedMemoryForTargetAruid(
    ::nn::applet::AppletResourceUserId aruid,
    void (*processor)(NpadInputStateUpdater* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid,
                entry.flags.Test<AppletResourceFlag::EnablesInput>());

            return;
        }
    }
}

void NpadInputStateUpdater::UpdateSharedMemoryForInputs(NpadSharedMemoryEntry* address,
                                     ::nn::applet::AppletResourceUserId aruid,
                                     bool enablesInput,
                                     bool updateOnlyWithEmpty,
                                     bool forceUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);

    if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetNpadApplicationRevision(aruid) == NpadApplicationRevision_0)
    {
        // リビジョンが初期の場合は互換性維持のために強制更新を常時有効にする
        forceUpdate = true;
    }

    // スタイルを更新
    NpadInternalState& lifo = ::nn::util::Get(address->internalState);
    auto style = m_pDeviceManager->GetAvailableNpadStyleSet(aruid);

    for (int styleIndex = 0; styleIndex < NpadStyleCountMax; ++styleIndex)
    {
        // 適切な State の形に変換しつつ Lifo に追加
        switch (styleIndex)
        {
        case NpadStyleDefinitionIndex_FullKey:
            UpdateSharedMemoryForNpadState<NpadFullKeyState, NpadFullKeyLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_Handheld:
            UpdateSharedMemoryForNpadState<NpadHandheldState, NpadHandheldLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_JoyDual:
            UpdateSharedMemoryForNpadState<NpadJoyDualState, NpadJoyDualLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_JoyLeft:
            UpdateSharedMemoryForNpadState<NpadJoyLeftState, NpadJoyLeftLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_JoyRight:
            UpdateSharedMemoryForNpadState<NpadJoyRightState, NpadJoyRightLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_SystemExt:
            UpdateSharedMemoryForNpadState<system::NpadSystemExtState, NpadSystemExtLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        case NpadStyleDefinitionIndex_Palma:
            UpdateSharedMemoryForNpadState<NpadPalmaState, NpadPalmaLifo>(style, styleIndex, aruid, &lifo, m_CommonStates[styleIndex], enablesInput, updateOnlyWithEmpty, forceUpdate);
            break;
        default:
            ;
        }
    }

    // 無効ボタンの入力取得は行わない
//    UpdateSharedMemoryForUnsupporetedButton(&lifo, enablesInput);
}

template <typename TState, typename TLifo>
void NpadInputStateUpdater::UpdateSharedMemoryForNpadState(NpadStyleSet currentStyle,
                                                      const int& styleIndex,
                                                      ::nn::applet::AppletResourceUserId aruid,
                                                      NpadInternalState* pInternalState,
                                                      const NpadCommonStateImpl& state,
                                                      bool enablesInput,
                                                      bool updateOnlyWithEmpty,
                                                      bool forceUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInternalState);

    // 値が更新されていないもしくは強制更新フラグが無効であれば更新しない
    if (state.isUpdated == false && updateOnlyWithEmpty == false && forceUpdate == false)
    {
        return;
    }

    // Lifo の取得
    TLifo* pLifo;
    pInternalState->GetLifo(&pLifo);

    // 空埋め更新時 Lifo の中が空でない場合は更新しない
    if (pLifo->IsEmpty() == false && updateOnlyWithEmpty == true && forceUpdate == false)
    {
        return;
    }

    // enablesInput が無効の場合は、forceUpdate == true 以外は更新しない
    // == BG 中はNpad の内部状態更新時以外は更新しない
    if (enablesInput == false && forceUpdate == false)
    {
        return;
    }

    // NpadIdType を取得
    auto id = m_pDeviceManager->GetNpadIdType();

    // Style が有効になっているかどうか
    auto& dedicatedStyle = NpadStyleDefinitions[styleIndex].style;
    auto styleAvailable = ((currentStyle & dedicatedStyle).IsAnyOn() == true ||
                           dedicatedStyle == system::NpadStyleSystem::Mask ||
                           dedicatedStyle == system::NpadStyleSystemExt::Mask);
    // GC 対応の暫定コード
    if (currentStyle.Test<NpadStyleGc>() && dedicatedStyle.Test<NpadStyleFullKey>())
    {
        styleAvailable = true;
    }

    // styleUpdate が無効の場合は、forceUpdate == true 以外は更新しない
    // == アプリにとって有効でないスタイルは Npad の内部状態更新時以外は更新しない
    if (styleAvailable == false && forceUpdate == false)
    {
        return;
    }
    // Button Mask の算出
    auto buttonMask = m_pCommonResourceHolder->GetNpadAppletPolicyManager().UpdateButtonMask(aruid, id, styleIndex, state.buttons, enablesInput);
    m_pCommonResourceHolder->GetNpadAppletPolicyManager().SetButtonMask(
        aruid, id, styleIndex, buttonMask);

    // AttributeMask の算出
    NpadAttributesSet attributeMask;
    // Handheld かつ ActivationMode が Dual のときは余計な Attribute をマスクする
    if (currentStyle.Test<NpadStyleHandheld>())
    {
        NpadHandheldActivationMode handheldActivatationMode = NpadHandheldActivationMode_Dual;
        m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetNpadHandheldActivationMode(&handheldActivatationMode, aruid);
        if (handheldActivatationMode == NpadHandheldActivationMode_Dual)
        {
            attributeMask = AttributesMask_DefaultAttributes;
        }
        else
        {
            attributeMask = AttributesMask_JoyAttributes;
        }
    }
    else
    {
        attributeMask = NpadStyleDefinitions[styleIndex].attributesMask;
    }

    // NpadXXXState に変換する
    TState stateToAppend = TState();
    if (styleAvailable)
    {
        // Style が有効な場合は、Attribute を反映する
        stateToAppend.attributes = state.attributes & attributeMask;
        if (enablesInput)
        {
            // FG の場合のみボタンやスティックの状態を反映
            stateToAppend.buttons = state.buttons & buttonMask;
            stateToAppend.analogStickL = state.analogStickL;
            stateToAppend.analogStickR = state.analogStickR;
        }
    }

    if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().PublishSamplingNumber(&stateToAppend.samplingNumber, aruid, id, styleIndex) == true)
    {
        pLifo->Append(stateToAppend);
    }

    // NpadFullKey Lifo をハンドリングするときは一緒に Gc のトリガーも更新
    if (dedicatedStyle.Test<NpadStyleFullKey>())
    {
        auto gcTriggerState = m_GcTriggerState;
        // 入力が無効の場合は無入力に
        if (currentStyle.Test<NpadStyleGc>() == false || enablesInput == false)
        {
            gcTriggerState.triggerR = 0;
            gcTriggerState.triggerR = 0;
        }
        if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().PublishSamplingNumberForGcTrigger(&gcTriggerState.samplingNumber, aruid, id) == true)
        {
            pInternalState->GetGcTriggerLifo()->Append(gcTriggerState);
        }
    }
}

void NpadInputStateUpdater::UpdateSharedMemoryForUnsupporetedButton(NpadInternalState* pInternalState,
                                                                    bool enablesInput) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInternalState);

#ifdef ENABLE_NPADSYSTEMSTYLE
    pInternalState->SetUnsupportedButtonStateOnNpadSystem((enablesInput) ?
        m_UnsupportedButtonStateForNpadSystem :
        false);
#endif
    pInternalState->SetUnsupportedButtonStateOnNpadSystemExt((enablesInput) ?
        m_UnsupportedButtonStateForNpadSystemExt :
        false);
}

void NpadInputStateUpdater::UpdateSharedMemoryForHomeButtonEnabled(NpadInternalState* pInternalState,
                                                                  ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    bool isEnabled = false;
    if (m_pCommonResourceHolder->GetNpadAppletPolicyManager().IsNpadHomeButtonEnabled(&isEnabled, aruid, m_pDeviceManager->GetNpadIdType()).IsSuccess())
    {
        pInternalState->SetHomeButtonEnabled(isEnabled);
    }
}

void NpadInputStateUpdater::NotifySystemButtons() NN_NOEXCEPT
{
    auto isHomeEnabled = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().IsHomeButtonEnabled(m_pDeviceManager->GetNpadIdType());
    if (isHomeEnabled)
    {
        // Home ボタンが離されてたら、Home 無視を解除する
        m_IgnoreHomeButton = m_IsHomeButtonDown ? m_IgnoreHomeButton : false;
        m_HomeButtonNotifier.SetSystemButtonState(m_IsHomeButtonDown && !m_IgnoreHomeButton);
    }
    else
    {
        // Home 無効化中に Home が押された場合、Home が解放されるまで無視する
        m_IgnoreHomeButton = m_IsHomeButtonDown;
        m_HomeButtonNotifier.SetSystemButtonState(false);
    }
    m_CaptureButtonNotifier.SetSystemButtonState(m_IsCaptureButtonDown);
}

namespace {

void MergeNpadState(NpadCommonStateImpl* pOutState,
                    const AbstractedPadState& inputState,
                    const system::DeviceTypeSet& deviceType,
                    bool isWired,
                    const NpadStyleDefinition& definition) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    // 対応しないデバイスの場合はマージしない
    if((deviceType & definition.deviceType).IsAllOff())
    {
        return;
    }

    // ボタンの割り当て
    pOutState->buttons |= ConvertButtonSet(inputState.buttons, deviceType);
    OverwriteAnalogStickStateIfAvailable(
        &(pOutState->analogStickL),
        &(pOutState->analogStickR),
        inputState.analogStickL,
        inputState.analogStickR,
        deviceType);

    pOutState->attributes |= GetNpadAttributesSet(isWired, deviceType);
}

void MergeNpadSystemState(NpadCommonStateImpl* pOutState,
    bool* pOutUnusedButtonState,
    const AbstractedPadState& inputState,
    const system::DeviceTypeSet& deviceType,
    bool isWired,
    bool isSingle,
    bool isHorizontal) NN_NOEXCEPT
{
    // 2本持ちやフルコンなど通常の割り当て
    if (deviceType.Test<system::DeviceType::FullKeyController>() ||
        (deviceType & DeviceTypeMask_HandheldJoyCons).IsAnyOn() ||
        ((deviceType & DeviceTypeMask_JoyCons).IsAnyOn() && isSingle == false))
    {
        // ボタンの割り当て
        pOutState->buttons |= ConvertButtonSet(inputState.buttons, deviceType);
        OverwriteAnalogStickStateIfAvailable(&(pOutState->analogStickL),
            &(pOutState->analogStickR),
            inputState.analogStickL,
            inputState.analogStickR,
            deviceType);
    }
    else if (deviceType.Test<system::DeviceType::JoyConLeft>() &&
        isSingle == true)
    {
        // 左スティックの読み出し
        if (isHorizontal == true)
        {
            pOutState->buttons |= ConvertButtonSetForLeftSingleHorizontal(inputState.buttons);
            *pOutUnusedButtonState |= (inputState.buttons & ButtonMask_UnSupportedOnJoyLeftHorizontal).IsAnyOn();
            pOutState->analogStickL = RotateAnalogSticks(inputState.analogStickL, false);
        }
        else
        {
            pOutState->buttons |= ConvertButtonSetForLeftSingleVertical(inputState.buttons);
            *pOutUnusedButtonState |= (inputState.buttons & ButtonMask_UnSupportedOnJoyLeftVertical).IsAnyOn();
            pOutState->analogStickL = inputState.analogStickL;
        }
    }
    else if (deviceType.Test<system::DeviceType::JoyConRight>() &&
        isSingle == true)
    {
        // 左スティックの読み出し
        if (isHorizontal == true)
        {
            pOutState->buttons |= ConvertButtonSetForRightSingleHorizontal(inputState.buttons);
            *pOutUnusedButtonState |= (inputState.buttons & ButtonMask_UnSupportedOnJoyRightHorizontal).IsAnyOn();
            pOutState->analogStickL = RotateAnalogSticks(inputState.analogStickR, true);
        }
        else
        {
            pOutState->buttons |= ConvertButtonSetForRightSingleVertical(inputState.buttons);
            *pOutUnusedButtonState |= (inputState.buttons & ButtonMask_UnSupportedOnJoyRightVertical).IsAnyOn();
            pOutState->analogStickL = inputState.analogStickR;
        }
    }
    else if (deviceType.Test<system::DeviceType::Palma>())
    {
        pOutState->buttons |= ConvertButtonSetForPalma(inputState.buttons);
        pOutState->analogStickL = inputState.analogStickL;
    }

    pOutState->attributes |= GetNpadAttributesSet(isWired, deviceType);
}

void MergeSystemButtonState(bool* pOutHomeButton,
                            bool* pOutCaptureButton,
                            const AbstractedPadState& inputState,
                            const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    if (deviceType.Test<system::DeviceType::FullKeyController>() ||
        (deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
    {
        if (inputState.buttons.Test<AbstractedPadButton::Home>())
        {
            *pOutHomeButton = true;
        }
    }
    if (deviceType.Test<system::DeviceType::FullKeyController>() ||
        (deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
    {
        if (inputState.buttons.Test<AbstractedPadButton::Shot>())
        {
            *pOutCaptureButton = true;
        }
    }
}

void MergeAbstractedPadState(AbstractedPadState* pOutState,
                             const AbstractedPadState& inputState,
                             const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    // ボタンの割り当て
    pOutState->buttons |= inputState.buttons;
    OverwriteAnalogStickStateIfAvailable(
        &(pOutState->analogStickL),
        &(pOutState->analogStickR),
        inputState.analogStickL,
        inputState.analogStickR,
        deviceType);
}

} // namespace

}}} // namespace nn::hid::detail
