﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/os/os_NativeHandle.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadCommonResourceHolder.h"

namespace nn { namespace hid { namespace detail {

//!< Npad の状態管理を担うマネージャーです。
class NpadDeviceManager final
{
    NN_DISALLOW_COPY(NpadDeviceManager);
    NN_DISALLOW_MOVE(NpadDeviceManager);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の Id
    NpadIdType m_NpadId;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< 現在の デバイスタイプ
    system::NpadDeviceTypeSet m_DeviceType;

    //!< ジョイコンの現在の操作形態
    NpadStyleSet m_StyleSet;

    //!< 本体機能のフッターに表示する操作形態タイプ : Main
    system::NpadSystemExtMainStyle m_SystemExtMainStyle;

    //!< 本体機能のフッターに表示する操作形態タイプ : Sub
    system::NpadSystemExtSubStyle m_SystemExtSubStyle;

    //!<  Abxy 表記が可能かどうか
    bool m_IsAbxyOriented;

    //!< SlSr 表記が可能かどうか
    bool m_IsSlSrOriented;

    //!< Plus ボタンがあるかどうか
    bool m_IsPlusAvailable;

    //!< Minus ボタンがあるかどうか
    bool m_IsMinusAvailable;

    //!< フルコンの色情報
    NpadFullKeyColorState m_FullKeyColor;

    //!< ジョイコンの色情報
    NpadJoyColorState m_JoyColor;

    //!< PlayReport を更新するかどうか
    bool m_IsPlayReportUpdateRequired;

public:
    NpadDeviceManager() NN_NOEXCEPT;

    ~NpadDeviceManager() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadIdType を設定します。
    void SetNpadIdType(NpadIdType id) NN_NOEXCEPT;

    //!< NpadIdType を取得します。
    NpadIdType GetNpadIdType() NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureNpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< デバイスの状態や制御状態など割り当て状態に変更に連動して変更される情報を更新します
    void UpdateControlStates() NN_NOEXCEPT;

    //!< 共有メモリの状態を更新します
    void UpdateSharedMemory() NN_NOEXCEPT;

    //!< 現在の Npad のデバイス種別を取得する
    system::NpadDeviceTypeSet GetNpadDeviceTypeSet() NN_NOEXCEPT;

    //!< 現在の Npad の操作スタイルを取得する
    NpadStyleSet GetNpadStyleSet() NN_NOEXCEPT;

    //!< フルコンの接続インタフェース
    system::InterfaceType GetInterfaceTypeOfFullKey() NN_NOEXCEPT;

    //!< サポートされている NpadStyleSet を踏まえた有効な Style を取得する
    NpadStyleSet GetAvailableNpadStyleSet(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 指定されたデバイスが有効な状態かどうかを検証します。デバイスの接続状態だけでなくアプリにとって、使用できる状態かどうかを検証します
    bool IsStyleActiveForDevice(system::DeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< 対応する NpadStyleSet が初期化済みかどうかを取得する
    bool IsSupportedNpadStyleSetInitialized() NN_NOEXCEPT;

    //!< 本体機能の操作形態の情報を取得します
    void GetSystemExtStyle(system::NpadSystemExtMainStyle* pOutMainStyle,
                           system::NpadSystemExtSubStyle* pOutSubStyle) NN_NOEXCEPT;

    //!< オーバーレイ通知のデータをつくる
    void CreateNotifyControllerInfo(nn::ovln::format::ControllerInfo* pOutValue) NN_NOEXCEPT;

    //!< 接続形式を取得します。
    system::InterfaceType GetInterfaceType() NN_NOEXCEPT;

    //!< 接続形式を取得します。
    void GetInterfaceType(system::InterfaceType* pOutLeftInterface, system::InterfaceType* pOutRightInterface) NN_NOEXCEPT;

    //!< Switch Pro Controller のグリップ色を取得します。
    Result GetGripColor(nn::util::Color4u8Type* pOutLeftGrip, nn::util::Color4u8Type* pOutRightGrip) NN_NOEXCEPT;

    //!< PlayReport として送る情報に更新がないかチェックします。
    void CheckForPlayReport() NN_NOEXCEPT;

private:
    //!< デバイスタイプを更新します。
    void UpdateDeviceType() NN_NOEXCEPT;

    //!< 操作スタイルを更新します。
    void UpdateStyleSet() NN_NOEXCEPT;

    //!< 本体機能のフッターに表示する操作形態情報を更新する
    void UpdateSystemExtStyle() NN_NOEXCEPT;

    //!< ボタンの UI 関連情報を更新します。
    void UpdateButtonOrient() NN_NOEXCEPT;

    //!< コントローラーの色情報を更新します。
    void UpdateControllerColor() NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(NpadDeviceManager* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 指定した Aruid のものについてのみ共有メモリを処理します。
    void ProcessSharedMemoryForTargetAruid(
        ::nn::applet::AppletResourceUserId aruid,
        void (*processor)(NpadDeviceManager* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 割り当て状態の変更に伴って更新される情報を共有メモリに反映します。
    void UpdateSharedMemoryForControlStates(::nn::applet::AppletResourceUserId aruid,
                                            NpadSharedMemoryEntry* address) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの利用状況を更新します
    void UpdateControllerUsageForPlayReport() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
