﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_MouseDriver-os.win.h"
#include "hid_MouseStateUtility.h"
#include "hid_Point.h"
#include "hid_Rectangle.h"
#include "hid_WindowsProcedure-os.win.h"
#include "hid_WindowsRawInput-os.win.h"

namespace nn { namespace hid { namespace detail {

MouseDriver::MouseDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_HasWindow(false)
    , m_HasCursor(false)
    , m_SamplingNumber(0)
    , m_RelativePosition()
    , m_AbsolutePosition()
{
    // 何もしない
}

MouseDriver::~MouseDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result MouseDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultMouseDriverActivationUpperLimitOver());

    // ウィンドウプロシージャをアクティブ化
    ActivateWindowsProcedure();

    if (m_ActivationCount.IsZero())
    {
        // 自身のアクティブ化は新規に要求された場合のみ実施

        // 初期状態ではスクリーンと見做せるウィンドウは無い
        m_HasWindow = false;

        // 初期状態ではカーソルを補足していない
        m_HasCursor = false;

        // カーソルの相対座標を初期化
        m_RelativePosition = Point();

        // カーソルの絶対座標を初期化
        m_AbsolutePosition = Point();
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    // ウィンドウプロシージャを維持
    KeepWindowsProcedureAlive();

    NN_RESULT_SUCCESS;
}

::nn::Result MouseDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultMouseDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    // ウィンドウプロシージャを非アクティブ化
    DeactivateWindowsProcedure();

    NN_RESULT_SUCCESS;
}

void MouseDriver::GetState(MouseState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // ウィンドウプロシージャを維持
    KeepWindowsProcedureAlive();

    // マウスの状態を計算
    pOutValue->samplingNumber = m_SamplingNumber++;
    pOutValue->deltaX = 0;
    pOutValue->deltaY = 0;
    pOutValue->wheelDelta = GetWindowsProcedureWheelDelta();
    pOutValue->buttons = GetWindowsProcedureMouseButtons();
    pOutValue->attributes.Reset();

    // マウスの接続状態を取得
    pOutValue->attributes.Set<MouseAttribute::IsConnected>(
        0 < GetRawInputPhysicalMouseCount());

    // マウスの状態を取得できない限りは上位システムへ移譲
    pOutValue->attributes.Set<MouseAttribute::Transferable>();

    // 極力新しいクライアント領域のサイズを利用するため先にマウスの座標を取得
    Point point = {};
    bool isPointAvailable = GetWindowsProcedureMousePosition(&point);

    Rectangle rect = {};
    if (!GetWindowsProcedureClientAreaSize(&rect))
    {
        // クライアント領域を取得できなかった場合はカーソル座標を (0, 0) に
        pOutValue->x = 0;
        pOutValue->y = 0;

        // スクリーンと見做せるウィンドウ無し
        m_HasWindow = false;
    }
    else if (!isPointAvailable)
    {
        // カーソルの座標を取得できなかった場合は適当な値で代用
        if (!m_HasWindow)
        {
            // 新規にウィンドウを検出した場合は座標を画面中央に
            pOutValue->x = rect.width / 2;
            pOutValue->y = rect.height / 2;
        }
        else
        {
            // それ以外の場合は前回の値を再利用
            pOutValue->x = m_AbsolutePosition.x;
            pOutValue->y = m_AbsolutePosition.y;
        }

        // スクリーンと見做せるウィンドウ有り
        m_HasWindow = true;
    }
    else
    {
        // カーソルを補足できたので上位システムへの移譲を取り止め
        pOutValue->attributes.Reset<MouseAttribute::Transferable>();

        // カーソル座標を計算
        const Point absolutePosition = rect.Clamp(point);
        pOutValue->x = absolutePosition.x;
        pOutValue->y = absolutePosition.y;

        // カーソル座標の縮尺を変域に投射
        *pOutValue = ScaleMousePosition(*pOutValue, rect);

        // スクリーンと見做せるウィンドウ有り
        m_HasWindow = true;
    }

    if (pOutValue->attributes.Test<MouseAttribute::Transferable>())
    {
        // カーソルを補足していない
        m_HasCursor = false;
    }
    else
    {
        if (m_HasCursor)
        {
            // カーソル座標の移動差分を計算
            pOutValue->deltaX = point.x - m_RelativePosition.x;
            pOutValue->deltaY = point.y - m_RelativePosition.y;
        }

        // カーソルを補足している
        m_HasCursor = true;

        // カーソルの相対座標を更新
        m_RelativePosition = point;
    }

    // 最後に観測したカーソルの座標を更新
    m_AbsolutePosition.x = pOutValue->x;
    m_AbsolutePosition.y = pOutValue->y;
}

}}} // namespace nn::hid::detail
