﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_KeyboardDriver-os.win.h"
#include "hid_WindowsKeyboard-os.win.h"

namespace nn { namespace hid { namespace detail {

KeyboardDriver::KeyboardDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_SamplingNumber(0)
{
    // 何もしない
}

KeyboardDriver::~KeyboardDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result KeyboardDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultKeyboardDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_WindowsKeyboard.Reset();
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void KeyboardDriver::GetState(KeyboardState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // キーボードの内部状態を更新
    m_WindowsKeyboard.UpdateConnectionStatus();
    m_WindowsKeyboard.UpdateKeys();
    m_WindowsKeyboard.UpdateModifiers();

    // キーボードの状態を計算
    pOutValue->samplingNumber = m_SamplingNumber++;
    pOutValue->modifiers = m_WindowsKeyboard.GetModifiers();
    pOutValue->attributes.Set<KeyboardAttribute::IsConnected>(
        m_WindowsKeyboard.IsConnected());
    pOutValue->keys = m_WindowsKeyboard.GetKeys();
}

::nn::Result KeyboardDriver::SendLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // ロックキーイベントを処理
    m_WindowsKeyboard.SendLockKeyEvent(value);

    // キーボードの内部状態を更新
    m_WindowsKeyboard.UpdateKeys();
    m_WindowsKeyboard.UpdateModifiers();

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
