﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_JoyXpadManager.h"

namespace nn { namespace hid { namespace detail {

JoyXpadManager::JoyXpadManager() NN_NOEXCEPT
    : m_JoyXpadRightSamplingNumber(0)
    , m_JoyXpadLeftSamplingNumber(0)
    , m_JoyXpadLifoHolder()
    , m_JoyXpadRightState()
    , m_JoyXpadLeftState()
    , m_RightSixAxisSensorSamplingNumber(0)
    , m_LeftSixAxisSensorSamplingNumber(0)
    , m_SixAxisSensorLifoHolder()
{
    // 何もしない
}

JoyXpadManager::~JoyXpadManager() NN_NOEXCEPT { /* 何もしない */ }

::nn::Result JoyXpadManager::ActivateJoyXpad() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsMax(),
                           ResultJoyXpadActivationUpperLimitOver());

    if (m_JoyXpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // LIFO を初期化
        NN_RESULT_DO(m_JoyXpadLifoHolder.Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_JoyXpadLifoHolder.Finalize();
            }
        };

        // LIFO の確保に失敗した場合は失敗
        NN_RESULT_THROW_UNLESS(m_JoyXpadLifoHolder.GetLifo() != nullptr,
                               ResultJoyXpadLifoAllocationFailed());

        // LIFO 上に最低 1 つの入力状態があることを保証
        this->AppendStateToJoyXpadLifo();

        // LIFO を初期化
        NN_RESULT_DO(m_SixAxisSensorLifoHolder.Initialize());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_SixAxisSensorLifoHolder.Finalize();
            }
        };

        // LIFO の確保に失敗した場合は失敗
        NN_RESULT_THROW_UNLESS(m_SixAxisSensorLifoHolder.GetLifo() != nullptr,
                               ResultSixAxisSensorLifoAllocationFailed());

        // LIFO 上に最低 1 つの入力状態があることを保証
        this->AppendStateToSixAxisSensorLifo();

        needsRollback = false;
    }

    ++m_JoyXpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result JoyXpadManager::DeactivateJoyXpad() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultJoyXpadDeactivationLowerLimitOver());

    --m_JoyXpadActivationCount;

    if (m_JoyXpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // LIFO を開放
        m_JoyXpadLifoHolder.Finalize();
        m_SixAxisSensorLifoHolder.Finalize();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result JoyXpadManager::GetJoyXpadLifoHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultJoyXpadNotInitialized());
    *outValue = m_JoyXpadLifoHolder.GetSharedMemoryHandle();
    NN_RESULT_SUCCESS;
}

::nn::Result JoyXpadManager::ActivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SixAxisSensorActivationCount.IsMax(),
                           ResultSixAxisSensorActivationUpperLimitOver());
    ++m_SixAxisSensorActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result JoyXpadManager::DeactivateSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_SixAxisSensorActivationCount.IsZero(),
                           ResultSixAxisSensorDeactivationLowerLimitOver());
    --m_SixAxisSensorActivationCount;
    NN_RESULT_SUCCESS;
}

::nn::Result JoyXpadManager::GetSixAxisSensorLifoHandle(::nn::os::NativeHandle* outValue
                                    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_RESULT_THROW_UNLESS(!m_JoyXpadActivationCount.IsZero(),
                           ResultSixAxisSensorNotInitialized());
    *outValue = m_SixAxisSensorLifoHolder.GetSharedMemoryHandle();
    NN_RESULT_SUCCESS;
}

void JoyXpadManager::Sample() NN_NOEXCEPT
{
    // 何もしない
}

void JoyXpadManager::AppendStateToJoyXpadLifo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_JoyXpadLifoHolder.GetLifo());

    JoyXpadState noInput;
    noInput.buttons.Reset();
    noInput.analogStick.x = 0;
    noInput.analogStick.y = 0;

    // 接続されていないときは無入力状態で値をFIFOに設定

    // サンプリング番号を設定
    noInput.samplingNumber = m_JoyXpadRightSamplingNumber++;
    // LIFO へ書き出し
    m_JoyXpadLifoHolder.GetLifo()->GetRightLifo().Append(noInput);

    // サンプリング番号を設定
    noInput.samplingNumber = m_JoyXpadLeftSamplingNumber++;
    // LIFO へ書き出し
    m_JoyXpadLifoHolder.GetLifo()->GetLeftLifo().Append(noInput);
}

//!< 入力状態を 6軸センサー の LIFO に追加します。
void JoyXpadManager::AppendStateToSixAxisSensorLifo() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_SixAxisSensorLifoHolder.GetLifo());

    // サンプリング番号を設定
    m_RightSixAxisSensorState.samplingNumber = m_RightSixAxisSensorSamplingNumber++;
    // LIFO へ書き出し
    m_SixAxisSensorLifoHolder.GetLifo()->GetRightLifo().Append(m_RightSixAxisSensorState);

    // サンプリング番号を設定
    m_LeftSixAxisSensorState.samplingNumber = m_LeftSixAxisSensorSamplingNumber++;
    // LIFO へ書き出し
    m_SixAxisSensorLifoHolder.GetLifo()->GetLeftLifo().Append(m_LeftSixAxisSensorState);
}


}}} // namespace nn::hid::detail
