﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/ahid/ahid.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/util/util_BitFlagSet.h>

#include "hid_RingFifo.h"

namespace nn { namespace hid { namespace detail {

const int GcAdapterCountMax = 2;           //!< 接続できる Gc コントローラーアダプターの最大数
const int GcPortPerAdapterCountMax = 4;    //!< Gc コントローラーアダプター毎に接続できる Gc コントローラーの最大数

typedef uint64_t GcAdapterIndex;   //!< 接続されている Gc コントローラーアダプターを一意に識別するためのインデックス
typedef uint8_t GcPortIndex;       //!< Gc コントローラーアダプター上のポートのインデックス

/**
 * @brief       Gc コントローラーを一意に識別するための識別子。アダプターの識別子とアダプター上のポートの識別子から構成される
 */
struct GcControllerIndex
{
    GcAdapterIndex adapterIndex;
    GcPortIndex portIndex;
};

/**
 * @brief       コントローラーのボタンの抽象化定義
 */
struct GcControllerButton
{
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<0> A;      //!< コントローラーの a ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<1> B;      //!< コントローラーの b ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<2> X;      //!< コントローラーの x ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<3> Y;      //!< コントローラーの y ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<4> L;      //!< コントローラーの L ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<5> R;      //!< コントローラーの R ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<6> Z;     //!< コントローラーの Z ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<7> Start; //!< コントローラーの Start ボタン
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<8> Left;  //!< コントローラーの 方向ボタン 左
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<9> Up;    //!< コントローラーの 方向ボタン 上
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<10> Right; //!< コントローラーの 方向ボタン 右
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<11> Down;  //!< コントローラーの 方向ボタン 下
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<12> StickLLeft;   //!< 左スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<13> StickLUp;     //!< 左スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<14> StickLRight;  //!< 左スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<15> StickLDown;   //!< 左スティックによる十字ボタンエミュレーション 下
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<16> StickRLeft;   //!< 右スティックによる十字ボタンエミュレーション 左
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<17> StickRUp;     //!< 右スティックによる十字ボタンエミュレーション 上
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<18> StickRRight;  //!< 右スティックによる十字ボタンエミュレーション 右
    typedef ::nn::util::BitFlagSet<32, GcControllerButton>::Flag<19> StickRDown;   //!< 右スティックによる十字ボタンエミュレーション 下
};

/**
 * @brief       デジタルボタンの集合を扱う型です。
 */
typedef ::nn::util::BitFlagSet<32, GcControllerButton> GcControllerButtonSet;

/**
 * @brief       ボタン, アナログスティック
 */
struct GcControllerState
{
    int64_t sampleNumber;             //!< サンプル番号
    GcControllerButtonSet buttons;   //!< デジタルボタンの状態です。
    AnalogStickState analogStickL;    //!< 左アナログスティックの状態です。
    AnalogStickState analogStickR;    //!< 右アナログスティックの状態です。
    int32_t triggerL;                //!< L トリガーの状態です。
    int32_t triggerR;                //!< R トリガーの状態です。
};

/**
 * @brief       モーターの状態の定義です
 */
enum GcControllerMotorState : uint8_t
{
    GcControllerMotorState_Off   = 0x00,        //!< モーターを停止します
    GcControllerMotorState_On    = 0x01,        //!< モーターを駆動します
    GcControllerMotorState_Brake = 0x10,        //!< モーターを即座に停止します
};

/**
 * @brief       Gc コントローラーのポートに接続されたデバイスの種類です
 */
enum GcPortDeviceType : int8_t
{
    GcPortDeviceType_None         = 0x00,        //!< デバイスが接続されていません
    GcPortDeviceType_GcController = 0x01,        //!< Gc コントローラー
    GcPortDeviceType_WaveBird     = 0x02,        //!< ウェーブバード
    GcPortDeviceType_Unknown      = -0x01,       //!< 未知のデバイス
};


/**
 * @brief       Gc コントローラーアダプターのリスト
 */
struct GcAdapterList
{
    GcAdapterIndex indexes[GcAdapterCountMax];      //!< アダプターのインデックス
    int count;                                      //!< アダプターの数
};

//!< GC コントローラータップ用のドライバ
class GcAdapterDriver final
{
    NN_DISALLOW_COPY(GcAdapterDriver);
    NN_DISALLOW_MOVE(GcAdapterDriver);
public:
    GcAdapterDriver() NN_NOEXCEPT;

    //!< デバイスが GC コントローラーアダプターかどうか評価する
    static bool IsGcAdapter(int vid, int pid) NN_NOEXCEPT;

    //!< GcAdapterDriver を有効にする
    void Enable() NN_NOEXCEPT;

    //!< 新規にデバイスを登録する。
    bool Attach(size_t port, nn::ahid::Ahid& ahid) NN_NOEXCEPT;

    //!< 登録済みのデバイスを切断する。
    void Detach(size_t port) NN_NOEXCEPT;

    //!< Input Report をパースします。
    void ParseInputReport(size_t port, const uint8_t* pReport, size_t length) NN_NOEXCEPT;

    //!< Output Report を送信します。
    bool SendOutputReport(size_t port) NN_NOEXCEPT;

    //!< 接続されている Gc コントローラーアダプターを列挙する
    GcAdapterList ListAdapters() NN_NOEXCEPT;

    //!< デバイスの状態を取得します。
    bool GetState(GcControllerState* pOutValue, const GcControllerIndex& index) NN_NOEXCEPT;

    //!< モーターの制御を行う
    Result ControlMotor(const GcControllerIndex& index, GcControllerMotorState motor) NN_NOEXCEPT;

    //!< 接続時のモーターの制御を行う
    Result ControlMotorOnConnect(const GcControllerIndex& index) NN_NOEXCEPT;

    //!< モーターの状態を取得する
    GcControllerMotorState GetMotorState(const GcControllerIndex& index) NN_NOEXCEPT;

    //!< 左スティックのキャリブレーション値を取得します
    Result GetLeftAnalogStickCalibration(AnalogStickState* pOutCenter,
                                         int16_t* pOutOriginPlay,
                                         int16_t* pOutRange,
                                         const GcControllerIndex& index) NN_NOEXCEPT;

    //!< 右スティックのキャリブレーション値を取得します
    Result GetRightAnalogStickCalibration(AnalogStickState* pOutCenter,
                                          int16_t* pOutOriginPlay,
                                          int16_t* pOutRange,
                                          const GcControllerIndex& index) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
