﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BitPack.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadGc.h>
#include <nn/hid/system/hid_Npad.h>
#include "hid_ExternalBusHandle.h"
#include "hid_NpadId.h"
#include "hid_ExternalBusImpl.h"

#ifdef NN_BUILD_CONFIG_OS_HORIZON
#include "hid_ResourceManager-os.horizon.h"
#endif

#ifdef NN_BUILD_CONFIG_OS_WIN
#include "hid_ResourceManager-os.win.h"
#endif

namespace
{

nn::Result SetInternalXcdHandle(nn::hid::NpadIdType npadId, nn::hid::detail::NpadJoyConRailIndex index) NN_NOEXCEPT
{
    auto state = nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetNpadJoyConRailAttachmentState(npadId, index);
    if (state != nn::hid::detail::NpadJoyConRailAttachmentState_NoDevice)
    {
        nn::xcd::DeviceHandle xcdHandle;
        nn::hid::detail::ExternalBusXcdDriver* pExternalBusDriver;
        NN_RESULT_DO(nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetXcdHandleForNpadWithRailAttachment(&xcdHandle, npadId, index));
        pExternalBusDriver = nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetExternalXcdDriver(npadId, index);
        NN_RESULT_DO(pExternalBusDriver->SetXcdDeviceHandle(xcdHandle));
        NN_RESULT_SUCCESS;
    }
    return nn::hid::ResultNoExternalBusFoundOnNpad();
}

typedef uint32_t ExternalBusHandleUpper32Bit;
typedef uint32_t ExternalBusHandleLower32Bit;

}

namespace nn { namespace hid { namespace detail {

//<! ExternalBusHandle の上位 32 ビットのビットフィールド定義です。
struct ExternalBusHandleUpper32BitField final
{
    //!< Internal Index (内部で使用する拡張バス機能の index)
    typedef ::nn::util::BitPack32::Field<0, 8, uint8_t> InternalIndex;

    //!< プレイヤー番号
    typedef ::nn::util::BitPack32::Field<8, 8, uint8_t> PlayerNumber;

    //!< BusTypeId
    typedef ::nn::util::BitPack32::Field<16, 8, uint8_t> ExternalBusTypeId;

    //!< Valid Flag
    typedef ::nn::util::BitPack32::Field<24, 1, uint8_t> ValidFlag;
};

nn::Result CheckValidExternalBusHandle(ExternalBusHandle handle) NN_NOEXCEPT
{
    nn::hid::detail::AbstractedPadId abstractedPadId;
    auto npadId = GetExternalBusHandlePlayerNumber(handle);
    nn::hid::detail::NpadJoyConRailIndex npadJoyConRailIndex;
    NN_RESULT_DO(GetNpadJoyConRailIndexFromExternalBusHandle(&npadJoyConRailIndex, handle));

    // Valid Flag をチェック
    uint32_t upperHandle = static_cast<uint32_t>(handle._storage >> 32);
    ::nn::util::BitPack32 bitPack = { upperHandle };
    if (bitPack.Get<ExternalBusHandleUpper32BitField::ValidFlag>() != 0x01)
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }

    // Abstracted Pad ID で判定する
    auto result = nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetAbstractedPadIdForRailAttachment(&abstractedPadId, npadId, npadJoyConRailIndex);
    if (result.IsFailure())
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }

    if (static_cast<uint32_t>(handle._storage) != static_cast<uint32_t>(abstractedPadId.value))
    {
        NN_SDK_LOG("handle = %d\n", handle._storage);
        NN_SDK_LOG("abstractedPadId = %d\n", abstractedPadId.value);
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    NN_RESULT_SUCCESS;
}

ExternalBusHandleUpper32Bit MakeExternalBusHandleUpper32Bit(int playerNumber, int externalBusTypeId) NN_NOEXCEPT
{
    auto bitPack = ::nn::util::BitPack32();
    bitPack.Set<ExternalBusHandleUpper32BitField::PlayerNumber>(static_cast<int8_t>(playerNumber));
    bitPack.Set<ExternalBusHandleUpper32BitField::ExternalBusTypeId>(static_cast<int8_t>(externalBusTypeId));

    // 現状、 Joy-Con だけ対応しているのでそれ以外はすべて index は playerNumber にしておく
    if (externalBusTypeId == ExternalBusType_LeftJoyRail)
    {
        bitPack.Set<ExternalBusHandleUpper32BitField::InternalIndex>(static_cast<uint8_t>(playerNumber * 2 + 0));
    }
    else if (externalBusTypeId == ExternalBusType_RightJoyRail)
    {
        bitPack.Set<ExternalBusHandleUpper32BitField::InternalIndex>(static_cast<uint8_t>(playerNumber * 2 + 1));
    }
    else
    {
        bitPack.Set<ExternalBusHandleUpper32BitField::InternalIndex>(static_cast<uint8_t>(playerNumber));
    }

    bitPack.Set<ExternalBusHandleUpper32BitField::ValidFlag>(1);

    ExternalBusHandleUpper32Bit handle = { bitPack.storage };
    return handle;
}

nn::Result MakeExternalBusHandleLower32bit(ExternalBusHandleLower32Bit* pOutHandle, int npadId, int externalBusTypeId)
{
    nn::hid::detail::AbstractedPadId abstractedPadId;

    // 現在は Joy-Con 前提の実装 (他のコントローラに対応したら、ここで実装分岐)
    NN_RESULT_DO(nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetAbstractedPadIdForRailAttachment(&abstractedPadId, npadId, static_cast<NpadJoyConRailIndex>(externalBusTypeId)));
    *pOutHandle = static_cast<ExternalBusHandleLower32Bit>(abstractedPadId.value);
    NN_RESULT_SUCCESS;
}

nn::Result MakeExternalBusHandle(ExternalBusHandle* pOutHandle, int npadId, ExternalBusType busType) NN_NOEXCEPT
{
    ExternalBusHandleUpper32Bit upper;
    ExternalBusHandleLower32Bit lower;
    NN_RESULT_DO(MakeExternalBusHandleLower32bit(&lower, npadId, busType));
    upper = MakeExternalBusHandleUpper32Bit( npadId, busType);

    pOutHandle->_storage = static_cast<uint64_t>(upper) << 32 | lower;
    NN_RESULT_SUCCESS;
}

bool GetExternalBusHandle(ExternalBusHandle* pOutHandle, const nn::hid::NpadIdType &id, ExternalBusType busType) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutHandle);
    if (detail::VerifyValidNpadId(id).IsFailure())
    {
        return false;
    }


    auto handleResult = MakeExternalBusHandle(pOutHandle, static_cast<int>(id), busType);
    if (handleResult.IsSuccess())
    {
        // TODO : JoyCon じゃないものが現れたら要修正
        auto xcdResult = SetInternalXcdHandle(id, static_cast<NpadJoyConRailIndex>(busType));
        if (xcdResult.IsSuccess())
        {
            return true;
        }
    }

    return false;
}

int GetExternalBusHandlePlayerNumber(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    uint32_t upperHandle = static_cast<uint32_t>(handle._storage >> 32);
    ::nn::util::BitPack32 bitPack = { upperHandle };
    return bitPack.Get<ExternalBusHandleUpper32BitField::PlayerNumber>();
}

int GetExternalBusHandleBusTypeId(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    uint32_t upperHandle = static_cast<uint32_t>(handle._storage >> 32);
    ::nn::util::BitPack32 bitPack = { upperHandle };
    return bitPack.Get<ExternalBusHandleUpper32BitField::ExternalBusTypeId>();
}

int GetExternalBusHandleInternalIndex(const ExternalBusHandle& handle) NN_NOEXCEPT
{
    uint32_t upperHandle = static_cast<uint32_t>(handle._storage >> 32);
    ::nn::util::BitPack32 bitPack = { upperHandle };
    return bitPack.Get<ExternalBusHandleUpper32BitField::InternalIndex>();
}

nn::Result GetNpadJoyConRailIndexFromExternalBusHandle(NpadJoyConRailIndex* pOutRailIndex, const ExternalBusHandle& handle) NN_NOEXCEPT
{
    auto busTypeId = GetExternalBusHandleBusTypeId(handle);
    if (busTypeId == ExternalBusType_LeftJoyRail)
    {
        *pOutRailIndex = nn::hid::detail::NpadJoyConRailIndex_Left;
    }
    else if (busTypeId == ExternalBusType_RightJoyRail)
    {
        *pOutRailIndex = nn::hid::detail::NpadJoyConRailIndex_Right;;
    }
    else
    {
        return nn::hid::ResultExternalBusDeviceInvalidHandle();
    }
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
