﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>

#include "hid_DebugPadDriver-os.win.h"
#include "hid_DebugPadMappingManager.h"
#include "hid_DebugPadStateUtility.h"
#include "hid_HidShellPortFile-os.win.h"
#include "hid_WindowsGenericPad-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< DebugPad のデジタルボタンの押下状態を返します。
DebugPadButtonSet GetButtons(const WindowsGenericPad& genericPad) NN_NOEXCEPT
{
    WindowsGenericPadButtonSet buttons = genericPad.GetButtons();
    DebugPadButtonSet value = {};
    value.Set<DebugPadButton::A>(
        buttons.Test<WindowsGenericPadButton::B>());
    value.Set<DebugPadButton::B>(
        buttons.Test<WindowsGenericPadButton::A>());
    value.Set<DebugPadButton::X>(
        buttons.Test<WindowsGenericPadButton::Y>());
    value.Set<DebugPadButton::Y>(
        buttons.Test<WindowsGenericPadButton::X>());
    value.Set<DebugPadButton::L>(
        buttons.Test<WindowsGenericPadButton::L>());
    value.Set<DebugPadButton::R>(
        buttons.Test<WindowsGenericPadButton::R>());
    value.Set<DebugPadButton::ZL>(
        buttons.Test<WindowsGenericPadButton::TriggerL>());
    value.Set<DebugPadButton::ZR>(
        buttons.Test<WindowsGenericPadButton::TriggerR>());
    value.Set<DebugPadButton::Start>(
        buttons.Test<WindowsGenericPadButton::Start>());
    value.Set<DebugPadButton::Select>(
        buttons.Test<WindowsGenericPadButton::Back>());
    value.Set<DebugPadButton::Left>(
        buttons.Test<WindowsGenericPadButton::Left>());
    value.Set<DebugPadButton::Up>(
        buttons.Test<WindowsGenericPadButton::Up>());
    value.Set<DebugPadButton::Right>(
        buttons.Test<WindowsGenericPadButton::Right>());
    value.Set<DebugPadButton::Down>(
        buttons.Test<WindowsGenericPadButton::Down>());
    return value;
}

//!< 指定の割り当てに基づいた DebugPad のデジタルボタンの押下状態を返します。
DebugPadButtonSet GetButtons(
    const WindowsGenericPad& genericPad,
    const ::nn::settings::DebugPadGenericPadMap& map) NN_NOEXCEPT
{
    DebugPadButtonSet value = {};
    value.Set<DebugPadButton::A>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonA)));
    value.Set<DebugPadButton::B>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonB)));
    value.Set<DebugPadButton::X>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonX)));
    value.Set<DebugPadButton::Y>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonY)));
    value.Set<DebugPadButton::L>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonL)));
    value.Set<DebugPadButton::R>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonR)));
    value.Set<DebugPadButton::ZL>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZL)));
    value.Set<DebugPadButton::ZR>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonZR)));
    value.Set<DebugPadButton::Start>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonStart)));
    value.Set<DebugPadButton::Select>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonSelect)));
    value.Set<DebugPadButton::Left>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonLeft)));
    value.Set<DebugPadButton::Up>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonUp)));
    value.Set<DebugPadButton::Right>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonRight)));
    value.Set<DebugPadButton::Down>(
        genericPad.IsButtonDown(
            static_cast<::nn::settings::GenericPadButton>(map.buttonDown)));
    return value;
}

//!< DebugPad の HidShell 向けに共有される自動操作状態を取得します。
void GetDebugPadFileState(
    DebugPadState* pOutValue, const DebugPadState& state) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    bool isAutoPilotEnabled = false;
    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetHidShellDebugPadState(&isAutoPilotEnabled, &autoPilotState));
    if (!isAutoPilotEnabled)
    {
        *pOutValue = state;
    }
    else
    {
        pOutValue->attributes = autoPilotState.attributes;
        pOutValue->buttons = autoPilotState.buttons;
        pOutValue->analogStickL = autoPilotState.analogStickL;
        pOutValue->analogStickR = autoPilotState.analogStickR;
    }
}

//!< DebugPad の HidShell 向けに共有される入力状態を更新します。
void SetDebugPadFileState(const DebugPadState& state) NN_NOEXCEPT
{
    ::nn::hid::debug::DebugPadAutoPilotState autoPilotState = {};
    autoPilotState.attributes = state.attributes;
    autoPilotState.buttons = state.buttons;
    autoPilotState.analogStickL = state.analogStickL;
    autoPilotState.analogStickR = state.analogStickR;

    NN_ABORT_UNLESS_RESULT_SUCCESS(SetHidShellDebugPadState(autoPilotState));
}

} // namespace

DebugPadDriver::DebugPadDriver() NN_NOEXCEPT
    : m_SamplingNumber(0)
    , m_pMappingManager(nullptr)
{
    // 何もしない
}

DebugPadDriver::~DebugPadDriver() NN_NOEXCEPT
{
    // 何もしない
}

void DebugPadDriver::SetMappingManager(DebugPadMappingManager* pMappingManager
                                       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMappingManager);
    m_pMappingManager = pMappingManager;
}

::nn::Result DebugPadDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultDebugPadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_WindowsGenericPad.Reset();

        m_WindowsKeyboard.Reset();
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result DebugPadDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultDebugPadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void DebugPadDriver::GetState(DebugPadState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_SDK_REQUIRES_NOT_NULL(m_pMappingManager);

    // 無入力状態に初期化
    *pOutValue = DebugPadState();
    pOutValue->samplingNumber = m_SamplingNumber++;

    // 汎用ゲームパッドとのマッピングを取得
    m_pMappingManager->GetDebugPadGenericPadMap(&m_GenericPadMap);

    if (!m_pMappingManager->IsValidDebugPadGenericPadMap(m_GenericPadMap))
    {
        // マッピングされた識別子が無効ならば内部状態をリセット
        m_WindowsGenericPad.Reset();
    }
    else
    {
        // マッピングされた識別子が有効ならば対象とする汎用コントローラに設定
        m_WindowsGenericPad.SetGenericPadId(m_GenericPadMap.genericPadId);

        // 汎用コントローラの内部状態を更新
        m_WindowsGenericPad.Update();

        // DebugPad のアナログスティックのデッドゾーンを計算
        const int deadZoneCount =
            CpadAnalogStickDeadZoneCount * AnalogStickMax /
            CpadAnalogStickTheoreticalCount;

        // DebugPad のアナログスティックの高さ（垂直方向）を計算
        const int horizontalCount =
            CpadAnalogStickHorizontalCount * AnalogStickMax /
            CpadAnalogStickTheoreticalCount;

        // DebugPad のアナログスティックの高さ（斜め方向）を計算
        const int diagonalCount =
            CpadAnalogStickDiagonalCount * AnalogStickMax /
            CpadAnalogStickTheoreticalCount;

        // DebugPad の入力状態を決定
        pOutValue->attributes.Set<DebugPadAttribute::IsConnected>(
            m_WindowsGenericPad.IsDetected());
        if (m_WindowsGenericPad.IsXInput())
        {
            pOutValue->buttons = RestrictDebugPadButtons(
                GetButtons(m_WindowsGenericPad));
            pOutValue->analogStickR = LimitDebugPadAnalogStick(
                m_WindowsGenericPad.GetStickR(),
                0, horizontalCount, diagonalCount);
            pOutValue->analogStickL = LimitDebugPadAnalogStick(
                m_WindowsGenericPad.GetStickL(),
                0, horizontalCount, diagonalCount);
        }
        else
        {
            pOutValue->buttons = RestrictDebugPadButtons(
                GetButtons(m_WindowsGenericPad, m_GenericPadMap));
            pOutValue->analogStickR = LimitDebugPadAnalogStick(
                m_WindowsGenericPad.GetStickR(m_GenericPadMap),
                deadZoneCount, horizontalCount, diagonalCount);
            pOutValue->analogStickL = LimitDebugPadAnalogStick(
                m_WindowsGenericPad.GetStickL(m_GenericPadMap),
                deadZoneCount, horizontalCount, diagonalCount);
        }
    }

    // キーボードとのマッピングを取得
    m_pMappingManager->GetDebugPadKeyboardMap(&m_KeyboardMap);

    if (!m_pMappingManager->IsValidDebugPadKeyboardMap(m_KeyboardMap))
    {
        // マッピングが無効ならば内部状態をリセット
        m_WindowsKeyboard.Reset();
    }
    else
    {
        // Keyboard のキーの状態について内部状態を更新
        m_WindowsKeyboard.UpdateKeys();

        DebugPadButtonSet buttons = RestrictDebugPadButtons(
            m_WindowsKeyboard.GetButtons(m_KeyboardMap));

        // マッピングが有効ならば接続状態
        pOutValue->attributes.Set<DebugPadAttribute::IsConnected>();

        // キーボードによるエミュレーション結果をマージ
        pOutValue->buttons = MergeDebugPadButtons(pOutValue->buttons, buttons);
    }

    // DebugPad の HidShell 向けに共有される入力状態を更新
    SetDebugPadFileState(*pOutValue);

    // DebugPad の HidShell 向けに共有される自動操作状態を取得
    GetDebugPadFileState(pOutValue, *pOutValue);
}

}}} // namespace nn::hid::detail
