﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>

#include "hid_ActivationCount.h"

#if defined(NN_BUILD_CONFIG_COMPILER_VC)

// C4351: 配列メンバの初期化が規定値で行われる旨の警告を抑止
#pragma warning(push)
#pragma warning(disable: 4351)

#endif  // if defined(NN_BUILD_CONFIG_COMPILER_VC)

namespace nn { namespace hid { namespace detail {

enum : size_t
{
    FirmwareImagePathLengthMax = 32     //!< ファームウェアイメージのパス最大長
};

enum DeviceTypeIndex
{
    DeviceTypeIndex_JoyLeft,            //!< Joy-Con (L)
    DeviceTypeIndex_JoyRight,           //!< Joy-Con (R)
    DeviceTypeIndex_FullKey,            //!< Switch Pro Controller

    DeviceTypeIndex_Max                 //!< デバイスインデックスの上限
};

/**
 * @brief   システムデータ内のコントローラーファームウェア関連にアクセスするためのクラス
 */
class ControllerFirmwareAccessor final
{
    NN_DISALLOW_COPY(ControllerFirmwareAccessor);
    NN_DISALLOW_MOVE(ControllerFirmwareAccessor);

public:
    ControllerFirmwareAccessor(
        const nn::ncm::SystemDataId& systemDataId,
        const char* mountName) NN_NOEXCEPT :
        m_Mutex(true),
        m_MountCount(),
        m_SystemDataId(systemDataId),
        m_pSystemDataMountName(mountName),
        m_SystemDataCache(),
        m_ReadBuffer(),
        m_IsImageInfoLoaded(false),
        m_ImageInfos(),
        m_ExpectVersions()
    {
        NN_SDK_REQUIRES_NOT_NULL(mountName);
    }

    /**
     * @brief   ファームウェア情報をクリア
     */
    void ClearFirmwareInfo() NN_NOEXCEPT;

    /**
     * @brief   システムデータをマウント
     */
    nn::Result Mount() NN_NOEXCEPT;

    /**
     * @brief   システムデータのマウントを解除
     */
    void Unmount() NN_NOEXCEPT;

    /**
     * @brief   更新先のファームウェアバージョンを取得
     */
    nn::Result GetDestinationVersion(
        system::FirmwareVersion* pOutVersion,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief   Hotfix 判定用のファームウェアバージョンを取得
     */
    nn::Result GetExpectVersion(
        system::FirmwareVersion* pOutVersion,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief   Bluetooth ファームウェアのファイルパスを取得
     */
    nn::Result GetBluetoothImageFilePath(
        char* pOutImagePath,
        size_t imagePathLengthMax,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief   MCU ファームウェアのファイルパスを取得
     */
    nn::Result GetMcuImageFilePath(
        char* pOutImagePath,
        size_t imagePathLengthMax,
        nn::xcd::DeviceHandle handle,
        bool isFullUpdate) NN_NOEXCEPT;

private:
    /**
     * @brief   デバイス毎のファームウェアイメージ情報
     */
    struct FirmwareImageInfo
    {
        system::FirmwareVersion version;
        char                    bluetoothPath[FirmwareImagePathLengthMax];
        char                    mcuPath[FirmwareImagePathLengthMax];

        /**
         * @brief   メンバの初期化
         */
        void Clear() NN_NOEXCEPT
        {
            std::memset(&version, 0, sizeof(version));
            bluetoothPath[0] = '\0';
            mcuPath[0]       = '\0';
        }
    };

private:
    /**
     * @brief   システムデータ内のファームウェア情報ファイルを読み込む
     */
    nn::Result ReadFirmwareInfoFile(
        char* pOutReadBuffer,
        size_t bufferSize,
        const char* filename) NN_NOEXCEPT;

    /**
     * @brief   システムデータ内のファームウェア情報を解析
     */
    nn::Result ParseFirmwareInfoInSystemData() NN_NOEXCEPT;

    /**
     * @brief   システムデータ内の Hotfix 用バージョン情報を解析
     */
    nn::Result ParseExpectVersionInSystemData() NN_NOEXCEPT;

    /**
     * @brief   指定したデバイスに対応する DeviceTypeIndex を取得
     */
    nn::Result GetDeviceTypeIndex(
        DeviceTypeIndex* pOutType,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    /**
     * @brief   指定したデバイスタイプに対応するファームウェア情報を取得
     */
    nn::Result GetFirmwareImageInfo(
        FirmwareImageInfo* pOutInfo,
        nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

private:
    mutable nn::os::Mutex m_Mutex;

    /**
     * @brief   マウントされた回数
     */
    ActivationCount m_MountCount;

    /**
     * @brief   参照するシステムデータの ID
     */
    nn::ncm::SystemDataId m_SystemDataId;

    /**
     * @brief   システムデータのマウント名
     */
    const char* m_pSystemDataMountName;

    /**
     * @brief   システムデータ用キャッシュ
     */
    char m_SystemDataCache[1 * 1024];

    /**
     * @brief   ファームウェア情報解析用の読み込みバッファ
     */
    char m_ReadBuffer[1 * 1024];

    /**
     * @brief   ファームウェア情報を読み込み済みか
     */
    bool m_IsImageInfoLoaded;

    /**
     * @brief   システムデータ内のファームウェア情報
     */
    FirmwareImageInfo m_ImageInfos[DeviceTypeIndex_Max];

    /**
     * @brief   Hotfix 判定のためのバージョン情報
     */
    system::FirmwareVersion m_ExpectVersions[DeviceTypeIndex_Max];
};

}}} // namespace nn::hid::detail

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
#pragma warning(pop)
#endif  // if defined(NN_BUILD_CONFIG_COMPILER_VC)
