﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <cstdio>
#include <cstring>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/xcd/xcd_Device.h>
#include <nn/xcd/xcd_Result.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "hid_ControllerFirmwareAccessor.h"

#if 0

#ifdef NN_BUILD_CONFIG_COMPILER_VC
#define NN_HID_DETAIL_SYSDATA_TRACE(...)            NN_DETAIL_HID_TRACE("SystemData: " ##__VA_ARGS__)
#else
#define NN_HID_DETAIL_SYSDATA_TRACE(format, ...)    NN_DETAIL_HID_TRACE("SystemData: " format, ##__VA_ARGS__)
#endif  // ifdef NN_BUILD_CONFIG_COMPILER_VC

#else

#define NN_HID_DETAIL_SYSDATA_TRACE(...)            static_cast<void>(0)

#endif

namespace nn { namespace hid { namespace detail {

namespace
{

// ファームウェア情報のファイルパス
const char FirmwareInfoFilePath[] = "FirmwareInfo.csv";

// Hotfix 判定用バージョン情報のファイルパス
const char ExpectVersionFilePath[] = "ExpectVersionInfo.csv";

// IAP 入りイメージの接尾辞
const char WithIapSuffix[] = "_iap.bin";

// デバイス種類を示すトークン
const char TokenJoyLeft[]  = "JoyLeft";
const char TokenJoyRight[] = "JoyRight";
const char TokenFullKey[]  = "FullKey";

// FW タイプを示すトークン
const char TokenBluetooth[] = "BT";
const char TokenMcu[]       = "MCU";

// FW イメージの種類
enum class ImageType
{
    Bluetooth,
    Mcu
};

// 文字列をトークンに分割
char* TokenizeString(char* token, const char* delimiter, char** context) NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_COMPILER_VC
    return strtok_s(token, delimiter, context);
#else
    return strtok_r(token, delimiter, context);
#endif  // ifdef NN_BUILD_CONFIG_COMPILER_VC
}

// 同一文字列か判定
bool IsSameString(const char* string1, const char* string2) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(string1);
    NN_SDK_REQUIRES_NOT_NULL(string2);

    const int MaxLength = 255;  // 仕様上到達しない長さ

    return nn::util::Strncmp(string1, string2, MaxLength) == 0 &&
        nn::util::Strnlen(string1, MaxLength) == nn::util::Strnlen(string2, MaxLength);
}

// ファームウェア情報を解析するクラスのベース
class FirmwareInfoParserBase
{
    NN_DISALLOW_COPY(FirmwareInfoParserBase);
    NN_DISALLOW_MOVE(FirmwareInfoParserBase);

public:
    explicit FirmwareInfoParserBase(char* pString) NN_NOEXCEPT :
        m_InfoString(pString),
        m_pCurrentLine(nullptr),
        m_pNextLineStart(pString)
    {
        NN_SDK_REQUIRES_NOT_NULL(pString);
        GoNextLine();
    }

    // 次の行に移動
    void GoNextLine() NN_NOEXCEPT
    {
        if (IsEnd())
        {
            return;
        }

        m_pCurrentLine = m_pNextLineStart;

        // 改行位置を探す
        char* newLine = std::strchr(m_pNextLineStart, '\r');
        if (newLine == nullptr)
        {
            newLine = std::strchr(m_pNextLineStart, '\n');
        }

        // 改行がなければ終端
        if (newLine == nullptr)
        {
            m_pNextLineStart = nullptr;
            NN_HID_DETAIL_SYSDATA_TRACE("%s: CurrentLine=%s\n", NN_CURRENT_FUNCTION_NAME, m_pCurrentLine);
            return;
        }

        // 改行と空白を消しつつ飛ばす
        while (std::isspace(*newLine))
        {
            *newLine = '\0';
            newLine++;
        }

        m_pNextLineStart = newLine;
        NN_HID_DETAIL_SYSDATA_TRACE("%s: CurrentLine=%s\n", NN_CURRENT_FUNCTION_NAME, m_pCurrentLine);
    }

    // 末尾に到達したか
    bool IsEnd() const NN_NOEXCEPT
    {
        return m_pNextLineStart == nullptr;
    }

protected:
    // 現在の行を取得
    char* GetCurrentLine() NN_NOEXCEPT
    {
        return m_pCurrentLine;
    }

    // トークンからデバイスタイプを取得
    bool ParseDeviceType(DeviceTypeIndex* pOutDeviceType, const char* pToken) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutDeviceType);
        NN_SDK_REQUIRES_NOT_NULL(pToken);

        if (IsSameString(pToken, TokenJoyLeft))
        {
            *pOutDeviceType = DeviceTypeIndex_JoyLeft;
        }
        else if (IsSameString(pToken, TokenJoyRight))
        {
            *pOutDeviceType = DeviceTypeIndex_JoyRight;
        }
        else if (IsSameString(pToken, TokenFullKey))
        {
            *pOutDeviceType = DeviceTypeIndex_FullKey;
        }
        else
        {
            return false;
        }

        return true;
    }

    // トークンから FW イメージの種類を取得
    bool ParseImageType(ImageType* pOutImageType, const char* pToken) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutImageType);
        NN_SDK_REQUIRES_NOT_NULL(pToken);

        if (IsSameString(pToken, TokenBluetooth))
        {
            *pOutImageType = ImageType::Bluetooth;
        }
        else if (IsSameString(pToken, TokenMcu))
        {
            *pOutImageType = ImageType::Mcu;
        }
        else
        {
            return false;
        }

        return true;
    }

    // トークンからバージョンを取得
    bool ParseVersion(uint16_t* pOutVersion, const char* pToken) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutVersion);
        NN_SDK_REQUIRES_NOT_NULL(pToken);

        int version;
        if (std::sscanf(pToken, "%x", &version) < 1 &&
            std::sscanf(pToken, "%X", &version) < 1)
        {
            return false;
        }

        *pOutVersion = static_cast<uint16_t>(version & 0xFFFF);

        return true;
    }

private:
    char* m_InfoString;         // 解析対象の文字列
    char* m_pCurrentLine;       // 現在の行開始位置
    char* m_pNextLineStart;     // 次の行開始位置
};

// 最新ファームウェア情報を解析するクラス
class FirmwareInfoParser final : public FirmwareInfoParserBase
{
    NN_DISALLOW_COPY(FirmwareInfoParser);
    NN_DISALLOW_MOVE(FirmwareInfoParser);

public:
    explicit FirmwareInfoParser(char* pString) NN_NOEXCEPT :
        FirmwareInfoParserBase(pString) {}

    // 現在の行の情報を解析
    bool ParseCurrentFirmwareInfo(
        DeviceTypeIndex* pOutDeviceType,
        ImageType* pOutImageType,
        uint16_t* pOutVersion,
        char* pOutImagePath,
        size_t imagePathLengthMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutDeviceType);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageType);
        NN_SDK_REQUIRES_NOT_NULL(pOutVersion);
        NN_SDK_REQUIRES_NOT_NULL(pOutImagePath);

        if (IsEnd())
        {
            return false;
        }

        char* pSaveToken;

        // デバイスタイプ
        {
            char* pToken = TokenizeString(GetCurrentLine(), ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseDeviceType(pOutDeviceType, pToken))
            {
                return false;
            }
        }

        // イメージタイプ
        {
            char* pToken = TokenizeString(nullptr, ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseImageType(pOutImageType, pToken))
            {
                return false;
            }
        }

        // バージョン
        {
            char* pToken = TokenizeString(nullptr, ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseVersion(pOutVersion, pToken))
            {
                return false;
            }
        }

        // ファイル名
        {
            char* pToken = TokenizeString(nullptr, ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            nn::util::Strlcpy(pOutImagePath, pToken, static_cast<int>(imagePathLengthMax));
        }

        return true;

    }  // NOLINT(impl/function_size)
};

// Hotfix 判定用ファームウェア情報を解析するクラス
class ExpectVersionInfoParser final : public FirmwareInfoParserBase
{
    NN_DISALLOW_COPY(ExpectVersionInfoParser);
    NN_DISALLOW_MOVE(ExpectVersionInfoParser);

public:
    explicit ExpectVersionInfoParser(char* pString) NN_NOEXCEPT :
        FirmwareInfoParserBase(pString) {}

    // 現在の行の情報を解析
    bool ParseCurrentFirmwareInfo(
        DeviceTypeIndex* pOutDeviceType,
        ImageType* pOutImageType,
        uint16_t* pOutVersion) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutDeviceType);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageType);
        NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

        if (IsEnd())
        {
            return false;
        }

        char* pSaveToken;

        // デバイスタイプ
        {
            char* pToken = TokenizeString(GetCurrentLine(), ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseDeviceType(pOutDeviceType, pToken))
            {
                return false;
            }
        }

        // イメージタイプ
        {
            char* pToken = TokenizeString(nullptr, ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseImageType(pOutImageType, pToken))
            {
                return false;
            }
        }

        // バージョン
        {
            char* pToken = TokenizeString(nullptr, ",", &pSaveToken);
            if (pToken == nullptr)
            {
                return false;
            }

            if (!ParseVersion(pOutVersion, pToken))
            {
                return false;
            }
        }

        return true;

    }  // NOLINT(impl/function_size)
};

}  // anonymous

void ControllerFirmwareAccessor::ClearFirmwareInfo() NN_NOEXCEPT
{
    m_IsImageInfoLoaded = false;
    for (auto& info : m_ImageInfos)
    {
        info.Clear();
    }

    for (auto& version : m_ExpectVersions)
    {
        std::memset(&version, 0, sizeof(version));
    }
}

nn::Result ControllerFirmwareAccessor::Mount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_MountCount.IsMax(),
        system::ResultFirmwareUpdateActivationUpperLimitOver());

    NN_HID_DETAIL_SYSDATA_TRACE("%s: Mount requested\n", NN_CURRENT_FUNCTION_NAME);

    // 初回のみシステムデータをマウント
    if (m_MountCount.IsZero())
    {
        size_t cacheSize;
        {
            auto result = nn::fs::QueryMountSystemDataCacheSize(&cacheSize, m_SystemDataId);
            if (result.IsFailure())
            {
                NN_DETAIL_HID_ERROR(
                    "SystemData-%s: nn::fs::QuertyMountSystemDataCache() returns 0x%08X\n",
                    NN_CURRENT_FUNCTION_NAME,
                    result.GetInnerValueForDebug());
                NN_RESULT_THROW(system::ResultFirmwareImageReadFailed());
            }
            NN_HID_DETAIL_SYSDATA_TRACE("%s: MountSystemDataCacheSize=%zd\n", NN_CURRENT_FUNCTION_NAME, cacheSize);

            // キャッシュサイズが上限に近い場合は警告
            if (cacheSize >= sizeof(m_SystemDataCache) * 3 / 4)
            {
                NN_DETAIL_HID_WARN(
                    "SystemData-%s: SystemDataCacheSize seems to exceed (%zd/%zd)\n",
                    NN_CURRENT_FUNCTION_NAME,
                    cacheSize,
                    sizeof(m_SystemDataCache));
            }
        }

        // キャッシュが足りない場合は実装ミス疑惑
        NN_ABORT_UNLESS_LESS_EQUAL(cacheSize, sizeof(m_SystemDataCache));

        {
            auto result = nn::fs::MountSystemData(m_pSystemDataMountName, m_SystemDataId, m_SystemDataCache, cacheSize);
            if (result.IsFailure())
            {
                NN_DETAIL_HID_ERROR(
                    "SystemData-%s: nn::fs::MountSystemData() returns 0x%08X\n",
                    NN_CURRENT_FUNCTION_NAME,
                    result.GetInnerValueForDebug());
                NN_RESULT_THROW(system::ResultFirmwareImageReadFailed());
            }
        }

        NN_HID_DETAIL_SYSDATA_TRACE("%s: SystemData is mounted\n", NN_CURRENT_FUNCTION_NAME);
    }

    ++m_MountCount;

    NN_RESULT_SUCCESS;
}

void ControllerFirmwareAccessor::Unmount() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_MountCount.IsZero())
    {
        return;
    }

    --m_MountCount;

    NN_HID_DETAIL_SYSDATA_TRACE("%s: Unmount requested\n", NN_CURRENT_FUNCTION_NAME);

    if (m_MountCount.IsZero())
    {
        nn::fs::Unmount(m_pSystemDataMountName);
        NN_HID_DETAIL_SYSDATA_TRACE("%s: SystemData is unmounted\n", NN_CURRENT_FUNCTION_NAME);
    }
}

nn::Result ControllerFirmwareAccessor::GetDestinationVersion(
    system::FirmwareVersion* pOutVersion,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    FirmwareImageInfo imageInfo;
    NN_RESULT_DO(GetFirmwareImageInfo(&imageInfo, handle));

    *pOutVersion = imageInfo.version;

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::GetExpectVersion(
    system::FirmwareVersion* pOutVersion,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    DeviceTypeIndex type;
    NN_RESULT_DO(GetDeviceTypeIndex(&type, handle));

    *pOutVersion = m_ExpectVersions[type];

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::GetBluetoothImageFilePath(
    char* pOutImagePath,
    size_t imagePathLengthMax,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutImagePath);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    FirmwareImageInfo imageInfo;
    NN_RESULT_DO(GetFirmwareImageInfo(&imageInfo, handle));

    // システムデータ内ファイルのパスを返す
    nn::util::SNPrintf(
        pOutImagePath,
        imagePathLengthMax,
        "%s:/%s",
        m_pSystemDataMountName,
        imageInfo.bluetoothPath);

    NN_HID_DETAIL_SYSDATA_TRACE("%s: Path=%s\n", NN_CURRENT_FUNCTION_NAME, pOutImagePath);

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::GetMcuImageFilePath(
    char* pOutImagePath,
    size_t imagePathLengthMax,
    nn::xcd::DeviceHandle handle,
    bool isFullUpdate) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutImagePath);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    FirmwareImageInfo imageInfo;
    NN_RESULT_DO(GetFirmwareImageInfo(&imageInfo, handle));

    // システムデータ内ファイルのパスを返す
    nn::util::SNPrintf(
        pOutImagePath,
        imagePathLengthMax,
        "%s:/%s",
        m_pSystemDataMountName,
        imageInfo.mcuPath);

    // 全領域更新の場合は IAP 入りイメージにする
    if (isFullUpdate)
    {
        // 拡張子を除く
        {
            auto* pDot = std::strrchr(pOutImagePath, '.');
            if (pDot != nullptr)
            {
                *pDot = '\0';
            }
        }

        // パス長は固定なので、接尾辞付与後にパス長がオーバーする場合は NG
        {
            auto length = nn::util::Strnlen(pOutImagePath, static_cast<int>(imagePathLengthMax));
            auto suffixLength = nn::util::Strnlen(WithIapSuffix, static_cast<int>(imagePathLengthMax));
            NN_SDK_REQUIRES_LESS(static_cast<size_t>(length + suffixLength), imagePathLengthMax);
            NN_UNUSED(length);
            NN_UNUSED(suffixLength);
        }

        // IAP 入りイメージの接尾辞を付加
#ifdef NN_BUILD_CONFIG_COMPILER_VC
        strcat_s(pOutImagePath, imagePathLengthMax, WithIapSuffix);
#else
        strlcat(pOutImagePath, WithIapSuffix, imagePathLengthMax);
#endif  // ifdef NN_BUILD_CONFIG_COMPILER_VC
    }

    NN_HID_DETAIL_SYSDATA_TRACE("%s: Path=%s\n", NN_CURRENT_FUNCTION_NAME, pOutImagePath);

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::ReadFirmwareInfoFile(
    char* pOutReadBuffer,
    size_t bufferSize,
    const char* filename) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReadBuffer);
    NN_SDK_REQUIRES_NOT_NULL(filename);

    // ファームウェア情報ファイルのパスを取得
    char filepath[48];
    {
        int charCount = nn::util::SNPrintf(
            filepath,
            sizeof(filepath),
            "%s:/%s",
            m_pSystemDataMountName,
            filename);

        // パスが切りつめられている場合はバッファ不足
        NN_SDK_ASSERT_LESS(static_cast<size_t>(charCount), sizeof(filepath));
        NN_UNUSED(charCount);
    }

    // ファームウェア情報ファイルを一括で読み込む
    {
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(&handle, filepath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t fileSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

        // 現状、情報ファイルはバッファに収まる想定なので、溢れる場合は読み込まない
        // Release ビルド時以外は ASSERT で止める
        bool isAcceptable = fileSize > 0 && static_cast<size_t>(fileSize) < bufferSize;
        NN_SDK_ASSERT(isAcceptable, "FW info file size is too large");
        NN_RESULT_THROW_UNLESS(
            isAcceptable,
            system::ResultFirmwareImageReadFailed()
        );

        NN_RESULT_DO(nn::fs::ReadFile(handle, 0, pOutReadBuffer, static_cast<size_t>(fileSize)));

        // 末尾の NULL 終端を保証
        pOutReadBuffer[fileSize] = '\0';
    }

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::ParseFirmwareInfoInSystemData() NN_NOEXCEPT
{
    // 一旦全情報を無効化
    ClearFirmwareInfo();

    NN_RESULT_DO(
        ReadFirmwareInfoFile(m_ReadBuffer, sizeof(m_ReadBuffer), FirmwareInfoFilePath)
    );

    // ファームウェア情報の解析
    FirmwareInfoParser parser(m_ReadBuffer);
    while (!parser.IsEnd())
    {
        DeviceTypeIndex typeIndex;
        ImageType imageType;
        uint16_t version;
        char path[FirmwareImagePathLengthMax];

        if (parser.ParseCurrentFirmwareInfo(&typeIndex, &imageType, &version, path, sizeof(path)))
        {
            auto& imageInfo = m_ImageInfos[typeIndex];
            switch (imageType)
            {
            case ImageType::Bluetooth:
                {
                    imageInfo.version.major = (version >> 8) & 0xFF;
                    imageInfo.version.minor = version & 0xFF;
                    nn::util::Strlcpy(imageInfo.bluetoothPath, path, FirmwareImagePathLengthMax);

                    NN_HID_DETAIL_SYSDATA_TRACE(
                        "%s: Device=%d  BT Version=%02x.%02x  Path=%s\n",
                        NN_CURRENT_FUNCTION_NAME,
                        typeIndex,
                        imageInfo.version.major,
                        imageInfo.version.minor,
                        imageInfo.bluetoothPath);
                }
                break;

            case ImageType::Mcu:
                {
                    imageInfo.version.micro    = (version >> 8) & 0xFF;
                    imageInfo.version.revision = version & 0xFF;
                    nn::util::Strlcpy(imageInfo.mcuPath, path, FirmwareImagePathLengthMax);

                    NN_HID_DETAIL_SYSDATA_TRACE(
                        "%s: Device=%d  MCU Version=%02x.%02x  Path=%s\n",
                        NN_CURRENT_FUNCTION_NAME,
                        typeIndex,
                        imageInfo.version.micro,
                        imageInfo.version.revision,
                        imageInfo.mcuPath);
                }
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        parser.GoNextLine();
    }

    NN_RESULT_DO(ParseExpectVersionInSystemData());

    m_IsImageInfoLoaded = true;

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::ParseExpectVersionInSystemData() NN_NOEXCEPT
{
    NN_RESULT_DO(
        ReadFirmwareInfoFile(m_ReadBuffer, sizeof(m_ReadBuffer), ExpectVersionFilePath)
    );

    // バージョン情報の解析
    ExpectVersionInfoParser parser(m_ReadBuffer);
    while (!parser.IsEnd())
    {
        DeviceTypeIndex typeIndex;
        ImageType imageType;
        uint16_t version;

        if (parser.ParseCurrentFirmwareInfo(&typeIndex, &imageType, &version))
        {
            auto& versionInfo = m_ExpectVersions[typeIndex];
            switch (imageType)
            {
            case ImageType::Bluetooth:
                {
                    versionInfo.major = (version >> 8) & 0xFF;
                    versionInfo.minor = version & 0xFF;

                    NN_HID_DETAIL_SYSDATA_TRACE(
                        "%s: Device=%d  BT Version=%02x.%02x\n",
                        NN_CURRENT_FUNCTION_NAME,
                        typeIndex,
                        versionInfo.major,
                        versionInfo.minor);
                }
                break;

            case ImageType::Mcu:
                {
                    versionInfo.micro    = (version >> 8) & 0xFF;
                    versionInfo.revision = version & 0xFF;

                    NN_HID_DETAIL_SYSDATA_TRACE(
                        "%s: Device=%d  MCU Version=%02x.%02x\n",
                        NN_CURRENT_FUNCTION_NAME,
                        typeIndex,
                        versionInfo.micro,
                        versionInfo.revision);
                }
                break;

            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        parser.GoNextLine();
    }

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::GetDeviceTypeIndex(
    DeviceTypeIndex* pOutType,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutType);

    // ファームウェア情報未解析なら先に解析する
    if (!m_IsImageInfoLoaded)
    {
        NN_RESULT_DO(Mount());
        NN_UTIL_SCOPE_EXIT
        {
            Unmount();
        };

        auto result = ParseFirmwareInfoInSystemData();
        if (result.IsFailure())
        {
            NN_DETAIL_HID_ERROR(
                "SystemData-%s: Failed to read firmware info (0x%08X)\n",
                NN_CURRENT_FUNCTION_NAME,
                result.GetInnerValueForDebug());
            NN_RESULT_THROW(system::ResultFirmwareImageReadFailed());
        }
    }

    nn::xcd::DeviceInfo deviceInfo;
    NN_RESULT_THROW_UNLESS(
        nn::xcd::GetDeviceInfo(&deviceInfo, handle).IsSuccess(),
        system::ResultUniquePadDisconnected()
    );

    switch (deviceInfo.deviceType)
    {
    case nn::xcd::DeviceType_Left:
        *pOutType = DeviceTypeIndex_JoyLeft;
        break;

    case nn::xcd::DeviceType_Right:
        *pOutType = DeviceTypeIndex_JoyRight;
        break;

    case nn::xcd::DeviceType_FullKey:
        *pOutType = DeviceTypeIndex_FullKey;
        break;

    case nn::xcd::DeviceType_MiyabiLeft:
        *pOutType = DeviceTypeIndex_JoyLeft;
        break;

    case nn::xcd::DeviceType_MiyabiRight:
        *pOutType = DeviceTypeIndex_JoyRight;
        break;

    default:
        // 非サポートデバイス
        NN_RESULT_THROW(system::ResultUniquePadDisconnected());
    }

    NN_RESULT_SUCCESS;
}

nn::Result ControllerFirmwareAccessor::GetFirmwareImageInfo(
    FirmwareImageInfo* pOutInfo,
    nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);

    DeviceTypeIndex type;
    NN_RESULT_DO(GetDeviceTypeIndex(&type, handle));

    *pOutInfo = m_ImageInfos[type];

    NN_RESULT_SUCCESS;
}

}}}  // nn::hid::detail
