﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/settings/system/settings_SixAxisSensor.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_ConsoleSixAxisSensorCalibrationTypes.h"
#include "hid_ConsoleSixAxisSensorProcessor.h"
#include "hid_HandheldManager.h"
#include "hid_InputDetectorManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SixAxisSensorAppletSettingManager.h"
#include "hid_ISixAxisSensorFilter.h"
#include "hid_ConsoleSixAxisSensorAppletSettingManager.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

//!< 一時的メモリ領域を表す構造体です。
struct ConsoleSixAxisSensorTemp final
{
    //!< ARUID
    ::nn::applet::AppletResourceUserId aruid;
};

//!< アプレットリソースの ConsoleSixAxisSensor 用拡張エントリを表す構造体です。
struct ConsoleSixAxisSensorAppletResourceEntry final
{
    ::nn::applet::AppletResourceUserId aruid;   //!< ARUID
};

//!< ConsoleSixAxisSensor インターフェイスの解決を担うマネージャを扱うクラスです。
class ConsoleSixAxisSensorManager final
{
    NN_DISALLOW_COPY(ConsoleSixAxisSensorManager);
    NN_DISALLOW_MOVE(ConsoleSixAxisSensorManager);

public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    enum CalibrationState
    {
        CalibrationState_On,
        CalibrationState_Off,
        CalibrationState_Invalid,
    };

    enum ConsoleSixAxisSensorIdx
    {
        ConsoleSixAxisSensorIdx_Count = 0,
        ConsoleSixAxisSensorIdx_JoyCon,
        ConsoleSixAxisSensorIdx_Seven,
        ConsoleSixAxisSensorIdx_Max,
    };

    enum ConsoleSixAxisSensorSamplingFrequency
    {
        ConsoleSixAxisSensorSamplingFrequency_Off     = 0,
        ConsoleSixAxisSensorSamplingFrequency_System  = 10,
        ConsoleSixAxisSensorSamplingFrequency_Npad    = 200,
        ConsoleSixAxisSensorSamplingFrequency_Seven   = 1000,
    };

private:
    //!< 共有リソースについてアクティブ化された回数
    ActivationCount m_CommonActivationCount;

    //!< ConsoleSixAxisSensor がアクティブ化された回数
    ActivationCount m_ConsoleSixAxisSensorActivationCount;

    //!< インフォーカス状態にある ARUID
    ::nn::applet::AppletResourceUserId m_InFocusAruid;

    //!< 起床状態にあるか否か
    bool m_IsAwake;

    //!< 現在のサンプリング周波数
    int m_SamplingFrequency;

    //!< Seven サンプリングが開始状態にあるか否か
    bool m_IsSevenSamplingStarted;

    //!< Npad Handheld サンプリングが開始状態にあるか否か
    bool m_IsNpadHandheldSamplingStarted;

    //!< ユーザーキャリブレーションが開始状態にあるか否か
    CalibrationState m_CalibrationState;

    //!< サンプリング番号
    int64_t m_SamplingNumbers[ConsoleSixAxisSensorIdx_Max];

    //!< タイマーイベント
    ::nn::os::TimerEventType* m_pTimerEvent;

    //!< ドライバ
    IConsoleSixAxisSensorDriver* m_pDriver;

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< SixAxisSensor の設定値マネージャ
    SixAxisSensorAppletSettingManager* m_pSixAxisSensorAppletSettingManager;

    //!< ConsoleSixAxisSensor の設定値マネージャ
    ConsoleSixAxisSensorAppletSettingManager* m_pConsoleSixAxisSensorAppletSettingManager;

    //!< Handheld マネージャ
    HandheldManager* m_pHandheldManager;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< InputDetector マネージャのミューテックス
    ::nn::os::SdkMutex* m_pInputDetectorManagerMutex;

    //!< ConsoleSixAxisSensorProcessor
    ConsoleSixAxisSensorProcessor* m_pConsoleSixAxisSensorProcessor;

    //!< 6 軸センサーのフィルタ
    ISixAxisSensorFilter* m_pSixAxisSensorFilter;

    //!< ConsoleSixAxisSensor の設定値
    const ConsoleSixAxisSensorSetting* m_pConsoleSixAxisSensorSetting;

    //!< 6 軸センサーのカウント値
    tmp::SixAxisSensorCountState m_SixAxisSensorCountStates[::nn::hid::ConsoleSixAxisSensorStateCountMax];

    //!< 6 軸センサーの入力状態
    SixAxisSensorState m_SixAxisSensorStates[::nn::hid::ConsoleSixAxisSensorStateCountMax];

    //!< Seven 向け 6 軸センサーの入力状態
    SevenSixAxisSensorState m_SevenSixAxisSensorStates[::nn::hid::SevenSixAxisSensorStateCountMax];

    //!< Seven 向け座標系番号
    int64_t m_CoordinateNumber;

    //!< Seven 向けサンプリング開始時刻
    ::nn::os::Tick m_SamplingStartedTick;

    //!< 6軸センサーのキャリブレーション値
    tmp::ConsoleSixAxisSensorCalibrationValues m_ConsoleSixAxisSensorCalibrationValues;

    //!< 最新の 6軸センサーのサンプル数
    int m_AppendStateCount;

    //!< 前回のサンプリング番号
    int64_t m_SixAxisSensorLastSamplingNumber;

    //!< アプレットリソースの ConsoleSixAxisSensor 用拡張エントリ
    ConsoleSixAxisSensorAppletResourceEntry
        m_AppletResourceEntries[AppletResourceEntryCountMax];

    //!< 一時的メモリ領域
    ConsoleSixAxisSensorTemp m_Temp;

    //!< 静止状態
    bool m_SevenSixAxisSensorIsAtRest;

    //!< 垂直ずれの角度
    float m_VerticalizationError;

    //!< 角速度のゼロ点
    ::nn::util::Float3 m_GyroBias;

    //!< 現在有効なキャリブレーションパラメータ
    ConsoleSixAxisSensorCalibrationParameters m_ConsoleSixAxisSensorCalibrationParameters;

public:
    ConsoleSixAxisSensorManager() NN_NOEXCEPT;

    //!< タイマーイベントを設定します。
    void SetTimerEvent(::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT;

    //!< ドライバを設定します。
    void SetDriver(IConsoleSixAxisSensorDriver* pDriver) NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
        ) NN_NOEXCEPT;

    //!< SixAxisSensor の設定値マネージャを設定します。
    void SetSixAxisSensorAppletSettingManager(SixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor の設定値マネージャを設定します。
    void SetConsoleSixAxisSensorAppletSettingManager(ConsoleSixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT;

    //!< Handheld マネージャを設定します。
    void SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT;

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(
        InputDetectorManager* pManager, ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensorProcessor を設定します。
    void SetConsoleSixAxisSensorProcessor(ConsoleSixAxisSensorProcessor* pProcessor) NN_NOEXCEPT;

    //!< SixAxisSensorFilter を設定します。
    void SetSixAxisSensorFilter(ISixAxisSensorFilter* pFilter) NN_NOEXCEPT;

    //!< マネージャを ConsoleSixAxisSensor 向けにアクティブ化します。
    ::nn::Result ActivateForConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< マネージャを ConsoleSixAxisSensor 向けに非アクティブ化します。
    ::nn::Result DeactivateForConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< アプレットリソースの状態を ConsoleSixAxisSensor 向けに保証します。
    ::nn::Result EnsureAppletResourceForConsoleSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

    //!< デバイスを起床状態に遷移させます。
    ::nn::Result WakeDeviceUp() NN_NOEXCEPT;

    //!< デバイスをスリープ状態に遷移させます。
    ::nn::Result PutDeviceToSleep() NN_NOEXCEPT;

    //!< 入力状態を取得します。
    int GetSixAxisSensorState(SixAxisSensorState* pOutStates, int count) const NN_NOEXCEPT;

    //!< 入力状態を取得します。
    int GetSevenSixAxisSensorState(SevenSixAxisSensorState* pOutStates, int count) const NN_NOEXCEPT;

    //!< 本体 NAND からキャリブレーションパラメータを取得します。
    void GetCalibrationParameters(
        ConsoleSixAxisSensorCalibrationParameters* pOutParameters) const NN_NOEXCEPT;

    //!< 本体 NAND へキャリブレーションパラメータを設定します。
    void SetCalibrationParameters(
        const ConsoleSixAxisSensorCalibrationParameters& parameters) NN_NOEXCEPT;

    //!< SixAxisSensorProcessor を返します。
    const SixAxisSensorProcessor& GetSixAxisSensorProcessor() const NN_NOEXCEPT;

    //!< 廃止予定
    ::nn::Result GetConsoleSixAxisSensorCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) const NN_NOEXCEPT;

    //!< 設定値に基づいたサンプリング周波数を返します。
    int GetSamplingFrequency() const NN_NOEXCEPT;

    //!< Npad Handheld のサンプリング設定値を更新します。
    void EnableNpadHandheldSampling(bool enable) NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsConsoleSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported) NN_NOEXCEPT;

    //!< 工程出荷時の本体 6 軸センサーのキャリブレーション値を書き込みます。
    ::nn::Result ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーションを開始します。
    ::nn::Result StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
    ::nn::Result CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション進捗状態を取得します。
    ::nn::Result GetSixAxisSensorAccurateUserCalibrationState(::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState) NN_NOEXCEPT;

    //!< アプリから割り当てられた入力状態格納用バッファの Handle を設定します。
    ::nn::Result SetStateBufferMemoryHandle(const ::nn::applet::AppletResourceUserId& aruid,
                                            ::nn::os::NativeHandle&& transferMemoryHandle,
                                            size_t size,
                                            bool isManaged) NN_NOEXCEPT;

    //!< アプリから割り当てられたワークバッファの Handle を設定します。
    ::nn::Result SetWorkBufferMemoryHandle(const ::nn::applet::AppletResourceUserId& aruid,
                                           ::nn::os::NativeHandle&& transferMemoryHandle,
                                           size_t size,
                                           bool isManaged) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を初期化します。
    ::nn::Result InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を終了します。
    ::nn::Result FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 工程検査向けに生値を取得します。
    int GetConsoleSixAxisSensorCountStates(::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues,
                                           int count) NN_NOEXCEPT;

private:

    //!< 設定値を更新します。
    void UpdateSixAxisSensorSetting() NN_NOEXCEPT;

    //!< SixAxisSensorSetting をアタッチします。
    void AttachSixAxisSensorSetting(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensorSetting をアタッチします。
    void AttachConsoleSixAxisSensorSetting(const nn::hid::ConsoleSixAxisSensorHandle& handle) NN_NOEXCEPT;

    //!< 入力状態を更新します。
    void Update() NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(
            ConsoleSixAxisSensorManager* that,
            ConsoleSixAxisSensorSharedMemoryFormat* ConsoleSixAxisSensorAddress,
            ::nn::applet::AppletResourceUserId aruid,
            bool enablesInput,
            bool isConsoleSixAxisSensorActivated) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 共有リソースをアクティブ化します。
    ::nn::Result ActivateCommon() NN_NOEXCEPT;

    //!< 共有リソースをアクティブ化します。
    ::nn::Result DeactivateCommon() NN_NOEXCEPT;

    //!< 6 軸センサー値を計算します。
    void CalculateSixAxisSensorStates() NN_NOEXCEPT;

    //!< Seven 向け 6 軸センサー値を計算します。
    void CalculateSevenSixAxisSensorStates() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
