﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_IHidTemporaryServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_ConsoleSixAxisSensorHandle.h"
#include "hid_ConsoleSixAxisSensorImpl.h"
#include "hid_ConsoleSixAxisSensorLifo.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_HidTemporaryServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SixAxisSensorHandle.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

const int InternalStateBufferSize =   4 * 1024; //!< 入力状態用バッファサイズです。
const int InternalWorkBufferSize  = 508 * 1024; //!< 作業領域用バッファサイズです。

NN_STATIC_ASSERT((InternalStateBufferSize + InternalWorkBufferSize) == SevenSixAxisSensorWorkBufferSize);
NN_STATIC_ASSERT(InternalStateBufferSize % ::nn::os::MemoryPageSize == 0);
NN_STATIC_ASSERT(InternalWorkBufferSize % ::nn::os::MemoryPageSize == 0);

//!< ConsoleSixAxisSensor の共有メモリへのアクセスを扱うクラスです。
class ConsoleSixAxisSensorClientResourceHolder final : public SharedMemoryAccessor
{
private:
    ::std::atomic<ConsoleSixAxisSensorInternalState*> m_pState;

    //!< 入力状態格納用バッファ
    SevenSixAxisSensorStateLifo* m_pStateBuffer;

public:
    ConsoleSixAxisSensorClientResourceHolder() NN_NOEXCEPT
        : m_pState(nullptr)
        , m_pStateBuffer(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultConsoleSixAxisSensorActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultConsoleSixAxisSensorDeactivationLowerLimitOver());
    }

    virtual ~ConsoleSixAxisSensorClientResourceHolder() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< ConsoleSixAxisSensorState を返します。
    ConsoleSixAxisSensorInternalState* GetInternalState() const NN_NOEXCEPT
    {
        return m_pState;
    }

    //!< TransferMemory 関連の初期化処理を行います。
    void InitializeTransferMemory(void* pStateBuffer) NN_NOEXCEPT
    {
        m_pStateBuffer = reinterpret_cast<SevenSixAxisSensorStateLifo*>(pStateBuffer);
    }

    //!< TransferMemory 関連の終了処理を行います。
    void FinalizeTransferMemory() NN_NOEXCEPT
    {
        m_pStateBuffer = nullptr;
    }

    //!< ReadOnly なバッファから入力状態を取得します。
    ::nn::Result GetSevenSixAxisSensorStates(
        int* pOutCount,
        SevenSixAxisSensorState outValues[],
        int count) const NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pStateBuffer != nullptr,
                               ResultSevenSixAxisSensorNullStateBuffer());

        *pOutCount = m_pStateBuffer->Read(outValues, count);
        NN_RESULT_SUCCESS;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(pHidServer->ActivateConsoleSixAxisSensor(GetAppletResourceUserId()));
        m_pState = &::nn::util::Get(ptr->consoleSixAxisSensor.internalState);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateConsoleSixAxisSensor());
        m_pState = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< ConsoleSixAxisSensor の共有メモリアクセッサを返します。
ConsoleSixAxisSensorClientResourceHolder& GetConsoleSixAxisSensorClientResourceHolder() NN_NOEXCEPT
{
    return StaticObject<ConsoleSixAxisSensorClientResourceHolder>::Get();
}

} // namespace

::nn::Result InitializeConsoleSixAxisSensor() NN_NOEXCEPT
{
    ConsoleSixAxisSensorClientResourceHolder& holder = GetConsoleSixAxisSensorClientResourceHolder();
    NN_RESULT_DO(holder.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(holder.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result StartSixAxisSensor(const ConsoleSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    NN_RESULT_DO(pProxy->StartConsoleSixAxisSensor(GetAppletResourceUserId(),
                                                   handle));
    NN_RESULT_SUCCESS;
}

::nn::Result StopSixAxisSensor(const ConsoleSixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    NN_RESULT_DO(pProxy->StopConsoleSixAxisSensor(GetAppletResourceUserId(),
                                                  handle));
    NN_RESULT_SUCCESS;
}

::nn::Result GetConsoleSixAxisSensorCountStates(int* pOutCount,
                                                tmp::SixAxisSensorCountState outValues[],
                                                int count,
                                                ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(pOutCount);
    NN_UNUSED(outValues);
    NN_UNUSED(count);
    NN_UNUSED(handle);

    NN_RESULT_THROW(ResultConsoleSixAxisSensorInvalidStateCount());
}

::nn::Result GetConsoleSixAxisSensorCalibrationValues(tmp::ConsoleSixAxisSensorCalibrationValues* pOutCalibrationValues,
                                                      ConsoleSixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_UNUSED(pOutCalibrationValues);
    NN_UNUSED(handle);

    NN_RESULT_THROW(ResultConsoleSixAxisSensorInvalidStateCount());
}

::nn::Result GetConsoleSixAxisSensorCountStates(int* pOutCount,
                                                debug::ConsoleSixAxisSensorCountState* pOutStates,
                                                int count) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));

    int32_t outCount;
    NN_RESULT_DO(
        pHidDebugServer->GetConsoleSixAxisSensorCountStates(
            &outCount,
            ::nn::sf::OutArray<debug::ConsoleSixAxisSensorCountState>(pOutStates, static_cast<size_t>(count)),
            GetAppletResourceUserId())
    );

    *pOutCount = static_cast<int>(outCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetConsoleSixAxisSensorSamplingFrequency(int64_t* pOutSamplingFrequency) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->GetConsoleSixAxisSensorSamplingFrequency(pOutSamplingFrequency,
                                                                           GetAppletResourceUserId()));

    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeConsoleSixAxisSensor() NN_NOEXCEPT
{
    NN_RESULT_DO(GetConsoleSixAxisSensorClientResourceHolder().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result InitializeSevenSixAxisSensor(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(workBufferSize);

    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_ALIGNED(workBuffer, ::nn::os::MemoryPageSize);
    NN_SDK_REQUIRES_EQUAL(static_cast<size_t>(SevenSixAxisSensorWorkBufferSize), workBufferSize);
    NN_SDK_REQUIRES_EQUAL(static_cast<size_t>(InternalStateBufferSize + InternalWorkBufferSize), workBufferSize);

    ConsoleSixAxisSensorClientResourceHolder& holder = GetConsoleSixAxisSensorClientResourceHolder();
    NN_RESULT_DO(holder.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(holder.Activate());

    // メモリ領域をワークバッファ(None)/ステートバッファ(ReadOnly)に分けて IPC 転送する
    auto internalStateBuffer = workBuffer;
    auto internalWorkBuffer = reinterpret_cast<unsigned char*>(workBuffer) + InternalStateBufferSize;

    // state バッファの設定
    ::nn::os::TransferMemoryType stateTransferMemory;
    NN_RESULT_THROW_UNLESS(::nn::os::CreateTransferMemory(&stateTransferMemory,
                                                          internalStateBuffer,
                                                          InternalStateBufferSize,
                                                          ::nn::os::MemoryPermission_ReadOnly).IsSuccess(),
                                                          ResultSevenSixAxisSensorNullStateBuffer());
    auto stateTransferMemoryHandle = ::nn::os::DetachTransferMemory(&stateTransferMemory);
    ::nn::os::DestroyTransferMemory(&stateTransferMemory);

    // work バッファの設定
    ::nn::os::TransferMemoryType workTransferMemory;
    NN_RESULT_THROW_UNLESS(::nn::os::CreateTransferMemory(&workTransferMemory,
                                                          internalWorkBuffer,
                                                          InternalWorkBufferSize,
                                                          ::nn::os::MemoryPermission_None).IsSuccess(),
                                                          ResultSevenSixAxisSensorNullStateBuffer());
    auto workTransferMemoryHandle = ::nn::os::DetachTransferMemory(&workTransferMemory);
    ::nn::os::DestroyTransferMemory(&workTransferMemory);

    // IPC 転送
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->InitializeSevenSixAxisSensor(GetAppletResourceUserId(),
                                                      ::nn::sf::NativeHandle(stateTransferMemoryHandle, true),
                                                      InternalStateBufferSize,
                                                      ::nn::sf::NativeHandle(workTransferMemoryHandle, true),
                                                      InternalWorkBufferSize));

    holder.InitializeTransferMemory(internalStateBuffer);
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeSevenSixAxisSensor() NN_NOEXCEPT
{
    // IPC 転送
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));
    NN_RESULT_DO(pProxy->FinalizeSevenSixAxisSensor(GetAppletResourceUserId()));

    ConsoleSixAxisSensorClientResourceHolder& holder = GetConsoleSixAxisSensorClientResourceHolder();
    holder.FinalizeTransferMemory();
    NN_RESULT_SUCCESS;
}

::nn::Result StartSevenSixAxisSensor() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    NN_RESULT_DO(pProxy->StartSevenSixAxisSensor(GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result StopSevenSixAxisSensor() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    NN_RESULT_DO(pProxy->StopSevenSixAxisSensor(GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSevenSixAxisSensorStates(int* pOutCount,
                                         SevenSixAxisSensorState outValues[],
                                         int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outValues != nullptr,
        ResultSevenSixAxisSensorNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultSevenSixAxisSensorInvalidStateCount());

    ConsoleSixAxisSensorClientResourceHolder& holder = GetConsoleSixAxisSensorClientResourceHolder();

    return holder.GetSevenSixAxisSensorStates(pOutCount, outValues, count);
}

::nn::Result IsSevenSixAxisSensorAtRest(bool* pOutIsAtRest) NN_NOEXCEPT
{
    const auto pInternalState = GetConsoleSixAxisSensorClientResourceHolder().GetInternalState();
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr,
                           ResultSevenSixAxisSensorNotInitialized());
    *pOutIsAtRest = pInternalState->GetSevenSixAxisSensorIsAtRest();
    NN_RESULT_SUCCESS;
}

::nn::Result GetSensorFusionError(float* pOutSensorFusionError) NN_NOEXCEPT
{
    const auto pInternalState = GetConsoleSixAxisSensorClientResourceHolder().GetInternalState();
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr,
                           ResultSevenSixAxisSensorNotInitialized());
    *pOutSensorFusionError = pInternalState->GetVerticalizationError();
    NN_RESULT_SUCCESS;
}

::nn::Result GetGyroBias(::nn::util::Float3* pOutGyroBias) NN_NOEXCEPT
{
    const auto pInternalState = GetConsoleSixAxisSensorClientResourceHolder().GetInternalState();
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr,
                           ResultSevenSixAxisSensorNotInitialized());
    *pOutGyroBias = pInternalState->GetGyroBias();
    NN_RESULT_SUCCESS;
}


::nn::Result SetSevenSixAxisSensorFusionStrength(float strength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0.f, strength);
    NN_SDK_REQUIRES_LESS_EQUAL(strength, 10.f);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    NN_RESULT_DO(pProxy->SetSevenSixAxisSensorFusionStrength(
        GetAppletResourceUserId(),
        strength)
    );
    NN_RESULT_SUCCESS;
}

::nn::Result GetSevenSixAxisSensorFusionStrength(float* pOutStrength) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutStrength);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    NN_RESULT_DO(CreateHidServerProxy(&pProxy));

    auto outStrength = float();
    NN_RESULT_DO(pProxy->GetSevenSixAxisSensorFusionStrength(
        ::nn::sf::Out<float>(&outStrength),
        GetAppletResourceUserId())
    );
    *pOutStrength = static_cast<float>(outStrength);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
