﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>

#include "hid_AudioControlDriver.h"

namespace nn { namespace hid { namespace detail {

namespace {

AudioControlDriver::AudioControlDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_SamplingNumber(0)
    , m_Count()
    , m_Reports()
    , m_HasNewReport(false)
{
    // 何もしない
}

AudioControlDriver::~AudioControlDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result AudioControlDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultAudioControlActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_Count = 0;

        for (size_t i = 0; i < PortCount; ++i)
        {
            m_Reports[i] = 0;
        }
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AudioControlDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultAudioControlDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void AudioControlDriver::GetState(system::AudioControlState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    *pOutValue = system::AudioControlState();

    pOutValue->samplingNumber = m_SamplingNumber;

    if(m_HasNewReport)
    {
        m_SamplingNumber++;
        m_HasNewReport = false;
    }
    // 値は取らない
}

void AudioControlDriver::Attach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_UNUSED(port);
    const int count = ++m_Count;
    NN_SDK_ASSERT_MINMAX(count, 1, static_cast<int>(PortCount));
    NN_UNUSED(count);
}

void AudioControlDriver::Detach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_UNUSED(port);
    const int count = --m_Count;
    NN_SDK_ASSERT_MINMAX(count, 0, static_cast<int>(PortCount - 1));
    NN_UNUSED(count);
}

void AudioControlDriver::SetReport(size_t port, uint64_t reports) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    m_Reports[port] = reports;
    m_HasNewReport = true;
}

}}} // namespace nn::hid::detail

