﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <new>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ahid/ahid.h>
#include <nn/ahid/hdr/hdr.h>
#include <nn/xcd/xcd_Usb.h>

#include "ahid_Hdr.h"
#include "cdhidUsb_Main.h"
#include "hid_AbstractedPadUsbDriver.h"
#include "hid_ActivationCount.h"
#include "hid_AhidManager-os.horizon.h"
#include "hid_AudioControlDriver.h"
#include "hid_KeyboardDriver-os.horizon.h"
#include "hid_MouseDriver-os.horizon.h"
#include "hid_SynchronizedTimer.h"

namespace nn { namespace hid { namespace detail {

NN_STATIC_ASSERT(AhidPortCount <= ::nn::ahid::hdr::AhidDevicesCountMax);
NN_STATIC_ASSERT(AhidPortCount <= ::nn::ahid::hdr::HdrEntriesCountMax);
NN_STATIC_ASSERT(AhidPortCount <= KeyboardDriver::PortCount);
NN_STATIC_ASSERT(AhidPortCount <= MouseDriver::PortCount);
NN_STATIC_ASSERT(AhidPortCount <= AbstractedPadUsbDriver::PortCount);
NN_STATIC_ASSERT(AhidPortCount <= AudioControlDriver::PortCount);

namespace {

//!< ahid のポートの制御フラグ定義です。
struct AhidPortFlag final
{
    typedef ::nn::util::BitFlagSet<32, AhidPortFlag>::Flag<0>
            IsInitialized;  //!< 初期化済みか否か
    typedef ::nn::util::BitFlagSet<32, AhidPortFlag>::Flag<1>
            IsInsane;       //!< 異常な状態か否か
    typedef ::nn::util::BitFlagSet<32, AhidPortFlag>::Flag<2>
            IsDirty;        //!< 検査待ちか否か
    typedef ::nn::util::BitFlagSet<32, AhidPortFlag>::Flag<3>
            IsXcdDevice;    //!< Xcd で管理されているデバイスかどうか
    typedef ::nn::util::BitFlagSet<32, AhidPortFlag>::Flag<4>
            IsSupportedGamePad;   //!< 対応している GamePad かどうか
};

//!< ahid のポートの制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, AhidPortFlag> AhidPortFlagSet;


//!< 拡張グリップから受け取るパケットの最大長
const size_t ExternalGripReportLengthMax = 64;

//!< 外付けUSBHIDデバイスから受け取るパケットの最大長
const size_t ExternalUsbHidDeviceLengthMax = 64;


//!< ahid のレポートを表す共用体です。
union AhidReport
{
    //!< ahid のマウスのレポート
    ::nn::Bit8 mouse[64];

    //!< ahid のキーボードのレポート
    uint64_t keyboard;

    //!< 拡張グリップから受け取るレポートサイズ
    uint8_t  externalGrip[ExternalGripReportLengthMax];

    //!< 外付けUSBHIDデバイスから受け取るレポートサイズ
    uint8_t  usbHidDevice[ExternalUsbHidDeviceLengthMax];

    //!< ahid のオーディオコントロールのレポート
    ::nn::Bit8 audioControl[64];

    //!< Gc コントローラーアダプター用のレポート
    uint8_t  gcAdapter[37];
};

//!< マウスのコードブックアイテムを表す構造体です。
struct MouseCodeBookItems
{
    ::nn::ahid::Item* pButtons[5];  //!< ボタンのアイテムです。
    ::nn::ahid::Item* pX;           //!< x 座標のアイテムです。
    ::nn::ahid::Item* pY;           //!< y 座標のアイテムです。
    ::nn::ahid::Item* pWheel;       //!< ホイールのアイテムです。
};

//!< オーディオコントロールのコードブックアイテムを表す構造体です。
struct AudioControlCodeBookItems
{
    ::nn::ahid::Item* pVolumeIncrement;  //!< ボリュームアップボタンのアイテムです。
    ::nn::ahid::Item* pVolumeDecrement;  //!< ボリュームダウンボタンのアイテムです。
    ::nn::ahid::Item* pMute;             //!< ミュートボタンのアイテムです。
    ::nn::ahid::Item* pStop;             //!< 停止ボタンのアイテムです。
};

//!< コードブックアイテムを表す共用体です。
union CodeBookItems
{
    MouseCodeBookItems mouse;                //!< マウスのコードブックアイテム
    AudioControlCodeBookItems audioControl;  //!< オーディオコントロールのコードブックアイテム
};

//!< ahid の受信ポートのコンテキストを表す構造体です。
struct AhidInPortContext final
{
    ::nn::ahid::hdr::DeviceHandle deviceHandle; //!< ahid のデバイスハンドル
    ::nn::Result result;                        //!< 処理結果
    uint32_t bytes;                             //!< 受信バイト数
    void* buffer;                               //!< バッファ
    size_t bufferSize;                          //!< バッファのサイズ
};

//!< ahid のポートを表す構造体です。
struct AhidPort final
{
    //!< 受信ポートのタイマーイベント
    SynchronizedTimer* pTimerEventIn;

    //!< 送信ポートのタイマーイベント
    SynchronizedTimer* pTimerEventOut;

    //!< キーボードドライバ
    KeyboardDriver* pKeyboardDriver;

    //!< マウスドライバ
    MouseDriver* pMouseDriver;

    //!< UsbPadドライバ
    AbstractedPadUsbDriver* pAbstractedPadUsbDriver;

    //!< オーディオコントロールドライバ
    AudioControlDriver* pAudioControlDriver;

    //!< ahid の制御フラグ
    AhidPortFlagSet flags;

    //!< ahid のインターフェイスのストレージ
    ::nn::util::TypedStorage<::nn::ahid::Ahid,
                      sizeof(::nn::ahid::Ahid),
                  NN_ALIGNOF(::nn::ahid::Ahid)> ahid;

    //!< ahid のデバイスハンドル
    ::nn::ahid::hdr::DeviceHandle deviceHandle;

    //!< ahid のデバイスパラメータ
    ::nn::ahid::hdr::DeviceParameters deviceParameters;

    //!< ahid のコードブックヘッダ
    ::nn::ahid::CodeBookHeader* pCodeBookHeader;

    //!< ahid のコードブックアイテム
    CodeBookItems items;

    //!< ahid のレポート
    AhidReport report;

    //!< キーボードの LED の点灯パターン
    uint8_t leds;

    //!< ahid の受信ポートのコンテキスト
    AhidInPortContext inPortContext;
};

//!< ahid のフィルタの最大数
const size_t AhidDeviceAttachFilterLengthMax = 3;

//!< ahid のポートの解決を担うマネージャを扱うクラスです。
class AhidPortManager final
{
    NN_DISALLOW_COPY(AhidPortManager);
    NN_DISALLOW_MOVE(AhidPortManager);

private:
    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< ahid のマネージャ
    ::nn::ahid::hdr::Hdr m_Hdr;

    //!< ahid のポート
    AhidPort m_Ports[AhidPortCount];

    //!< Gc コントローラーアダプタードライバ
    GcAdapterDriver* m_pGcAdapterDriver;

    //!< ahid のフィルタ
    ::nn::ahid::hdr::AttachFilter m_Filters[AhidDeviceAttachFilterLengthMax];

    //!< ahid のハンドル
    ::nn::ahid::hdr::DeviceHandle m_Handles[AhidPortCount];

public:
    AhidPortManager() NN_NOEXCEPT;

    //!< 受信ポートのタイマーイベントを設定します。
    void SetTimerEventIn(
        size_t portIndex, SynchronizedTimer* pTimerEvent) NN_NOEXCEPT;

    //!< 送信ポートのタイマーイベントを設定します。
    void SetTimerEventOut(
        size_t portIndex, SynchronizedTimer* pTimerEvent) NN_NOEXCEPT;

    //!< キーボードドライバを設定します。
    void SetKeyboardDriver(
        size_t portIndex, KeyboardDriver* pDriver) NN_NOEXCEPT;

    //!< マウスドライバを設定します。
    void SetMouseDriver(
        size_t portIndex, MouseDriver* pDriver) NN_NOEXCEPT;

    //!< USBPadドライバを設定します。
    void SetAbstractedPadUsbDriver(
        size_t portIndex, AbstractedPadUsbDriver* pDriver) NN_NOEXCEPT;

    //!< オーディオコントロールドライバを設定します。
    void SetAudioControlDriver(
        size_t portIndex, AudioControlDriver* pDriver) NN_NOEXCEPT;

    //!< Gc コントローラーアダプタードライバーを設定する
    void SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< ポートを解決します。
    void Dispatch() NN_NOEXCEPT;

    //!< 送信ポートを処理します。
    void ProcessOutPort(size_t portIndex) NN_NOEXCEPT;

    //!< 受信ポートのコンテキストを用意します。
    void PrepareInPortContext(size_t portIndex) NN_NOEXCEPT;

    //!< 受信ポートのコンテキストを処理します。
    void ProcessInPortContext(size_t portIndex) NN_NOEXCEPT;

    //!< 受信ポートのコンテキストをパースします。
    void ParseInPortContext(size_t portIndex) NN_NOEXCEPT;

private:
    //!< ahid のハンドルを取得します。
    size_t GetDeviceHandles(
        ::nn::ahid::hdr::DeviceHandle outHandles[], size_t count) NN_NOEXCEPT;

    //!< Xcd 対応の Usb デバイスを初期化します。
    ::nn::Result ActivateXcdDevice(size_t portIndex) NN_NOEXCEPT;

    //!< デバイスをアタッチします。
    void AttachDevice(
        size_t portIndex, ::nn::ahid::hdr::DeviceHandle handle) NN_NOEXCEPT;

    //!< デバイスをデタッチします。
    void DetachDevice(size_t portIndex) NN_NOEXCEPT;

    //!< キーボードの LED の状態を更新します。
    void UpdateKeyboardLeds(size_t portIndex) NN_NOEXCEPT;

    //!< NX コントローラーに OutputReport を送信します
    void SendOutputReportToNxController(size_t portIndex) NN_NOEXCEPT;

    //!< Gc コントローラーアダプターに OutputReport を送信します
    void SendOutputReportToGcAdapter(size_t portIndex) NN_NOEXCEPT;
};

//!< ahid のポートの解決を担うマネージャを返します。
AhidPortManager& GetAhidPortManager() NN_NOEXCEPT;

} // namespace

const ::nn::TimeSpan AhidManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(8);

AhidManager::AhidManager() NN_NOEXCEPT
    : m_ActivationCount()
    , m_pTimerEvent(nullptr)
    , m_pKeyboardDriver(nullptr)
    , m_pMouseDriver(nullptr)
    , m_pAbstractedPadUsbDriver(nullptr)
    , m_pAudioControlDriver(nullptr)
    , m_PortType(PortType_Control)
    , m_Port(AhidPortCount)
{
}

void AhidManager::SetTimerEvent(SynchronizedTimer* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimerEvent = pTimerEvent;
}

void AhidManager::SetKeyboardDriver(KeyboardDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pKeyboardDriver = pDriver;
}

void AhidManager::SetMouseDriver(MouseDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pMouseDriver = pDriver;
}

void AhidManager::SetAbstractedPadUsbDriver(
    AbstractedPadUsbDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAbstractedPadUsbDriver = pDriver;
}

void AhidManager::SetAudioControlDriver(AudioControlDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAudioControlDriver = pDriver;
}

void AhidManager::SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);

    m_pGcAdapterDriver = pDriver;
}

void AhidManager::SetPort(PortType type, size_t port) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(port, 0u, AhidPortCount);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_PortType = type;
    m_Port = port;
}

::nn::Result AhidManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadUsbDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pGcAdapterDriver);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsMax(), ResultAhidActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        // キーボードドライバをアクティブ化
        NN_RESULT_DO(m_pKeyboardDriver->Activate());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_pKeyboardDriver->Deactivate();
            }
        };

        // マウスドライバをアクティブ化
        NN_RESULT_DO(m_pMouseDriver->Activate());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_pMouseDriver->Deactivate();
            }
        };

        // Usb コントローラードライバをアクティブ化
        NN_RESULT_DO(m_pAbstractedPadUsbDriver->Activate());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_pAbstractedPadUsbDriver->Deactivate();
            }
        };

        // オーディオコントロールドライバをアクティブ化
        NN_RESULT_DO(m_pAudioControlDriver->Activate());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_pAudioControlDriver->Deactivate();
            }
        };

        // キーボードドライバを設定
        GetAhidPortManager().SetKeyboardDriver(m_Port, m_pKeyboardDriver);

        // マウスドライバを設定
        GetAhidPortManager().SetMouseDriver(m_Port, m_pMouseDriver);

        // USBHIDドライバを設定
        GetAhidPortManager()
            .SetAbstractedPadUsbDriver(m_Port, m_pAbstractedPadUsbDriver);

        // オーディオコントロールドライバを設定
        GetAhidPortManager()
            .SetAudioControlDriver(m_Port, m_pAudioControlDriver);

        // Gc コントローラーアダプタードライバーを設定
        GetAhidPortManager().SetGcAdapterDriver(m_pGcAdapterDriver);

        // ポートマネージャをアクティブ化
        NN_RESULT_DO(GetAhidPortManager().Activate());

        needsRollback = false;

        switch (m_PortType)
        {
        case PortType_Control:
            // ポート解決を担当するスレッドのみタイマーイベントをアクティブ化
            m_pTimerEvent->Enable(SamplingInterval);
            break;

        case PortType_In:
            GetAhidPortManager().SetTimerEventIn(m_Port, m_pTimerEvent);
            break;

        case PortType_Out:
            GetAhidPortManager().SetTimerEventOut(m_Port, m_pTimerEvent);
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AhidManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimerEvent);
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseDriver);
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadUsbDriver);

    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsZero(), ResultAhidDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // タイマーイベントを非アクティブ化
        m_pTimerEvent->Disable();

        // ポートマネージャを非アクティブ化
        NN_RESULT_DO(GetAhidPortManager().Deactivate());

        // マウスドライバを非アクティブ化
        NN_RESULT_DO(m_pMouseDriver->Deactivate());

        // Usb コントローラードライバを非アクティブ化
        NN_RESULT_DO(m_pAbstractedPadUsbDriver->Deactivate());

        // キーボードドライバを非アクティブ化
        NN_RESULT_DO(m_pKeyboardDriver->Deactivate());

        // オーディオコントロールドライバを非アクティブ化
        NN_RESULT_DO(m_pAudioControlDriver->Deactivate());
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AhidManager::DoPreProcessing() NN_NOEXCEPT
{
    switch (m_PortType)
    {
    case PortType_Control:
        GetAhidPortManager().Dispatch();
        break;

    case PortType_In:
        GetAhidPortManager().PrepareInPortContext(m_Port);
        break;

    case PortType_Out:
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AhidManager::DoPostProcessing() NN_NOEXCEPT
{
    switch (m_PortType)
    {
    case PortType_Control:
        break;

    case PortType_In:
        GetAhidPortManager().ParseInPortContext(m_Port);
        break;

    case PortType_Out:
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AhidManager::Communicate() NN_NOEXCEPT
{
    switch (m_PortType)
    {
    case PortType_Control:
        break;

    case PortType_In:
        GetAhidPortManager().ProcessInPortContext(m_Port);
        break;

    case PortType_Out:
        GetAhidPortManager().ProcessOutPort(m_Port);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

namespace {

//!< Usage Page の定義です。
struct UsagePage final
{
    static const uint16_t GenericDesktop = 0x01;
    static const uint16_t GameControls = 0x05;
    static const uint16_t Button = 0x09;
    static const uint16_t Consumer = 0x0C;
};

//!< Generic Desktop Page の Usage Id の定義です。
struct GenericDesktopUsageId final
{
    static const uint16_t Pointer = 0x01;
    static const uint16_t Mouse = 0x02;
    static const uint16_t GamePad = 0x05;
    static const uint16_t Keyboard = 0x06;
    static const uint16_t X = 0x30;
    static const uint16_t Y = 0x31;
    static const uint16_t Wheel = 0x38;
};

//!< Consumer Page の Usage Id の定義です。
struct ConsumerUsageId final
{
    static const uint16_t ConsumerControl = 0x01;
    static const uint16_t Stop = 0xB7;
    static const uint16_t Mute = 0xE2;
    static const uint16_t VolumeIncrement = 0xE9;
    static const uint16_t VolumeDecrement = 0xEA;
};

//!< Descriptor Type の定義です。
struct DescriptorType final
{
    static const uint8_t Configuration = 0x02;
};

//!< 指定された Usage Id と対応する Report Id を取得します。
template<size_t N>
bool GetReportId(
    ::nn::ahid::Ahid& ahid, uint8_t* pOutValue, const uint16_t (&usageIds)[N]
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto reportCount = uint8_t();

    ::nn::Result result = ahid.GetReportCount(&reportCount);

    if (result.IsFailure())
    {
        return false;
    }

    auto reportId = uint8_t();

    auto usagePage = uint8_t();

    auto usage = uint8_t();

    for (size_t i = 0; i < reportCount; ++i)
    {
        result = ahid.GetReportIdUsage(&reportId, &usagePage, &usage, i);

        if (result.IsFailure())
        {
            return false;
        }
        else if (usagePage == UsagePage::GenericDesktop)
        {
            for (int16_t usageId : usageIds)
            {
                if (usage == usageId)
                {
                    *pOutValue = reportId;

                    return true;
                }
            }
        }
        else if (usagePage == UsagePage::Consumer)
        {
            for (int16_t usageId : usageIds)
            {
                if (usage == usageId)
                {
                    *pOutValue = reportId;

                    return true;
                }
            }
        }
    }

    return false;
}

//!< 指定された Report Id からマウスのコードブックアイテムを取得します。
bool GetMouseCodeBookItems(
    ::nn::ahid::Ahid& ahid, MouseCodeBookItems* pOutValue, uint8_t reportId
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto result = ::nn::Result();

    for (uint8_t i = 0; i < NN_ARRAY_SIZE(pOutValue->pButtons); ++i)
    {
        result = ahid.GetInputItem(
            &pOutValue->pButtons[i], reportId, UsagePage::Button,
            i + static_cast<uint8_t>(1),
            i + static_cast<uint8_t>(1), 0);

        if (result.IsSuccess())
        {
            if (pOutValue->pButtons[i] == nullptr)
            {
                return false;
            }
        }
        else
        {
            if (i < 2)
            {
                return false;
            }
            else
            {
                for (uint8_t j = i; j < NN_ARRAY_SIZE(pOutValue->pButtons); ++j)
                {
                    pOutValue->pButtons[j] = nullptr;
                }

                break;
            }
        }
    }

    result = ahid.GetInputItem(
        &pOutValue->pX, reportId, UsagePage::GenericDesktop,
        GenericDesktopUsageId::X, GenericDesktopUsageId::X, 0);

    if (result.IsFailure() || pOutValue->pX == nullptr)
    {
        return false;
    }

    result = ahid.GetInputItem(
        &pOutValue->pY, reportId, UsagePage::GenericDesktop,
        GenericDesktopUsageId::Y, GenericDesktopUsageId::Y, 0);

    if (result.IsFailure() || pOutValue->pY == nullptr)
    {
        return false;
    }

    result = ahid.GetInputItem(
        &pOutValue->pWheel, reportId, UsagePage::GenericDesktop,
        GenericDesktopUsageId::Wheel, GenericDesktopUsageId::Wheel, 0);

    if (result.IsFailure())
    {
        pOutValue->pWheel = nullptr;
    }

    return true;
}

//!< 指定された Report Id からオーディオコントロールのコードブックアイテムを取得します。
bool GetAudioControlCodeBookItems(
    ::nn::ahid::Ahid& ahid, AudioControlCodeBookItems* pOutValue, uint8_t reportId
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    auto result = ::nn::Result();

    result = ahid.GetInputItem(
        &pOutValue->pVolumeIncrement, reportId, UsagePage::Consumer,
        ConsumerUsageId::VolumeIncrement, ConsumerUsageId::VolumeIncrement, 0);

    if (result.IsFailure() || pOutValue->pVolumeIncrement == nullptr)
    {
        return false;
    }

    result = ahid.GetInputItem(
        &pOutValue->pVolumeDecrement, reportId, UsagePage::Consumer,
        ConsumerUsageId::VolumeDecrement, ConsumerUsageId::VolumeDecrement, 0);

    if (result.IsFailure() || pOutValue->pVolumeDecrement == nullptr)
    {
        return false;
    }

    result = ahid.GetInputItem(
        &pOutValue->pMute, reportId, UsagePage::Consumer,
        ConsumerUsageId::Mute, ConsumerUsageId::Mute, 0);

    if (result.IsFailure())
    {
        pOutValue->pMute = nullptr;
    }

    result = ahid.GetInputItem(
        &pOutValue->pStop, reportId, UsagePage::Consumer,
        ConsumerUsageId::Stop, ConsumerUsageId::Stop, 0);

    if (result.IsFailure())
    {
        pOutValue->pStop = nullptr;
    }

    return true;
}

// マウスのレポートをデコードします。
bool DecodeMouseReport(
    ::nn::ahid::Ahid& ahid, MouseReport* pOutValue,
    uint8_t bytes[], MouseCodeBookItems* pItems) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(bytes);
    NN_SDK_REQUIRES_NOT_NULL(pItems);

    auto result = ::nn::Result();

    pOutValue->buttons.Reset();

    for (uint8_t i = 0; i < NN_ARRAY_SIZE(pItems->pButtons); ++i)
    {
        ::nn::ahid::Item* const pButton = pItems->pButtons[i];

        if (pButton == nullptr)
        {
            break;
        }

        auto button = int32_t();

        result = ahid.Decode(&button, bytes, pButton);

        if (result.IsFailure())
        {
            return false;
        }

        pOutValue->buttons.Set(i, button != 0);
    }

    result = ahid.Decode(&pOutValue->deltaX, bytes, pItems->pX);

    if (result.IsFailure())
    {
        return false;
    }

    result = ahid.Decode(&pOutValue->deltaY, bytes, pItems->pY);

    if (result.IsFailure())
    {
        return false;
    }

    if (pItems->pWheel == nullptr)
    {
        pOutValue->wheelDelta = 0;
    }
    else
    {
        result = ahid.Decode(&pOutValue->wheelDelta, bytes, pItems->pWheel);

        if (result.IsFailure())
        {
            return false;
        }
    }

    return true;
}

// オーディオコントロールのレポートをデコードします。
bool DecodeAudioControlReport(
    ::nn::ahid::Ahid& ahid, AudioControlReport* pOutValue,
    uint8_t bytes[], AudioControlCodeBookItems* pItems) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(bytes);
    NN_SDK_REQUIRES_NOT_NULL(pItems);

    auto result = ::nn::Result();

    result = ahid.Decode(&pOutValue->VolumeDecrement, bytes, pItems->pVolumeDecrement);

    if (result.IsFailure())
    {
        return false;
    }

    result = ahid.Decode(&pOutValue->VolumeIncrement, bytes, pItems->pVolumeIncrement);

    if (result.IsFailure())
    {
        return false;
    }

    if (pItems->pMute == nullptr)
    {
        pOutValue->Mute = 0;
    }
    else
    {
        result = ahid.Decode(&pOutValue->Mute, bytes, pItems->pMute);

        if (result.IsFailure())
        {
            return false;
        }
    }

    if (pItems->pStop == nullptr)
    {
        pOutValue->Mute = 0;
    }
    else
    {
        result = ahid.Decode(&pOutValue->Stop, bytes, pItems->pStop);

        if (result.IsFailure())
        {
            return false;
        }
    }

    //NN_DETAIL_HID_TRACE("AudioControl Vol+:%d, Vol-:%d Mute:%d, Stop:%d\n",
    //                    pOutValue->VolumeIncrement, pOutValue->VolumeDecrement, pOutValue->Mute, pOutValue->Stop);

    return true;
}

//!< サポート対象のUSBコントローラであるか否かを表す値をが返します。
bool IsSupportedUsbPad(uint32_t vid, uint32_t pid)
{
    //!< サポートするコントローラの定義
    const uint32_t HoriVid = 0x0F0D;
    const uint32_t HoriPid[] = { 0x0092,0x00AA,0x00AB,0x00C1 };
    const uint32_t HoriPidStart = 0x00F0;
    const uint32_t HoriPidEnd = 0x00FF;

    const uint32_t PdpVid = 0x0E6F;
    const uint32_t PdpPidStart = 0x0180;
    const uint32_t PdpPidEnd = 0x0189;

    const uint32_t BdaVid = 0x20D6;
    const uint32_t BdaPidStart = 0xA710;
    const uint32_t BdaPidEnd = 0xA719;

    //HORI
    if (vid == HoriVid)
    {
        for (int i = 0; i < NN_ARRAY_SIZE(HoriPid); i++)
        {
            if (pid == HoriPid[i])
            {
                return true;
            }
        }

        if (HoriPidStart <= pid && pid <= HoriPidEnd)
        {
            return true;
        }
    }

    //PDP
    if (vid == PdpVid)
    {
        if (PdpPidStart <= pid && pid <= PdpPidEnd)
        {
            return true;
        }
    }

    //BDA
    if (vid == BdaVid)
    {
        if (BdaPidStart <= pid && pid <= BdaPidEnd)
        {
            return true;
        }
    }

    return false;
}

//!< ahid のデバイスパラメータがサポート対象か否かを表す値をが返します。
bool IsSupportedDeviceParameters(
    const ::nn::ahid::hdr::DeviceParameters& deviceParameters) NN_NOEXCEPT
{
    if(deviceParameters.usagePage == UsagePage::GenericDesktop)
    {
        if (deviceParameters.usageId == GenericDesktopUsageId::Mouse)
        {
            return true;
        }

        if (deviceParameters.usageId == GenericDesktopUsageId::Keyboard)
        {
            return true;
        }

        if (deviceParameters.usageId == GenericDesktopUsageId::GamePad &&
            IsSupportedUsbPad(deviceParameters.vendorId, deviceParameters.productId)
        )
        {
            return true;
        }
    }

    if(deviceParameters.usagePage == UsagePage::Consumer)
    {
        if (deviceParameters.usageId == ConsumerUsageId::ConsumerControl)
        {
            return true;
        }
    }

    ::nn::xcd::UsbHidDeviceInfo xcdDeviceInfo;
    xcdDeviceInfo.pid = deviceParameters.productId;
    xcdDeviceInfo.vid = deviceParameters.vendorId;
    xcdDeviceInfo.interval = ::nn::TimeSpan::FromMilliSeconds(8); // FixMe : Ahid 経由でインターバルがとれれば修正する
    if (::nn::xcd::IsUsbHidSupported(xcdDeviceInfo))
    {
        return true;
    }

    if (GcAdapterDriver::IsGcAdapter(deviceParameters.vendorId, deviceParameters.productId))
    {
        return true;
    }
    return false;
}

//!< 指定されたデバイスがポートに割り当てられているか否かを表す値を返します。
bool IsDeviceAssigned(
    const AhidPort& port, ::nn::ahid::hdr::DeviceHandle handle) NN_NOEXCEPT
{
    return (port.flags.Test<AhidPortFlag::IsInitialized>() &&
            port.deviceHandle == handle);
}

AhidPortManager::AhidPortManager() NN_NOEXCEPT
    : m_ActivationCount()
    , m_Filters()
{
    for (AhidPort& port : m_Ports)
    {
        port.flags.Reset();
        new(&::nn::util::Get(port.ahid)) ::nn::ahid::Ahid();
        port.deviceHandle = 0;
        port.pKeyboardDriver = nullptr;
    }

    memset(m_Filters, 0, sizeof(m_Filters));
    m_Filters[0].usagePage = UsagePage::GenericDesktop;
    m_Filters[1].usagePage = UsagePage::Consumer;
    m_Filters[2].usagePage = UsagePage::GameControls;
}

void AhidPortManager::SetTimerEventIn(
    size_t portIndex, SynchronizedTimer* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    m_Ports[portIndex].pTimerEventIn = pTimerEvent;
}

void AhidPortManager::SetTimerEventOut(
    size_t portIndex, SynchronizedTimer* pTimerEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pTimerEvent);
    m_Ports[portIndex].pTimerEventOut = pTimerEvent;
}

void AhidPortManager::SetKeyboardDriver(
    size_t portIndex, KeyboardDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_Ports[portIndex].pKeyboardDriver = pDriver;
}

void AhidPortManager::SetMouseDriver(
    size_t portIndex, MouseDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_Ports[portIndex].pMouseDriver = pDriver;
}

void AhidPortManager::SetAbstractedPadUsbDriver(
    size_t portIndex, AbstractedPadUsbDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_Ports[portIndex].pAbstractedPadUsbDriver = pDriver;
}

void AhidPortManager::SetAudioControlDriver(
    size_t portIndex, AudioControlDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    m_Ports[portIndex].pAudioControlDriver = pDriver;
}

void AhidPortManager::SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);

    m_pGcAdapterDriver = pDriver;
}

::nn::Result AhidPortManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsMax(), ResultAhidActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        NN_RESULT_DO(m_Hdr.InitializeWith(::nn::ahid::hdr::HdrGetDfcSession()));
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AhidPortManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsZero(), ResultAhidDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        for (AhidPort& port : m_Ports)
        {
            if (port.flags.Test<AhidPortFlag::IsInitialized>())
            {
                this->DetachDevice(static_cast<size_t>(&port - m_Ports));
            }
        }

        m_Hdr.Finalize();
    }

    NN_RESULT_SUCCESS;
}

void AhidPortManager::Dispatch() NN_NOEXCEPT
{
    const size_t count =
        this->GetDeviceHandles(m_Handles, NN_ARRAY_SIZE(m_Handles));

    for (AhidPort& port : m_Ports)
    {
        port.flags.Set<AhidPortFlag::IsDirty>();
    }

    for (size_t i = 0; i < count; ++i)
    {
        for (AhidPort& port : m_Ports)
        {
            if (IsDeviceAssigned(port, m_Handles[i]))
            {
                port.flags.Reset<AhidPortFlag::IsDirty>();
            }
        }
    }

    for (AhidPort& port : m_Ports)
    {
        if (port.flags.Test<AhidPortFlag::IsInitialized>())
        {
            if (port.flags.Test<AhidPortFlag::IsInsane>() ||
                port.flags.Test<AhidPortFlag::IsDirty>())
            {
                this->DetachDevice(static_cast<size_t>(&port - m_Ports));
            }
        }
    }

    for (size_t i = 0; i < count; ++i)
    {
        const ::nn::ahid::hdr::DeviceHandle handle = m_Handles[i];

        bool finds = false;

        for (AhidPort& port : m_Ports)
        {
            if (IsDeviceAssigned(port, handle))
            {
                finds = true;
                break;
            }
        }

        if (!finds)
        {
            for (AhidPort& port : m_Ports)
            {
                if (!port.flags.Test<AhidPortFlag::IsInitialized>())
                {
                    this->AttachDevice(
                        static_cast<size_t>(&port - m_Ports), handle);
                    break;
                }
            }
        }
    }
}

void AhidPortManager::ProcessOutPort(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    if (!port.flags.Test<AhidPortFlag::IsInitialized>())
    {
        return;
    }

    if (port.flags.Test<AhidPortFlag::IsInsane>())
    {
        return;
    }

    const uint16_t usagePage = port.deviceParameters.usagePage;
    const uint16_t usageId = port.deviceParameters.usageId;

    if(usagePage == UsagePage::GenericDesktop &&
       usageId == GenericDesktopUsageId::Mouse)
    {
        // 何もしない
    }
    else if (usagePage == UsagePage::GenericDesktop &&
             usageId == GenericDesktopUsageId::Keyboard)
    {
        this->UpdateKeyboardLeds(portIndex);
    }
    else if (usagePage == UsagePage::Consumer)
    {
        //何もしない
    }
    else if(port.flags.Test<AhidPortFlag::IsSupportedGamePad>())
    {
        //何もしない
    }
    else if(port.flags.Test<AhidPortFlag::IsXcdDevice>())
    {
        this->SendOutputReportToNxController(portIndex);
    }
    else if (GcAdapterDriver::IsGcAdapter(port.deviceParameters.vendorId, port.deviceParameters.productId))
    {
        this->SendOutputReportToGcAdapter(portIndex);
    }
}

void AhidPortManager::PrepareInPortContext(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    AhidInPortContext& inPortContext = port.inPortContext;
    inPortContext.deviceHandle = 0;
    inPortContext.result = ::nn::ResultSuccess();
    inPortContext.bytes = 0u;
    inPortContext.buffer = nullptr;
    inPortContext.bufferSize = 0u;

    if (!port.flags.Test<AhidPortFlag::IsInitialized>())
    {
        return;
    }

    inPortContext.deviceHandle = port.deviceHandle;

    const uint16_t usagePage = port.deviceParameters.usagePage;
    const uint16_t usageId = port.deviceParameters.usageId;

    if(usagePage == UsagePage::GenericDesktop &&
       usageId == GenericDesktopUsageId::Mouse)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pCodeBookHeader);
        inPortContext.buffer = port.report.mouse;
        inPortContext.bufferSize = ::std::min<size_t>(
            port.pCodeBookHeader->inputSize, sizeof(port.report.mouse));
    }
    else if (usagePage == UsagePage::GenericDesktop &&
             usageId == GenericDesktopUsageId::Keyboard)
    {
        inPortContext.buffer = &port.report.keyboard;
        inPortContext.bufferSize = sizeof(port.report.keyboard);
    }
    else if (usagePage == UsagePage::Consumer)
    {
        if (usageId == ConsumerUsageId::ConsumerControl)
        {
            // USB スピーカ音量コントロール向けの実装
            NN_SDK_REQUIRES_NOT_NULL(port.pCodeBookHeader);
            inPortContext.buffer = port.report.audioControl;
            inPortContext.bufferSize = ::std::min<size_t>(
                port.pCodeBookHeader->inputSize, sizeof(port.report.audioControl));
        }
    }
    else if(port.flags.Test<AhidPortFlag::IsSupportedGamePad>())
    {
        inPortContext.buffer = &port.report.usbHidDevice;
        inPortContext.bufferSize = sizeof(port.report.usbHidDevice);
    }
    else if(port.flags.Test<AhidPortFlag::IsXcdDevice>())
    {
        inPortContext.buffer = static_cast<void*>(&port.report.externalGrip[0]);
        inPortContext.bufferSize = sizeof(port.report.externalGrip);
    }
    else if (GcAdapterDriver::IsGcAdapter(port.deviceParameters.vendorId, port.deviceParameters.productId))
    {
        inPortContext.buffer = static_cast<void*>(&port.report.gcAdapter[0]);
        inPortContext.bufferSize = sizeof(port.report.gcAdapter);
    }
}

void AhidPortManager::ProcessInPortContext(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    AhidInPortContext& inPortContext = port.inPortContext;

    if (inPortContext.buffer != nullptr)
    {
        inPortContext.result = ::nn::util::Get(port.ahid).Read(
            &inPortContext.bytes,
            inPortContext.buffer,
            inPortContext.bufferSize);
    }
}

void AhidPortManager::ParseInPortContext(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    if (!port.flags.Test<AhidPortFlag::IsInitialized>())
    {
        return;
    }

    AhidInPortContext& inPortContext = port.inPortContext;

    if (inPortContext.deviceHandle != port.deviceHandle)
    {
        return;
    }

    if(port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
       port.deviceParameters.usageId == GenericDesktopUsageId::Mouse)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pMouseDriver);

        if (inPortContext.result.IsFailure())
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }

        MouseReport report = {};

        if (!port.flags.Test<AhidPortFlag::IsInsane>())
        {
            if (!DecodeMouseReport(
                ::nn::util::Get(port.ahid),
                &report, port.report.mouse, &port.items.mouse))
            {
                report = MouseReport();
            }
        }

        port.pMouseDriver->SetReport(portIndex, report);
    }
    else if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
             port.deviceParameters.usageId == GenericDesktopUsageId::Keyboard)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pKeyboardDriver);

        auto report = uint64_t();

        if (inPortContext.result.IsFailure())
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }

        if (!port.flags.Test<AhidPortFlag::IsInsane>() &&
            inPortContext.bytes == sizeof(port.report.keyboard))
        {
            report = port.report.keyboard;
        }

        port.pKeyboardDriver->SetReport(portIndex, report);
    }
    else if(port.deviceParameters.usagePage == UsagePage::Consumer)
    {
        if (port.deviceParameters.usageId == ConsumerUsageId::ConsumerControl)
        {
            NN_SDK_REQUIRES_NOT_NULL(port.pAudioControlDriver);
            if (inPortContext.result.IsFailure())
            {
                port.flags.Set<AhidPortFlag::IsInsane>();
            }

            AudioControlReport report = {};

            if (!port.flags.Test<AhidPortFlag::IsInsane>())
            {
                if (!DecodeAudioControlReport(
                        ::nn::util::Get(port.ahid),
                        &report, port.report.audioControl, &port.items.audioControl))
                {
                    report = AudioControlReport();
                }
            }

            port.pAudioControlDriver->SetReport(portIndex, report);
        }
    }
    else if(port.flags.Test<AhidPortFlag::IsSupportedGamePad>())
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pAbstractedPadUsbDriver);

        if (inPortContext.result.IsFailure())
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }

        if (!port.flags.Test<AhidPortFlag::IsInsane>())
        {
            port.pAbstractedPadUsbDriver->SetReport(
                portIndex, port.report.usbHidDevice, inPortContext.bytes);
        }
    }
    else if(port.flags.Test<AhidPortFlag::IsXcdDevice>())
    {
        if (inPortContext.result.IsFailure())
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }

        if (!port.flags.Test<AhidPortFlag::IsInsane>())
        {
            ::nn::xcd::SetUsbHidInputReport(
                static_cast<int>(portIndex), port.report.externalGrip,
                inPortContext.bytes);
        }
    }
    else if (GcAdapterDriver::IsGcAdapter(port.deviceParameters.vendorId, port.deviceParameters.productId))
    {
        if (inPortContext.result.IsFailure())
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }

        if (!port.flags.Test<AhidPortFlag::IsInsane>())
        {
            m_pGcAdapterDriver->ParseInputReport(
                static_cast<int>(portIndex), port.report.gcAdapter,
                inPortContext.bytes);
        }

    }
} // NOLINT(impl/function_size)

size_t AhidPortManager::GetDeviceHandles(
    ::nn::ahid::hdr::DeviceHandle outHandles[], size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandles);

    auto entryCount = size_t();

    // HID デバイス数の取得
    if (m_Hdr.GetDeviceEntries(&entryCount).IsFailure())
    {
        return 0u;
    }

    if (entryCount == 0u)
    {
        return 0u;
    }

    // フィルタルールが適用されたデバイスを列挙する
    auto amountEntryCount = size_t();
    for(int i=0; i<NN_ARRAY_SIZE(m_Filters); i++)
    {
        if(amountEntryCount >= count)
        {
            break;
        }

        if (m_Hdr.GetDeviceList(
                &entryCount, count - amountEntryCount, outHandles + amountEntryCount, m_Filters + i).IsFailure())
        {
            return 0u;
        }

        amountEntryCount += entryCount;
    }

    return amountEntryCount;
}

::nn::Result AhidPortManager::ActivateXcdDevice(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    ::nn::xcd::UsbHidDeviceInfo xcdDeviceInfo = {};
    xcdDeviceInfo.pid = port.deviceParameters.productId;
    xcdDeviceInfo.vid = port.deviceParameters.vendorId;

    NN_RESULT_DO(
        ::nn::xcd::AddUsbHidDevice(static_cast<int>(portIndex), xcdDeviceInfo));

    NN_RESULT_SUCCESS;
}

void AhidPortManager::AttachDevice(
    size_t portIndex, ::nn::ahid::hdr::DeviceHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];
    NN_SDK_REQUIRES_NOT_NULL(port.pTimerEventIn);
    NN_SDK_REQUIRES_NOT_NULL(port.pTimerEventOut);

    ::nn::ahid::Ahid& ahid = ::nn::util::Get(port.ahid);

    if (m_Hdr.GetDeviceParameters(&port.deviceParameters, handle).IsFailure() ||
        !IsSupportedDeviceParameters(port.deviceParameters) ||
        ahid.InitializeWith(
            ::nn::cdhid::usb::UsbHidGetDfcSession(), &port.deviceParameters
            ).IsFailure())
    {
        return;
    }
    auto needsRollback = true;

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            port.deviceHandle = 0;
            port.flags.Reset();
            port.pCodeBookHeader = nullptr;
            ahid.Finalize();
        }
    };

    if (ahid.GetCodeBookHeader(&port.pCodeBookHeader).IsFailure() ||
        port.pCodeBookHeader == nullptr)
    {
        return;
    }

    port.flags.Reset();
    port.flags.Set<AhidPortFlag::IsInitialized>();
    port.deviceHandle = handle;

    if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
        port.deviceParameters.usageId == GenericDesktopUsageId::Mouse)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pMouseDriver);

        auto reportId = uint8_t();

        const uint16_t usageIds[] = {
            GenericDesktopUsageId::Pointer,
            GenericDesktopUsageId::Mouse
        };

        if (!GetReportId(ahid, &reportId, usageIds) ||
            !GetMouseCodeBookItems(ahid, &port.items.mouse, reportId))
        {
            return;
        }

        port.pMouseDriver->Attach(portIndex);
    }
    else if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
             port.deviceParameters.usageId == GenericDesktopUsageId::Keyboard)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pKeyboardDriver);

        if (ahid.SetProtocol(0).IsFailure() || ahid.SetIdle(0, 0).IsFailure())
        {
            return;
        }

        port.pKeyboardDriver->Attach(portIndex);
    }
    else if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
             port.deviceParameters.usageId == GenericDesktopUsageId::GamePad &&
             IsSupportedUsbPad(
                 port.deviceParameters.vendorId,
                 port.deviceParameters.productId))
    {
        port.flags.Set<AhidPortFlag::IsSupportedGamePad>();
        port.pAbstractedPadUsbDriver->Attach(portIndex, static_cast<uint16_t>(port.deviceParameters.vendorId), static_cast<uint16_t>(port.deviceParameters.productId));
    }
    else if (port.deviceParameters.usagePage == UsagePage::Consumer)
    {
        if (port.deviceParameters.usageId == ConsumerUsageId::ConsumerControl)
        {

            NN_SDK_REQUIRES_NOT_NULL(port.pAudioControlDriver);

            auto reportId = uint8_t();

            const uint16_t usageIds[] = {
                ConsumerUsageId::ConsumerControl
            };

            if (!GetReportId(ahid, &reportId, usageIds) ||
                !GetAudioControlCodeBookItems(ahid, &port.items.audioControl, reportId))
            {
                return;
            }
            port.pAudioControlDriver->Attach(portIndex);
        }
    }
    else if (this->ActivateXcdDevice(portIndex).IsSuccess())
    {
        port.flags.Set<AhidPortFlag::IsXcdDevice>();
    }
    else if (GcAdapterDriver::IsGcAdapter(port.deviceParameters.vendorId, port.deviceParameters.productId))
    {
        m_pGcAdapterDriver->Attach(portIndex, ::nn::util::Get(port.ahid));
    }
    port.pTimerEventIn->Enable(AhidManager::SamplingInterval);
    port.pTimerEventOut->Enable(AhidManager::SamplingInterval);

    needsRollback = false;
}

void AhidPortManager::DetachDevice(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];
    NN_SDK_REQUIRES_NOT_NULL(port.pTimerEventIn);
    NN_SDK_REQUIRES_NOT_NULL(port.pTimerEventOut);

    port.pTimerEventIn->Disable();
    port.pTimerEventOut->Disable();

    ::nn::util::Get(port.ahid).Finalize();

    port.leds = 0;

    if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
        port.deviceParameters.usageId == GenericDesktopUsageId::Mouse)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pMouseDriver);

        port.pMouseDriver->Detach(portIndex);
    }
    else if (port.deviceParameters.usagePage == UsagePage::GenericDesktop &&
             port.deviceParameters.usageId == GenericDesktopUsageId::Keyboard)
    {
        NN_SDK_REQUIRES_NOT_NULL(port.pKeyboardDriver);

        port.pKeyboardDriver->Detach(portIndex);
    }
    else if (port.deviceParameters.usagePage == UsagePage::Consumer)
    {
        if (port.deviceParameters.usageId == ConsumerUsageId::ConsumerControl)
        {
            NN_SDK_REQUIRES_NOT_NULL(port.pAudioControlDriver);

            port.pAudioControlDriver->Detach(portIndex);
        }
    }
    else if(port.flags.Test<AhidPortFlag::IsSupportedGamePad>())
    {
        port.pAbstractedPadUsbDriver->Detach(portIndex);
    }
    else if(port.flags.Test<AhidPortFlag::IsXcdDevice>())
    {
        ::nn::xcd::RemoveUsbHidDevice(static_cast<int>(portIndex));
    }
    else if (GcAdapterDriver::IsGcAdapter(port.deviceParameters.vendorId, port.deviceParameters.productId))
    {
        m_pGcAdapterDriver->Detach(portIndex);
    }

    port.deviceHandle = 0;
    port.flags.Reset();
    port.pCodeBookHeader = nullptr;
}

void AhidPortManager::UpdateKeyboardLeds(size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    NN_SDK_REQUIRES_NOT_NULL(port.pKeyboardDriver);

    NN_ALIGNAS(8) uint8_t leds = port.pKeyboardDriver->GetLedPattern();

    if (port.leds != leds)
    {
        const nn::Result result = ::nn::util::Get(port.ahid).SetReport(
            &leds, sizeof(leds), DescriptorType::Configuration, 0);

        if (result.IsSuccess())
        {
            port.leds = leds;
        }
        else
        {
            port.flags.Set<AhidPortFlag::IsInsane>();
        }
    }
}

void AhidPortManager::SendOutputReportToNxController(
    size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    uint8_t buffer[::nn::xcd::UsbHidReportLengthMax] = {};

    const size_t bytesToSend =
        ::nn::xcd::GetUsbHidOutputReport(
            static_cast<int>(portIndex), buffer, sizeof(buffer));

    if (bytesToSend != 0)
    {
        auto bytesSent = uint32_t();
        nn::Result result = ::nn::util::Get(port.ahid)
            .WriteWithTimeout(
                &bytesSent, reinterpret_cast<void*>(buffer),
                bytesToSend, nn::TimeSpan::FromMilliSeconds(15));

        // timeout is only a little bit crazy, not yet insane
        if (result.IsFailure())
        {
            if (nn::ahid::ResultTimeout::Includes(result))
            {
                static int s_TimeoutCount = 0;
                s_TimeoutCount++;
                NN_DETAIL_HID_TRACE("AhidPortManager::SendOutputReportToNxController() - Ahid Write Timed Out %d times.\n", s_TimeoutCount);
            }
            else
            {
                port.flags.Set<AhidPortFlag::IsInsane>();
            }
        }
    }
}

void AhidPortManager::SendOutputReportToGcAdapter(
    size_t portIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(portIndex, 0u, AhidPortCount);

    AhidPort& port = m_Ports[portIndex];

    auto success = m_pGcAdapterDriver->SendOutputReport(portIndex);

    if (success == false)
    {
        port.flags.Set<AhidPortFlag::IsInsane>();
    }
}

AhidPortManager& GetAhidPortManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(AhidPortManager, s_AhidPortManager);
    return s_AhidPortManager;
}

} // namespace

}}} // namespace nn::hid::detail
