﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_TickTypes.h>
#include <nn/xcd/xcd.h>

#include "hid_IAbstractedPad.h"

namespace nn { namespace hid { namespace detail {

class AbstractedPalmaPad final : public IAbstractedPad
{
private:
    static const AbstractedPadType s_Type = AbstractedPadType_Palma;

    //!< コントローラーの接続状態
    bool m_IsConnected;

    //!< コントローラーの入力状態
    AbstractedPadState m_PadState;

    //!< Xcd のデバイスハンドル
    nn::xcd::BleConnectionHandle m_XcdHandle;

    //!< Bluetooth アドレス
    nn::bluetooth::Address m_Address;

    //!< 6軸センサーのキャリブレーション値
    nn::xcd::SensorCalibrationValue m_SensorCalibrationValue;

    //!< インジケーターパターン
    uint8_t m_Pattern;

    //!< ケーブル接続状態
    bool m_IsWired;

    //!< 最後に Read したサンプルの番号
    int64_t m_LastSampleNumber;

    //!< 右Stickの Cal
    ::nn::xcd::AnalogStickValidRange m_RightRange;

    //!< 左Stickの Cal
    ::nn::xcd::AnalogStickValidRange m_LeftRange;


public:
    AbstractedPalmaPad() NN_NOEXCEPT;
    virtual ~AbstractedPalmaPad() NN_NOEXCEPT NN_OVERRIDE;

    //!< AbstractedPadType を取得する
    virtual AbstractedPadType GetType() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態を取得する
    virtual bool IsConnected() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの電源状態を取得する
    virtual system::PowerInfo GetPowerInfo() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスのボタン/アナログスティックの入力状態を取得する
    virtual AbstractedPadState GetPadState() NN_NOEXCEPT NN_OVERRIDE;

    //!< 最後にボタンが押下された時間を取得する
    virtual bool GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーの色情報を取得する
    virtual system::ControllerColor GetColor() NN_NOEXCEPT NN_OVERRIDE;

    //!< 6軸センサーの値を取得する。暫定的に xcd の構造体を利用する
    virtual int GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT NN_OVERRIDE;

    //!< 6軸センサーの CAL 値を取得する。暫定的に xcd の構造体を利用する。有効な値がない場合は false を返す
    virtual bool GetSensorCalibrationValue(nn::xcd::SensorCalibrationValue* pOutValue) NN_NOEXCEPT NN_OVERRIDE;

    //!< 6軸センサーのサンプル間隔を取得する
    virtual nn::TimeSpan GetSensorSamplingInterval() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーのインジケーターを制御する
    virtual void SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーのインジケーターパターンを取得する
    virtual uint8_t GetIndicator() NN_NOEXCEPT NN_OVERRIDE;

    //!< ケーブル接続されているかどうか
    virtual bool IsWired() NN_NOEXCEPT NN_OVERRIDE;

    //!< Usb の接続状態を取得します
    virtual bool IsUsbConnected() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーを接続状態に変更します
    virtual void Connect() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーを切断する
    virtual void Detach() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーを再起動する
    virtual void Reboot(bool reconnect) NN_NOEXCEPT NN_OVERRIDE;

    //!< 接続時の振動を行う
    virtual void VibrationOnConnect() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの内部状態をリセットします
    virtual void ResetInternalDeviceState() NN_NOEXCEPT NN_OVERRIDE;

    //!< バッテリーを搭載しているかどうか取得します
    virtual bool HasBattery() NN_NOEXCEPT NN_OVERRIDE;

    //!< プレイレポート用のデフォルトの情報を取得します
    virtual bool SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT NN_OVERRIDE;

    //!< AbstractedPad の入力状態を取得します。
    virtual void GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT NN_OVERRIDE;

    //!< 6軸センサーの状態を取得します。
    virtual int GetSixAxisSensorDriverStates(SixAxisSensorDriverState* pOutStates,
                                             int count) NN_NOEXCEPT NN_OVERRIDE;

    //!< パケットの受信履歴を取得します。
    virtual RxPacketHistory GetRxPacketHistory() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスを追加する
    void AttachDevice(::nn::xcd::BleConnectionHandle handle, AbstractedPadId id) NN_NOEXCEPT;

    //!< デバイスを削除する
    void RemoveDevice() NN_NOEXCEPT;

    //!< Xcd のデバイスハンドルを取得する
    ::nn::xcd::BleConnectionHandle GetXcdDeviceHandle() NN_NOEXCEPT;

    //!< Pad の内部状態を更新する
    void Update() NN_NOEXCEPT;

    //!< Bluetooth アドレスを取得する
    ::nn::bluetooth::Address GetAddress() NN_NOEXCEPT;

private:
    //!< デバイスの入力状態を更新する
    void UpdatePadState() NN_NOEXCEPT;

    //!< デバイスの電源状態を更新する
    void UpdatePowerInfo() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
