﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include <nn/nn_SdkLog.h>

#include "hid_AbstractedPalmaPad.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Gc コントローラー共通の色
const system::ControllerColor StaticPalmaControllerColor = { {{0xff, 0x00, 0x00, 0xff}}, {{0xff, 0xff, 0xff, 0xff}} };

} // namespace

AbstractedPalmaPad::AbstractedPalmaPad() NN_NOEXCEPT
    : m_IsConnected(false)
{
    // 何もしない
}

AbstractedPalmaPad::~AbstractedPalmaPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedPalmaPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedPalmaPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedPalmaPad::GetPowerInfo() NN_NOEXCEPT
{
    return m_PowerInfo;
}

AbstractedPadState AbstractedPalmaPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedPalmaPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン検出をさかのぼる機能は提供しない
    return false;
}

system::ControllerColor AbstractedPalmaPad::GetColor() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return system::ControllerColor();
    }
    return StaticPalmaControllerColor;
}

int AbstractedPalmaPad::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    int gotCount = 0;
    if (nn::xcd::GetSensorStates(&gotCount, pOutValue, count, m_XcdHandle).IsFailure())
    {
        return 0;
    }
    return gotCount;
}

bool AbstractedPalmaPad::GetSensorCalibrationValue(nn::xcd::SensorCalibrationValue* pOutValue) NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    *pOutValue = m_SensorCalibrationValue;
    return true;
}

nn::TimeSpan AbstractedPalmaPad::GetSensorSamplingInterval() NN_NOEXCEPT
{
    return nn::TimeSpan::FromMilliSeconds(15);
}

void AbstractedPalmaPad::Connect() NN_NOEXCEPT
{
    // Connect 機能には対応しない
}

void AbstractedPalmaPad::Detach() NN_NOEXCEPT
{
    if (m_IsAttached)
    {
        m_IsConnected = false;
        ::nn::xcd::Detach(m_XcdHandle);
    }
}

void AbstractedPalmaPad::Reboot(bool reconnect) NN_NOEXCEPT
{
    NN_UNUSED(reconnect);

    if (m_IsConnected)
    {
        // TODO: 実装追加
    }
}

void AbstractedPalmaPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedPalmaPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

void AbstractedPalmaPad::VibrationOnConnect() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        ::nn::xcd::PlayDeviceSystemActivity(2, m_XcdHandle);
    }
}

bool AbstractedPalmaPad::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    return m_IsWired;
}


void AbstractedPalmaPad::ResetInternalDeviceState() NN_NOEXCEPT
{
    // 何もしない
    // USBからBTへの通信切り替えなど、内部のデバイスをいったんリセットする場合の処理を記載する
}

bool AbstractedPalmaPad::IsUsbConnected() NN_NOEXCEPT
{
    return false;
}

bool AbstractedPalmaPad::HasBattery() NN_NOEXCEPT
{
    return true;
}

bool AbstractedPalmaPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    // デバイスタイプの変換
    pOutValue->deviceType = system::PlayReportDeviceType_Palma;
    pOutValue->nxControllerInfo.interfaceType = m_InterfaceType;
    pOutValue->nxControllerInfo.subType = 0;  // サブタイプは現状設定しない
    return true;
}

void AbstractedPalmaPad::GetPadDriverState(PadDriverState* pOutState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);

    nn::xcd::PadState padState;
    if (nn::xcd::GetPadState(&padState, m_XcdHandle).IsFailure())
    {
        // エラーの場合は無入力を返す
        pOutState->buttons.Reset();
        pOutState->analogStickL = AnalogStickDriverState();
        pOutState->analogStickR = AnalogStickDriverState();
        return;
    }

    pOutState->sampleNumber = padState.sampleNumber;

    //!< Button の計算
    pOutState->buttons.Reset();
    pOutState->buttons.Set<PadButton::A>(padState.buttons.Test<nn::xcd::PadButton::A>());
    pOutState->buttons.Set<PadButton::B>(padState.buttons.Test<nn::xcd::PadButton::B>());
    pOutState->buttons.Set<PadButton::X>(padState.buttons.Test<nn::xcd::PadButton::X>());
    pOutState->buttons.Set<PadButton::Y>(padState.buttons.Test<nn::xcd::PadButton::Y>());
    pOutState->buttons.Set<PadButton::StickL>(padState.buttons.Test<nn::xcd::PadButton::StickL>());
    pOutState->buttons.Set<PadButton::StickR>(padState.buttons.Test<nn::xcd::PadButton::StickR>());
    pOutState->buttons.Set<PadButton::L>(padState.buttons.Test<nn::xcd::PadButton::L>());
    pOutState->buttons.Set<PadButton::R>(padState.buttons.Test<nn::xcd::PadButton::R>());
    pOutState->buttons.Set<PadButton::ZL>(padState.buttons.Test<nn::xcd::PadButton::ZL>());
    pOutState->buttons.Set<PadButton::ZR>(padState.buttons.Test<nn::xcd::PadButton::ZR>());
    pOutState->buttons.Set<PadButton::Start>(padState.buttons.Test<nn::xcd::PadButton::Start>());
    pOutState->buttons.Set<PadButton::Select>(padState.buttons.Test<nn::xcd::PadButton::Select>());
    pOutState->buttons.Set<PadButton::Left>(padState.buttons.Test<nn::xcd::PadButton::Left>());
    pOutState->buttons.Set<PadButton::Up>(padState.buttons.Test<nn::xcd::PadButton::Up>());
    pOutState->buttons.Set<PadButton::Right>(padState.buttons.Test<nn::xcd::PadButton::Right>());
    pOutState->buttons.Set<PadButton::Down>(padState.buttons.Test<nn::xcd::PadButton::Down>());
    pOutState->buttons.Set<PadButton::SL>(padState.buttons.Test<nn::xcd::PadButton::SL>());
    pOutState->buttons.Set<PadButton::SR>(padState.buttons.Test<nn::xcd::PadButton::SR>());
    pOutState->buttons.Set<PadButton::Home>(padState.buttons.Test<nn::xcd::PadButton::Home>());
    pOutState->buttons.Set<PadButton::Shot>(padState.buttons.Test<nn::xcd::PadButton::Shot>());

    //!< Analog Stick の計算
    pOutState->analogStickR.x = padState.analogStickR.x;
    pOutState->analogStickR.y = padState.analogStickR.y;
    pOutState->analogStickL.x = padState.analogStickL.x;
    pOutState->analogStickL.y = padState.analogStickL.y;
}

int AbstractedPalmaPad::GetSixAxisSensorDriverStates(SixAxisSensorDriverState* pOutStates,
                                                   int count) NN_NOEXCEPT
{
    int gotCount = 0;
    nn::xcd::SixAxisSensorState states[nn::xcd::SixAxisSensorSampleCountMax];
    if (nn::xcd::GetSensorStates(&gotCount, states, NN_ARRAY_SIZE(states), m_XcdHandle).IsFailure())
    {
        return 0;
    }
    gotCount = std::min(gotCount, count);
    for (int i = 0; i < gotCount; ++i)
    {
        pOutStates[i].accelerometer.x = static_cast<int32_t>(states[i].accelerometer.x);
        pOutStates[i].accelerometer.y = static_cast<int32_t>(states[i].accelerometer.y);
        pOutStates[i].accelerometer.z = static_cast<int32_t>(states[i].accelerometer.z);

        pOutStates[i].gyroscope.x = static_cast<int32_t>(states[i].gyroscope.x);
        pOutStates[i].gyroscope.y = static_cast<int32_t>(states[i].gyroscope.y);
        pOutStates[i].gyroscope.z = static_cast<int32_t>(states[i].gyroscope.z);

        pOutStates[i].sampleNumber = states[i].sampleNumber;
    }
    return gotCount;
}

RxPacketHistory AbstractedPalmaPad::GetRxPacketHistory() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return RxPacketHistory();
    }
    return nn::xcd::GetRxPacketHistory(m_XcdHandle);
}

void AbstractedPalmaPad::AttachDevice(::nn::xcd::BleConnectionHandle handle, AbstractedPadId id) NN_NOEXCEPT
{
    NN_UNUSED(handle);

    m_XcdHandle = handle;
    m_Id = id;
    m_LastSampleNumber = std::numeric_limits<int64_t>::min();
    m_IsNewSampleReceived = false;

    ::nn::xcd::BleDeviceInfo info;
    nn::xcd::PadState state;
    if (::nn::xcd::GetDeviceInfo(&info, handle).IsFailure() ||
        ::nn::xcd::GetPadState(&state, handle).IsFailure() ||
        ::nn::xcd::GetSensorCalibrationValue(&m_SensorCalibrationValue, handle).IsFailure()
        )
    {
        // デバイス情報が正しく読み出せないので切断扱い
        return;
    }

    m_InterfaceType = system::InterfaceType_Bluetooth;

    switch (info.deviceType)
    {
    case nn::xcd::BleDeviceType_Palma:
        m_DeviceType = system::DeviceType::Palma::Mask;
        break;
    default:
        m_DeviceType = system::DeviceType::Unknown::Mask;
    }
    m_Address = info.address;

    // CAL 値の埋め込み
    ::nn::xcd::GetAnalogStickValidRange(&m_RightRange, handle);
    ::nn::xcd::GetAnalogStickValidRange(&m_LeftRange, handle);
    // 原点が入っていない場合は、取得位置を原点とする
    if (m_RightRange.origin.x == 0 && m_RightRange.origin.y == 0)
    {
        m_RightRange.origin = state.analogStickR;
    }
    if (m_LeftRange.origin.x == 0 && m_LeftRange.origin.y == 0)
    {
        m_LeftRange.origin = state.analogStickL;
    }

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
    m_FeatureSet.Set<AbstractedPadFeature::SixAxisSensorCenter>();

    m_IsAttached = true;
    m_IsConnected = true;
}

void AbstractedPalmaPad::RemoveDevice() NN_NOEXCEPT
{
    m_XcdHandle = 0; // FIXME
    m_Address = nn::bluetooth::Address();
    m_Id = AbstractedPadId();
    m_IsWired = false;
    m_IsConnected = false;
    m_IsAttached = false;
}

::nn::xcd::BleConnectionHandle AbstractedPalmaPad::GetXcdDeviceHandle() NN_NOEXCEPT
{
    return m_XcdHandle;
}

void AbstractedPalmaPad::Update() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }
    // 入力状態の更新
    UpdatePadState();

    // PowerInfo の更新
    UpdatePowerInfo();
}

::nn::bluetooth::Address AbstractedPalmaPad::GetAddress() NN_NOEXCEPT
{
    return m_Address;
}

void AbstractedPalmaPad::UpdatePadState() NN_NOEXCEPT
{
    nn::xcd::PadState padState;
    if (nn::xcd::GetPadState(&padState, m_XcdHandle).IsFailure())
    {
        // 入力状態を更新しない
        return;
    }

    // 新しいサンプルを受信したときのみ更新
    if (m_LastSampleNumber == padState.sampleNumber)
    {
        return;
    }
    m_LastSampleNumber = padState.sampleNumber;
    m_IsNewSampleReceived = true;

    //!< Button の計算
    m_PadState.buttons.Reset();
    m_PadState.buttons.Set<AbstractedPadButton::A>(padState.buttons.Test<nn::xcd::PadButton::A>());
    m_PadState.buttons.Set<AbstractedPadButton::B>(padState.buttons.Test<nn::xcd::PadButton::B>());
    m_PadState.buttons.Set<AbstractedPadButton::X>(padState.buttons.Test<nn::xcd::PadButton::X>());
    m_PadState.buttons.Set<AbstractedPadButton::Y>(padState.buttons.Test<nn::xcd::PadButton::Y>());
    m_PadState.buttons.Set<AbstractedPadButton::StickL>(padState.buttons.Test<nn::xcd::PadButton::StickL>());
    m_PadState.buttons.Set<AbstractedPadButton::StickR>(padState.buttons.Test<nn::xcd::PadButton::StickR>());
    m_PadState.buttons.Set<AbstractedPadButton::L>(padState.buttons.Test<nn::xcd::PadButton::L>());
    m_PadState.buttons.Set<AbstractedPadButton::R>(padState.buttons.Test<nn::xcd::PadButton::R>());
    m_PadState.buttons.Set<AbstractedPadButton::ZL>(padState.buttons.Test<nn::xcd::PadButton::ZL>());
    m_PadState.buttons.Set<AbstractedPadButton::ZR>(padState.buttons.Test<nn::xcd::PadButton::ZR>());
    m_PadState.buttons.Set<AbstractedPadButton::Start>(padState.buttons.Test<nn::xcd::PadButton::Start>());
    m_PadState.buttons.Set<AbstractedPadButton::Select>(padState.buttons.Test<nn::xcd::PadButton::Select>());
    m_PadState.buttons.Set<AbstractedPadButton::Left>(padState.buttons.Test<nn::xcd::PadButton::Left>());
    m_PadState.buttons.Set<AbstractedPadButton::Up>(padState.buttons.Test<nn::xcd::PadButton::Up>());
    m_PadState.buttons.Set<AbstractedPadButton::Right>(padState.buttons.Test<nn::xcd::PadButton::Right>());
    m_PadState.buttons.Set<AbstractedPadButton::Down>(padState.buttons.Test<nn::xcd::PadButton::Down>());
    m_PadState.buttons.Set<AbstractedPadButton::SL>(padState.buttons.Test<nn::xcd::PadButton::SL>());
    m_PadState.buttons.Set<AbstractedPadButton::SR>(padState.buttons.Test<nn::xcd::PadButton::SR>());
    m_PadState.buttons.Set<AbstractedPadButton::Home>(padState.buttons.Test<nn::xcd::PadButton::Home>());
    m_PadState.buttons.Set<AbstractedPadButton::Shot>(padState.buttons.Test<nn::xcd::PadButton::Shot>());
    // A に割り当たっている Palma の Top ボタンを Palma ボタンに割り当てる
    m_PadState.buttons.Set<AbstractedPadButton::Palma>(padState.buttons.Test<nn::xcd::PadButton::A>());

    //!< Analog Stick の計算
    m_PadState.analogStickR = ConvertXcdAnalogStickState(padState.analogStickR, m_RightRange);
    m_PadState.analogStickL = ConvertXcdAnalogStickState(padState.analogStickL, m_LeftRange);

    //!< Analog Stick のY軸を反転する
    ConvertPalmaAxis(&m_PadState.analogStickR);
    ConvertPalmaAxis(&m_PadState.analogStickL);

    SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);
}

void AbstractedPalmaPad::UpdatePowerInfo() NN_NOEXCEPT
{
    nn::xcd::DeviceStatus status;
    if (nn::xcd::GetDeviceStatus(&status, m_XcdHandle).IsFailure())
    {
        // 電源状態を更新しない
        return;
    }

    m_PowerInfo.batteryLevel = static_cast<system::BatteryLevel>(status.batteryLevel);
    m_PowerInfo.isCharging = status.charged;
    m_PowerInfo.isPowered = status.powered;

    // ケーブル接続状態もあわせて更新
    m_IsWired = status.cablePlugged;
}

}}} // namespace nn::hid::detail
