﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadUsbDriver.h"

namespace nn { namespace hid { namespace detail {

AbstractedPadUsbDriver::AbstractedPadUsbDriver() NN_NOEXCEPT
    : m_ActivationCount()
{
}

AbstractedPadUsbDriver::~AbstractedPadUsbDriver() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedPadUsbDriver::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pIdPublisher = pPublisher;
}

void AbstractedPadUsbDriver::SetAbstractedPadUsbs(AbstractedPadUsb* pPads, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_MINMAX(count, 1, static_cast<int>(PortCount));
    for (int padIndex = 0; padIndex < count; padIndex++)
    {
        m_pPads[padIndex] = &pPads[padIndex];
    }
    m_PadCount = count;
}

::nn::Result AbstractedPadUsbDriver::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPads);
    NN_SDK_REQUIRES_GREATER(m_PadCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(m_pIdPublisher);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result AbstractedPadUsbDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void AbstractedPadUsbDriver::Attach(size_t port, uint16_t vid, uint16_t pid) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    // あいている AbstractedPadUsb を探索
    for (int i = 0; i < m_PadCount; i++)
    {
        if (m_pPads[i]->IsAttached() == false)
        {
            m_pPads[i]->AttachDevice(port, m_pIdPublisher->PublishId(), vid, pid);
            return;
        }
    }
}

void AbstractedPadUsbDriver::Detach(size_t port) NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    // あいている AbstractedPadUsb を探索
    for (int i = 0; i < m_PadCount; i++)
    {
        if (m_pPads[i]->IsAttached() == true && m_pPads[i]->GetPort() == port)
        {
            m_pPads[i]->RemoveDevice();
            return;
        }
    }
}

void AbstractedPadUsbDriver::SetReport(size_t port, const uint8_t* pReport, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReport);

    if (m_ActivationCount.IsZero() || length == 0)
    {
        return;
    }

    // あいている AbstractedPadUsb を探索
    for (int i = 0; i < m_PadCount; i++)
    {
        if (m_pPads[i]->IsAttached() == true && m_pPads[i]->GetPort() == port)
        {
            m_pPads[i]->SetReport(pReport, length);
            return;
        }
    }
}

}}} // namespace nn::hid::detail
