﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/hid/debug/hid_AbstractedPad.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>

#include "hid_AbstractedAutoPilotVirtualPadManager.h"
#include "hid_AbstractedPadManager.h"
#include "hid_HashTable.h"
#include "hid_IAbstractedPad.h"
#include "hid_Vector.h"

namespace nn { namespace hid { namespace detail {

//!< AbstractedPad の HidShell 対応を担うマネージャを扱うクラスです。
class AbstractedPadHidShellManager final
{
    NN_DISALLOW_COPY(AbstractedPadHidShellManager);
    NN_DISALLOW_MOVE(AbstractedPadHidShellManager);

private:
    //!< AbstractedPad の数の最大値
    static const int PadCountMax =
        ::nn::hid::debug::AbstractedPadHandleCountMax;

    //!< AbstractedPad の Id のハッシュ値計算を扱う構造体です。
    struct IdHasher final
    {
        //!< AbstractedPad の Id のハッシュ値を返します。
        static size_t Get(const AbstractedPadId& id) NN_NOEXCEPT
        {
            return static_cast<size_t>(id.value);
        }
    };

    //!< AbstractedPad のマネージャー
    AbstractedPadManager* m_pAbstractedPadManager = nullptr;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotVirtualPadManager* m_pVirtualPadManager;

    //!< AbstractedPad の Id とインデックス値の対応表
    HashTable<AbstractedPadId, size_t, IdHasher, PadCountMax> m_Table;

    //!< HidShell の利用可能なインデックス値
    Vector<size_t, PadCountMax> m_Indices;

    //!< AbstractedPad の最後の Id
    Vector<AbstractedPadId, PadCountMax> m_LastIds;

    //!< AbstractedPad の最新の Id
    AbstractedPadId m_NextIds[PadCountMax];

    //!< AbstractedPad の入力状態
    ::nn::hid::debug::AbstractedPadState m_State;

    //!< AbstractedPad の入力状態の一時的な保存領域
    AbstractedPadState m_Temp;

public:
    AbstractedPadHidShellManager() NN_NOEXCEPT;

    //!< AbstractedPad のマネージャーを設定します。
    void SetAbstractedPadManager(AbstractedPadManager* pManager) NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedAutoPilotVirtualPadManager(
        AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT;

    //!< HidShell から AbstractedPad の状態を受信します。
    void Recieve() NN_NOEXCEPT;

    //!< HidShell から AbstractedPad の状態を送信します。
    void Send() NN_NOEXCEPT;

private:
    //!< HidShell の利用可能なインデックス値をリセットします。
    void ResetIndices() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
