﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd_Fifty.h>

#include "hid_AbstractedFiftyPad.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Fifty コントローラー共通の電源状態
const system::PowerInfo StaticUsbPowerInfo = { true, false, system::BatteryLevel_High };

} // namespace

AbstractedFiftyPad::AbstractedFiftyPad() NN_NOEXCEPT
    : m_pIdPublisher(nullptr)
    , m_IsConnected(false)
    , m_Pattern(0)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Rail;
    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::HandheldJoyLeft::Mask;

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
}

AbstractedFiftyPad::~AbstractedFiftyPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedFiftyPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedFiftyPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedFiftyPad::GetPowerInfo() NN_NOEXCEPT
{
    return StaticUsbPowerInfo;
}

AbstractedPadState AbstractedFiftyPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedFiftyPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

void AbstractedFiftyPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedFiftyPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedFiftyPad::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // 有線接続は常時 Wired 状態
    return true;
}

bool AbstractedFiftyPad::IsUsbConnected() NN_NOEXCEPT
{
    return false;
}

void AbstractedFiftyPad::Connect() NN_NOEXCEPT
{
    // Connect は自動的に行われる
}

void AbstractedFiftyPad::Detach() NN_NOEXCEPT
{
    // Detach できない
}

bool AbstractedFiftyPad::HasBattery() NN_NOEXCEPT
{
    return false;
}

bool AbstractedFiftyPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    // TODO : Fifty 用を追加する
    pOutValue->deviceType = system::PlayReportDeviceType_JoyConLeft;
    for (auto& byte : pOutValue->raw)
    {
        byte = 0;
    }
    return true;
}

void AbstractedFiftyPad::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    m_pIdPublisher = pPublisher;
}

void AbstractedFiftyPad::Update() NN_NOEXCEPT
{
    if (nn::xcd::IsFiftyConnected() == false)
    {
        m_IsConnected = false;
        m_PadState = AbstractedPadState();
        return;
    }

    if (m_IsConnected == false)
    {
        m_Id = m_pIdPublisher->PublishId();
        m_IsConnected = true;
    }

    auto xcdState = nn::xcd::GetFiftyPadState();

    //!< Button の計算
    m_PadState.buttons.Reset();
    m_PadState.buttons.Set<AbstractedPadButton::A>(xcdState.buttons.Test<nn::xcd::PadButton::A>());
    m_PadState.buttons.Set<AbstractedPadButton::B>(xcdState.buttons.Test<nn::xcd::PadButton::B>());
    m_PadState.buttons.Set<AbstractedPadButton::X>(xcdState.buttons.Test<nn::xcd::PadButton::X>());
    m_PadState.buttons.Set<AbstractedPadButton::Y>(xcdState.buttons.Test<nn::xcd::PadButton::Y>());
    m_PadState.buttons.Set<AbstractedPadButton::StickL>(xcdState.buttons.Test<nn::xcd::PadButton::StickL>());
    m_PadState.buttons.Set<AbstractedPadButton::StickR>(xcdState.buttons.Test<nn::xcd::PadButton::StickR>());
    m_PadState.buttons.Set<AbstractedPadButton::L>(xcdState.buttons.Test<nn::xcd::PadButton::L>());
    m_PadState.buttons.Set<AbstractedPadButton::R>(xcdState.buttons.Test<nn::xcd::PadButton::R>());
    m_PadState.buttons.Set<AbstractedPadButton::ZL>(xcdState.buttons.Test<nn::xcd::PadButton::ZL>());
    m_PadState.buttons.Set<AbstractedPadButton::ZR>(xcdState.buttons.Test<nn::xcd::PadButton::ZR>());
    m_PadState.buttons.Set<AbstractedPadButton::Start>(xcdState.buttons.Test<nn::xcd::PadButton::Start>());
    m_PadState.buttons.Set<AbstractedPadButton::Select>(xcdState.buttons.Test<nn::xcd::PadButton::Select>());
    m_PadState.buttons.Set<AbstractedPadButton::Left>(xcdState.buttons.Test<nn::xcd::PadButton::Left>());
    m_PadState.buttons.Set<AbstractedPadButton::Up>(xcdState.buttons.Test<nn::xcd::PadButton::Up>());
    m_PadState.buttons.Set<AbstractedPadButton::Right>(xcdState.buttons.Test<nn::xcd::PadButton::Right>());
    m_PadState.buttons.Set<AbstractedPadButton::Down>(xcdState.buttons.Test<nn::xcd::PadButton::Down>());
    m_PadState.buttons.Set<AbstractedPadButton::SL>(xcdState.buttons.Test<nn::xcd::PadButton::SL>());
    m_PadState.buttons.Set<AbstractedPadButton::SR>(xcdState.buttons.Test<nn::xcd::PadButton::SR>());
    m_PadState.buttons.Set<AbstractedPadButton::Home>(xcdState.buttons.Test<nn::xcd::PadButton::Home>());
    m_PadState.buttons.Set<AbstractedPadButton::Shot>(xcdState.buttons.Test<nn::xcd::PadButton::Shot>());

    //!< Analog Stick の計算
    nn::xcd::AnalogStickValidRange range =
    {
        { 2048, 2048 },
        { 2048, 2048 },
        { 2048, 2048 },
        0
    };
    m_PadState.analogStickR = ConvertXcdAnalogStickState(xcdState.analogStickR, range);
    m_PadState.analogStickL = ConvertXcdAnalogStickState(xcdState.analogStickL, range);

    SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);
}

}}} // namespace nn::hid::detail
