﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grcsrv/grcsrv_Offscreen.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/lmem/lmem_FrameHeap.h>
#include <nn/mem.h>
#include <nn/vi/vi_Result.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nv/nv_MemoryManagement.h>
#include <mm_MemoryManagement.h>

#include "grcsrvOffscreen_Config.h"
#include "grcsrvOffscreen_Macro.h"
#include "grcsrvOffscreen_LibraryState.h"
#include "grcsrvOffscreen_ConvertCapsrvError.h"
#include "grcsrvOffscreen_VicCopyCaptureImageBuffer.h"
#include "native/grcsrv_ParcelIo.h"
#include "native/grcsrv_SyncpointWaiter.h"
#include <nn/grcsrv/offscreen/grcsrvOffscreen_CheckPointForTesting.h>

namespace nerd{ namespace scoop{
    nn::Result InitializeResourcesOnce();
}}

namespace nn{ namespace grcsrv{
    void SetStandardAllocatorForMalloc(nn::mem::StandardAllocator* p) NN_NOEXCEPT;
}}

namespace movie { namespace NnMediaCommon {
    extern void Initialize();
    extern void Finalize();
}}


namespace nn{ namespace grcsrv{ namespace offscreen{

    std::atomic_uint64_t g_TestingCheckPoint;

    namespace {
        nn::os::TransferMemoryType g_TransferMemory;
    }

    nn::Result BindTransferMemory(void** pOutAddress, nn::sf::NativeHandle&& handle, size_t size) NN_NOEXCEPT
    {
        NN_GRCSRV_PROCESS_START();
        nn::os::AttachTransferMemory(&g_TransferMemory, size, handle.GetOsHandle(), handle.IsManaged());
        handle.Detach();
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyTransferMemory(&g_TransferMemory));

        NN_RESULT_THROW_UNLESS(size >= RequiredTransferMemorySize, nn::grc::ResultOutOfMemory());

        void* addr = nullptr;
        NN_RESULT_DO(nn::os::MapTransferMemory(&addr, &g_TransferMemory, nn::os::MemoryPermission_None));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::UnmapTransferMemory(&g_TransferMemory));

        NN_GRCSRV_PROCESS_SUCCESS();
        *pOutAddress = addr;
        NN_RESULT_SUCCESS;
    }

    void UnbindTransferMemory() NN_NOEXCEPT
    {
        nn::os::UnmapTransferMemory(&g_TransferMemory);
        nn::os::DestroyTransferMemory(&g_TransferMemory);
    }

    //-------------------------

    namespace {

        void* g_pMemory = nullptr;
        size_t g_MemorySize = 0;
        LibraryState* g_pState = nullptr;

        nn::os::ThreadType g_SyncpointWaiterThread;
        nn::os::ThreadType g_VideoEncoderWorkerThread;
        nn::os::ThreadType g_VideoMuxerWorkerThread;
        nn::os::ThreadType g_AudioEncoderWorkerThread;
        nn::os::ThreadType g_AudioMuxerWorkerThread;
        NN_ALIGNAS(4096) char g_SyncpointWaiterThreadStack[4096];
        NN_ALIGNAS(4096) char g_VideoEncoderWorkerThreadStack[4096];
        NN_ALIGNAS(4096) char g_VideoMuxerWorkerThreadStack[4096];
        NN_ALIGNAS(4096) char g_AudioEncoderWorkerThreadStack[4096];
        NN_ALIGNAS(4096) char g_AudioMuxerWorkerThreadStack[4096];

        nn::mem::StandardAllocator g_MultimediaAllocator;

        void* MultimediaAllocate(size_t size, size_t alignment, void*) NN_NOEXCEPT
        {
            //NN_SDK_LOG("mmalloc size=%lld, align=%lld\n", size, alignment);
            return g_MultimediaAllocator.Allocate(size, alignment);
        }

        void MultimediaFree(void* ptr, void*) NN_NOEXCEPT
        {
            //NN_SDK_LOG("mmfree ptr=%llX\n", ptr);
            g_MultimediaAllocator.Free(ptr);
        }

        void* MultimediaReallocate(void* ptr, size_t newSize, void*) NN_NOEXCEPT
        {
            //NN_SDK_LOG("mmrealloc ptr=%llX, size=%lld\n", ptr, newSize);
            return g_MultimediaAllocator.Reallocate(ptr, newSize);
        }
    }

    bool IsInitialized() NN_NOEXCEPT
    {
        return g_pState != nullptr;
    }

    nn::Result Initialize(void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(!IsInitialized(), nn::grc::ResultPreConditionViolation());

        NN_ABORT_UNLESS_RESULT_SUCCESS(nerd::scoop::InitializeResourcesOnce());

        NN_GRCSRV_PROCESS_START();
        std::memset(pMemory, 0, size);
        NN_GRCSRV_PROCESS_ROLLBACK(std::memset(pMemory, 0, size));

        char* pMemoryEnd = reinterpret_cast<char*>(pMemory) + size;
        // メモリ割り当てを作成
        void* pLibraryStateMemory = nullptr;
        void* pImageBufferMemory = nullptr;
        void* pMovieFileMemory = nullptr;
        void* pMovieMetaMemory = nullptr;
        void* pMultimediaMemory = nullptr;
        {
            nn::lmem::HeapCommonHead heapHead;
            auto hHeap = nn::lmem::CreateFrameHeap(pMemory, size, nn::lmem::CreationOption_NoOption, &heapHead);
            NN_UTIL_SCOPE_EXIT{ nn::lmem::DestroyFrameHeap(hHeap); };

            pLibraryStateMemory = nn::lmem::AllocateFromFrameHeap(hHeap, LibraryStateMemorySize, 4096);
            pImageBufferMemory  = nn::lmem::AllocateFromFrameHeap(hHeap, ImageBufferMemorySize, static_cast<int>(CaptureImageBuffer::GetRequiredMemoryAlignment()));
            pMovieFileMemory    = nn::lmem::AllocateFromFrameHeap(hHeap, MovieFileMemorySize, 4096);
            pMovieMetaMemory    = nn::lmem::AllocateFromFrameHeap(hHeap, nn::capsrv::movie::MovieMetaDataSize, 4096);
            pMultimediaMemory   = nn::lmem::AllocateFromFrameHeap(hHeap, MultimediaHeapSize, 4096);

            NN_GRCSRV_ASSERT_THROW(pLibraryStateMemory != nullptr, nn::grc::ResultOutOfMemory());
            NN_GRCSRV_ASSERT_THROW(pImageBufferMemory != nullptr, nn::grc::ResultOutOfMemory());
            NN_GRCSRV_ASSERT_THROW(pMovieFileMemory != nullptr, nn::grc::ResultOutOfMemory());
            NN_GRCSRV_ASSERT_THROW(pMovieMetaMemory != nullptr, nn::grc::ResultOutOfMemory());
            NN_GRCSRV_ASSERT_THROW(pMultimediaMemory != nullptr, nn::grc::ResultOutOfMemory());
        }

        // アロケータの初期化
        {
            // 余分なメモリもすべて multimedia のアロケータに割り当てる
            ptrdiff_t actualHeapSize = pMemoryEnd - reinterpret_cast<char*>(pMultimediaMemory);
            NN_ABORT_UNLESS_MINMAX(actualHeapSize, MultimediaHeapSize, size);
            g_MultimediaAllocator.Initialize(pMultimediaMemory, actualHeapSize);
        }
        nn::grcsrv::SetStandardAllocatorForMalloc(&g_MultimediaAllocator);
        nv::mm::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, nullptr);

        // multimedia の初期化
        movie::NnMediaCommon::Initialize();

        // SyncpointWaiter を初期化
        native::g_SyncpointWaiter.Initialize();
        NN_GRCSRV_PROCESS_ROLLBACK(native::g_SyncpointWaiter.Finalize());

        // LibraryState を作成
        NN_STATIC_ASSERT(sizeof(LibraryState) <= LibraryStateMemorySize);
        auto pState = new(pLibraryStateMemory) LibraryState();
        NN_RESULT_THROW_UNLESS(pState != nullptr, nn::grc::ResultOutOfMemory());
        NN_GRCSRV_PROCESS_ROLLBACK(pState->~LibraryState());

        // VIC 用コンテキスト
        NN_RESULT_DO(pState->InitializeCaptureModule());
        NN_GRCSRV_PROCESS_ROLLBACK(pState->FinalizeCaptureModule());

        // VIC 用バッファ
        NN_RESULT_DO(pState->m_ImageBuffer.InitializeMemoryPool(pState->m_pCaptureModule, pImageBufferMemory, ImageBufferMemorySize, RecordingVideoWidthMax, RecordingVideoHeightMax));
        NN_GRCSRV_PROCESS_ROLLBACK(pState->m_ImageBuffer.FinalizeMemoryPool());

        // 動画用ファイルシステム
        NN_RESULT_DO(pState->InitializeMovieFileSystem(pMovieFileMemory, MovieFileMemorySize));
        NN_GRCSRV_PROCESS_ROLLBACK(pState->FinalizeMovieFileSystem());

        // 動画メタデータ
        NN_RESULT_DO(pState->InitializeMovieMeta(pMovieMetaMemory, nn::capsrv::movie::MovieMetaDataSize));
        NN_GRCSRV_PROCESS_ROLLBACK(pState->FinalizeMovieMeta());

        // ワーカースレッドの作成
        NN_RESULT_DO(nn::os::CreateThread(
            &g_SyncpointWaiterThread,
            [](void*){ native::g_SyncpointWaiter.Run(); },
            nullptr,
            g_SyncpointWaiterThreadStack,
            sizeof(g_SyncpointWaiterThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(grc, OffscreenSyncpointWaiter)
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyThread(&g_SyncpointWaiterThread));
        nn::os::SetThreadNamePointer(&g_SyncpointWaiterThread, NN_SYSTEM_THREAD_NAME(grc, OffscreenSyncpointWaiter));

        // SyncpointWaiter を起動
        nn::os::StartThread(&g_SyncpointWaiterThread);

        NN_GRCSRV_PROCESS_SUCCESS();
        g_pMemory = pMemory;
        g_MemorySize = size;
        g_pState = pState;
        NN_RESULT_SUCCESS;
    }

    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(IsInitialized());
        if(!IsInitialized())
        {
            return;
        }

        // SyncpointWaiter を止める
        native::g_SyncpointWaiter.StopSync();
        nn::os::WaitThread(&g_SyncpointWaiterThread);

        // 破棄
        g_pState->FinalizeMovieMeta();
        g_pState->FinalizeMovieFileSystem();
        g_pState->m_ImageBuffer.FinalizeMemoryPool();
        g_pState->FinalizeCaptureModule();

        nn::os::DestroyThread(&g_SyncpointWaiterThread);
        native::g_SyncpointWaiter.Finalize();

        movie::NnMediaCommon::Finalize();
        g_MultimediaAllocator.Finalize();
        nn::grcsrv::SetStandardAllocatorForMalloc(nullptr);

        g_pState->~LibraryState();

        std::memset(g_pMemory, 0, g_MemorySize);
        g_pMemory = nullptr;
        g_MemorySize = 0;
        g_pState = nullptr;

        nv::FinalizeGraphics();
    }


    nn::Result CreateLayer(uint64_t* pOutHandle, nn::applet::AppletResourceUserId rendererAruid) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer == nn::util::nullopt, nn::grc::ResultOutOfResource());

        NN_GRCSRV_PROCESS_START();

        // Layer
        uint64_t h = {};
        g_pState->m_Layer.emplace();
        NN_GRCSRV_PROCESS_ROLLBACK(g_pState->m_Layer = nn::util::nullopt);

        NN_RESULT_DO(g_pState->m_Layer->Initialize(
            &h,
            rendererAruid,
            g_pState->m_pCaptureModule
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(g_pState->m_Layer->Finalize());

        NN_GRCSRV_PROCESS_SUCCESS();
        *pOutHandle = h;
        NN_RESULT_SUCCESS;
    }

    nn::Result DestroyLayer(uint64_t handle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == handle, nn::grc::ResultInvalidCall());

        g_pState->m_Layer->Finalize();
        g_pState->m_Layer = nn::util::nullopt;

        NN_RESULT_SUCCESS;
    }

    nn::Result GetLayerError(uint64_t handle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == handle, nn::grc::ResultInvalidCall());

        return g_pState->m_Layer->GetErrorResult();
    }

    nn::Result BindRendererLayer(
        int32_t* pOutBinderHandle,
        uint64_t layerHandle,
        nn::applet::AppletResourceUserId rendererAruid
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        NN_RESULT_DO(g_pState->m_Layer->BindRenderer(rendererAruid));

        int32_t binderHandle = g_pState->m_BinderHandle;

        *pOutBinderHandle = binderHandle;
        NN_RESULT_SUCCESS;
    }

    nn::Result UnbindRendererLayer(uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->IsRendererBound(), nn::grc::ResultInvalidCall());
        g_pState->m_Layer->UnbindRenderer();

        NN_RESULT_SUCCESS;
    }

    nn::Result StartLayerRecording(uint64_t layerHandle, const nn::grc::OffscreenRecordingParameter& param) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(param.CheckParameterValid(), nn::grc::ResultAlbumInvalidParameter());

        NN_GRCSRV_PROCESS_START();

        NN_GRCSRV_PROCESS_ROLLBACK({
            for(int i = 0; i < VideoCaptureImageBufferCount; i++)
            {
                if(g_pState->m_ImageBuffer.IsBufferInitialized(i))
                {
                    g_pState->m_ImageBuffer.FinalizeBuffer(i);
                }
            }
        });
        for(int i = 0; i < VideoCaptureImageBufferCount; i++)
        {
            NN_RESULT_DO(g_pState->m_ImageBuffer.InitializeBuffer(i, param.videoWidth, param.videoHeight));
        }

        nn::capsrv::AlbumFileId fileId;
        NN_RESULT_DO(ConvertCapsrvError(nn::capsrv::GenerateCurrentAlbumFileId(&fileId, param.applicationId, nn::capsrv::AlbumFileContents_ExtraMovie)));
        NN_GRCSRV_PROCESS_ROLLBACK({});

        nn::capsrv::AlbumMovieWriteStreamHandle hStream = {};
        NN_RESULT_DO(ConvertCapsrvError(nn::capsrv::OpenAlbumMovieWriteStream(&hStream, fileId)));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::capsrv::DiscardAlbumMovieWriteStream(hStream));

        NN_RESULT_DO(ConvertCapsrvError(nn::capsrv::StartAlbumMovieWriteStreamDataSection(hStream)));
        // ROLLBACK 不要。失敗したら Discard される。

        nn::fs::FileHandle hFile = {};
        NN_RESULT_DO(g_pState->AttachMovieFileData(&hFile, hStream));
        NN_GRCSRV_PROCESS_ROLLBACK(g_pState->DetachMovieFileData(hFile));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &g_VideoEncoderWorkerThread,
            Layer::VideoEncodeThreadFunction,
            &*g_pState->m_Layer,
            g_VideoEncoderWorkerThreadStack,
            sizeof(g_VideoEncoderWorkerThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(grc, OffscreenVideoEncoder)
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyThread(&g_VideoEncoderWorkerThread));
        nn::os::SetThreadNamePointer(&g_VideoEncoderWorkerThread, NN_SYSTEM_THREAD_NAME(grc, OffscreenVideoEncoder));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &g_VideoMuxerWorkerThread,
            Layer::VideoMuxerThreadFunction,
            &*g_pState->m_Layer,
            g_VideoMuxerWorkerThreadStack,
            sizeof(g_VideoMuxerWorkerThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(grc, OffscreenVideoMuxer)
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyThread(&g_VideoMuxerWorkerThread));
        nn::os::SetThreadNamePointer(&g_VideoMuxerWorkerThread, NN_SYSTEM_THREAD_NAME(grc, OffscreenVideoMuxer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &g_AudioEncoderWorkerThread,
            Layer::AudioEncodeThreadFunction,
            &*g_pState->m_Layer,
            g_AudioEncoderWorkerThreadStack,
            sizeof(g_AudioEncoderWorkerThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(grc, OffscreenAudioEncoder)
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyThread(&g_AudioEncoderWorkerThread));
        nn::os::SetThreadNamePointer(&g_AudioEncoderWorkerThread, NN_SYSTEM_THREAD_NAME(grc, OffscreenAudioEncoder));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &g_AudioMuxerWorkerThread,
            Layer::AudioMuxerThreadFunction,
            &*g_pState->m_Layer,
            g_AudioMuxerWorkerThreadStack,
            sizeof(g_AudioMuxerWorkerThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(grc, OffscreenAudioMuxer)
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(nn::os::DestroyThread(&g_AudioMuxerWorkerThread));
        nn::os::SetThreadNamePointer(&g_AudioMuxerWorkerThread, NN_SYSTEM_THREAD_NAME(grc, OffscreenAudioMuxer));

        // CPU キャッシュを Flush しておく
        g_pState->m_ImageBuffer.FlushDataCache();

        NN_RESULT_DO(g_pState->m_Layer->Start(
            hFile,
            param,
            VideoCaptureImageBufferCount,
            g_pState->m_ImageBuffer.GetGraphicBufferListRef(),
            &g_VideoEncoderWorkerThread,
            &g_VideoMuxerWorkerThread,
            &g_AudioEncoderWorkerThread,
            &g_AudioMuxerWorkerThread
        ));
        NN_GRCSRV_PROCESS_ROLLBACK(g_pState->m_Layer->Discard());

        NN_GRCSRV_PROCESS_SUCCESS();
        g_pState->m_FileId = fileId;
        g_pState->m_StreamHandle = hStream;
        g_pState->m_FileHandle = hFile;
        NN_RESULT_SUCCESS;
    }

    nn::Result AbortLayerRecording(uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        if(g_pState->m_Layer->IsStarted())
        {
            g_pState->m_Layer->Discard();
        }

        if(g_pState->m_FileHandle)
        {
            auto hFile = *g_pState->m_FileHandle;
            (void)nn::fs::FlushFile(hFile);
            g_pState->DetachMovieFileData(hFile);
            g_pState->m_FileHandle = nn::util::nullopt;
        }

        if(g_pState->m_StreamHandle)
        {
            auto hStream = *g_pState->m_StreamHandle;
            nn::capsrv::DiscardAlbumMovieWriteStream(hStream);
            g_pState->m_FileId = nn::util::nullopt;
            g_pState->m_StreamHandle = nn::util::nullopt;
        }

        nn::os::DestroyThread(&g_AudioMuxerWorkerThread);
        nn::os::DestroyThread(&g_AudioEncoderWorkerThread);
        nn::os::DestroyThread(&g_VideoMuxerWorkerThread);
        nn::os::DestroyThread(&g_VideoEncoderWorkerThread);

        for(int i = 0; i < VideoCaptureImageBufferCount; i++)
        {
            g_pState->m_ImageBuffer.FinalizeBuffer(i);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result RequestFinishLayerRecording(uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        NN_RESULT_DO(g_pState->m_Layer->RequestFinishReady());
        NN_RESULT_SUCCESS;
    }

    nn::Result FinishLayerRecording(uint64_t layerHandle, const capsrv::ApplicationData& applicationData, const sf::InBuffer& thumbnailImage) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV("finishing layer recording...\n");
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->IsStarted(), nn::grc::ResultInvalidCall());
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingPrecondition);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  checking is layer finish ready\n");
        NN_RESULT_DO(g_pState->m_Layer->GetErrorResult()); // エラー発生時も FinishReady イベントがシグナルするのでエラーチェックが必要
        NN_RESULT_DO(g_pState->m_Layer->CheckFinishReady());
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingCheckLayerFinishReady);

        auto& layer  = *g_pState->m_Layer;
        auto hStream = *g_pState->m_StreamHandle;
        auto hFile   = *g_pState->m_FileHandle;

        // 最終バッファのインデックスを取得
        int videoFrameCount;
        int lastBufferIndex;
        nn::grc::OffscreenRecordingParameter param;
        {
            videoFrameCount = layer.GetVideoFrameCount();
            lastBufferIndex = layer.GetLastCaptureBufferIndex();
            param = layer.GetRunningParameter();
            NN_GRCSRV_OFFSCRN_LOG_DEV("    video frame count = %d\n", videoFrameCount);
            NN_GRCSRV_OFFSCRN_LOG_DEV("    last buffer index = %d\n", lastBufferIndex);
            NN_GRCSRV_ASSERT_THROW(videoFrameCount > 0, nn::grc::ResultInternalOffscreenLayerError());
            NN_GRCSRV_ASSERT_THROW(lastBufferIndex >= 0 && lastBufferIndex < VideoCaptureImageBufferCount, nn::grc::ResultInternalOffscreenLayerError());
            NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingGetLastCaptureBufferIndex);
        }

        // 動画サイズのチェック
        const auto movieWidth  = g_pState->m_ImageBuffer.GetBufferWidth(lastBufferIndex);
        const auto movieHeight = g_pState->m_ImageBuffer.GetBufferHeight(lastBufferIndex);
        auto movieSize = capsrv::ScreenShotSize_1280x720;
        if (movieWidth == 1280 && movieHeight == 720)
        {
            // Pass
        }
        else if (movieWidth == 1920 && movieHeight == 1080)
        {
            movieSize = capsrv::ScreenShotSize_1920x1080;
        }
        else
        {
            // 通常、ここで落ちることはない
            NN_SDK_ASSERT(false, "Invalid movieWidth=%d or movieHeight=%d", movieWidth, movieHeight);
        }

        // 最終バッファが 720p でなければ 720p にスケーリングする
        if(movieWidth != 1280 || movieHeight != 720)
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV("  scale last buffer to 720p\n");
            int shadowIndex = (lastBufferIndex + 1) % VideoCaptureImageBufferCount;
            g_pState->m_ImageBuffer.FinalizeBuffer(shadowIndex);
            NN_ABORT_UNLESS_RESULT_SUCCESS(g_pState->m_ImageBuffer.InitializeBuffer(shadowIndex, 1280, 720));

            // コピー
            android::sp<android::Fence> fence;
            NN_ABORT_UNLESS_RESULT_SUCCESS(VicCopyCaptureImageBuffer(
                &fence,
                g_pState->m_ImageBuffer.GetGraphicBufferListRef()[shadowIndex],
                g_pState->m_ImageBuffer.GetGraphicBufferListRef()[lastBufferIndex],
                g_pState->m_pCaptureModule
            ));
            fence->wait(android::Fence::TIMEOUT_NEVER);

            lastBufferIndex = shadowIndex;
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV("  finish layer\n");
        NN_RESULT_DO(g_pState->m_Layer->Finish());
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingFinishLayer);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  flush file\n");
        NN_RESULT_DO(ConvertCapsrvError(nn::fs::FlushFile(hFile)));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingFlushFile);

        // 最終的なデータサイズを取得
        int64_t movieDataSize = 0;
        {
            NN_RESULT_DO(nn::fs::GetFileSize(&movieDataSize, hFile));
            NN_GRCSRV_OFFSCRN_LOG_DEV("    movie data size = %lld\n", movieDataSize);
            NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingGetMovieDataSize);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV("  detach file\n");
        {
            g_pState->DetachMovieFileData(hFile);
            g_pState->m_FileHandle = nn::util::nullopt;
            NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingDetachFile);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV("  end data section\n");
        NN_RESULT_DO(nn::capsrv::EndAlbumMovieWriteStreamDataSection(hStream));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingEndDataSection);

        // メタ情報を準備
        auto& builder = g_pState->m_MovieMeta.m_Builder;
        NN_GRCSRV_OFFSCRN_LOG_DEV("  building meta...\n");
        NN_GRCSRV_OFFSCRN_LOG_DEV("    initialize builder\n");
        builder.Initialize(
            nn::capsrv::movie::MovieMetaDataVersion_1,
            g_pState->m_MovieMeta.m_pMemory,
            g_pState->m_MovieMeta.m_MemorySize,
            *g_pState->m_FileId
        );
        NN_UTIL_SCOPE_EXIT{
            builder.Finalize();
        };
        {
            NN_GRCSRV_OFFSCRN_LOG_DEV("    set data size %lld\n", movieDataSize);
            builder.SetMovieDataSize(static_cast<size_t>(movieDataSize));

            NN_GRCSRV_OFFSCRN_LOG_DEV("    set last image\n");

            if (thumbnailImage.GetSize() > 0)
            {
                // サムネイルの独自指定がある場合
                builder.SetImageDataRgba(
                    thumbnailImage.GetPointerUnsafe(),
                    thumbnailImage.GetSize(),
                    1280,
                    720
                );
            }
            else
            {
                // サムネイルの独自指定がない場合
                builder.SetImageDataNv12(
                    g_pState->m_ImageBuffer.GetBufferMemoryNv12Y(lastBufferIndex),
                    g_pState->m_ImageBuffer.GetBufferMemorySizeNv12Y(lastBufferIndex),
                    g_pState->m_ImageBuffer.GetBufferMemoryNv12Uv(lastBufferIndex),
                    g_pState->m_ImageBuffer.GetBufferMemorySizeNv12Uv(lastBufferIndex),
                    g_pState->m_ImageBuffer.GetBufferWidth(lastBufferIndex),
                    g_pState->m_ImageBuffer.GetBufferHeight(lastBufferIndex)
                );
            }

            NN_GRCSRV_OFFSCRN_LOG_DEV("    set description\n");
            builder.SetDescription(param.description);

            NN_GRCSRV_OFFSCRN_LOG_DEV("    set attribute(s)\n");
            builder.SetAttribute(
                videoFrameCount,
                1,
                param.videoFrameRate,
                videoFrameCount * 1000 / param.videoFrameRate,
                param.videoFrameCountBetweenIdr,
                false,
                movieSize,
                param.videoImageOrientation
            );

            NN_GRCSRV_OFFSCRN_LOG_DEV("    set applicationData\n");
            builder.SetApplicationData(applicationData);

            NN_GRCSRV_OFFSCRN_LOG_DEV("    build\n");
            NN_RESULT_DO(builder.Build());
            NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingBuildMetaData);
        }
        NN_GRCSRV_OFFSCRN_LOG_DEV("  built meta\n");

        NN_GRCSRV_OFFSCRN_LOG_DEV("  start meta section\n");
        NN_RESULT_DO(nn::capsrv::StartAlbumMovieWriteStreamMetaSection(hStream));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingStartMetaSection);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  write meta\n");
        NN_RESULT_DO(g_pState->m_MovieMeta.m_Builder.WriteToStream(hStream));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingWriteMetaData);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  end meta section\n");
        NN_RESULT_DO(nn::capsrv::EndAlbumMovieWriteStreamMetaSection(hStream));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingEndMetaSection);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  finish\n");
        NN_RESULT_DO(nn::capsrv::FinishAlbumMovieWriteStream(hStream));
        NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingFinishStream);

        NN_GRCSRV_OFFSCRN_LOG_DEV("  commit\n");
        {
            NN_RESULT_DO(nn::capsrv::CommitAlbumMovieWriteStream(hStream));
            g_pState->m_FileId = nn::util::nullopt;
            g_pState->m_StreamHandle = nn::util::nullopt;
            NN_GRCSRV_THROW_CHECKPOINT(CheckPointForTesting::CheckPoint_FinishLayerRecordingCommitStream);
        }

        // 以降失敗しない
        nn::os::DestroyThread(&g_AudioMuxerWorkerThread);
        nn::os::DestroyThread(&g_AudioEncoderWorkerThread);
        nn::os::DestroyThread(&g_VideoMuxerWorkerThread);
        nn::os::DestroyThread(&g_VideoEncoderWorkerThread);

        for(int i = 0; i < VideoCaptureImageBufferCount; i++)
        {
            g_pState->m_ImageBuffer.FinalizeBuffer(i);
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV("finished layer recording\n");
        NN_RESULT_SUCCESS;
    }// NOLINT(impl/function_size)

    nn::Result EncodeLayerAudioSample(size_t* pOutSize, uint64_t layerHandle, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        NN_RESULT_DO(g_pState->m_Layer->EncodeAudioSample(pOutSize, buffer, size));

        NN_RESULT_SUCCESS;
    }

    nn::Result GetLayerRecordingFinishReadyEvent(nn::sf::NativeHandle& outHandle, uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        g_pState->m_Layer->GetFinishReadySystemEventHandle(outHandle);

        NN_RESULT_SUCCESS;
    }

    nn::Result GetLayerAudioEncodeReadyEvent(nn::sf::NativeHandle& outHandle, uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_GRCSRV_ASSERT_THROW(IsInitialized(), nn::grc::ResultPreConditionViolation());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(g_pState->m_Layer->GetHandle() == layerHandle, nn::grc::ResultInvalidCall());

        g_pState->m_Layer->GetAudioEncodeReadySystemEventHandle(outHandle);

        NN_RESULT_SUCCESS;
    }


    //------------------------------------------------

    void TransactParcel(
        std::int32_t handle,
        std::uint32_t code,
        const nn::sf::InBuffer& requestBuffer,
        const nn::sf::OutBuffer& replyBuffer,
        std::uint32_t flags
        ) NN_NOEXCEPT
    {
        android::Parcel request;
        android::Parcel reply;
        size_t pos = 0;
        size_t writtenSize = 0;

        if(!IsInitialized())
        {
            goto FAIL;
        }

        if(g_pState->m_Layer == nn::util::nullopt)
        {
            goto FAIL;
        }

        if(handle != g_pState->m_BinderHandle)
        {
            goto FAIL;
        }

        if(!g_pState->m_Layer->IsRendererBound())
        {
            goto FAIL;
        }

        if(native::ParcelIo::OpenParcel(&request, &pos, requestBuffer.GetPointerUnsafe(), requestBuffer.GetSize()) != android::NO_ERROR)
        {
            goto FAIL;
        }

        NN_GRCSRV_OFFSCRN_LOG_DEV_TRANSACT("code(%d)\n", code);
        if(g_pState->m_Layer->GetRendererProducer()->asBinder()->transact(code, request, &reply, flags) != android::NO_ERROR)
        {
            goto FAIL;
        }

        native::ParcelIo::WriteParcel(&writtenSize, replyBuffer.GetPointerUnsafe(), replyBuffer.GetSize(), &reply);
        if(writtenSize == 0)
        {
            goto FAIL;
        }

        return;

    FAIL:
        std::memset(replyBuffer.GetPointerUnsafe(), 0, replyBuffer.GetSize());
        return;
    }

    void AdjustRefcount(
        std::int32_t handle,
        std::int32_t diff,
        std::int32_t isStrong
        ) NN_NOEXCEPT
    {
        NN_UNUSED(handle);
        NN_UNUSED(diff);
        NN_UNUSED(isStrong);
    }

    void GetNativeHandle(
        std::int32_t handle,
        std::uint32_t code,
        nn::sf::Out<nn::sf::NativeHandle> result
        ) NN_NOEXCEPT
    {
        nn::os::NativeHandle h = {};
        bool isManaged = false;

        if(!IsInitialized())
        {
            goto FAIL;
        }

        if(g_pState->m_Layer == nn::util::nullopt)
        {
            goto FAIL;
        }

        if(handle != g_pState->m_BinderHandle)
        {
            goto FAIL;
        }

        if(!g_pState->m_Layer->IsRendererBound())
        {
            goto FAIL;
        }

        g_pState->m_Layer->GetRendererProducer()->asBinder()->getNativeHandle(code, h, isManaged);
        result.Set(nn::sf::NativeHandle(h, isManaged));
        return;

    FAIL:
        return;
    }

    void SetCheckPointForTesting(std::uint64_t value) NN_NOEXCEPT
    {
        g_TestingCheckPoint = value;
    }

}}}

