﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/util/util_Optional.h>
#include <nn/grc/grc_CommonTypes.h>
#include <nvnflinger_service.h>

#include "grcsrvOffscreen_Config.h"
#include "grcsrvOffscreen_MultiWaitHandler.h"
#include "util/grcsrv_TQueue.h"
#include "util/grcsrv_TNullableContainer.h"
#include "../../capsrv/capture/capsrv_CaptureModule.h"
#include "native/grcsrv_SyncpointWaiter.h"

#include "detail/grcsrvOffscreen_VideoRendererHandler.h"
#include "detail/grcsrvOffscreen_VideoCaptureHandler.h"
#include "detail/grcsrvOffscreen_VideoEncoderHandler.h"
#include "detail/grcsrvOffscreen_AudioRendererHandler.h"
#include "detail/grcsrvOffscreen_AudioEncoderHandler.h"
#include "detail/grcsrvOffscreen_MuxerHandler.h"

namespace nn{ namespace grcsrv{ namespace offscreen{

    namespace detail{
        struct LayerEncodingBuffer
        {
        public:
            // NullableContainer 用。無効値を返す。
            struct InvalidValue
            {
                static LayerEncodingBuffer Get() NN_NOEXCEPT
                {
                    LayerEncodingBuffer v = {};
                    v.captureBufferIndex = -1;
                    return v;
                }
            };

        public:
            bool operator==(const LayerEncodingBuffer& v) const NN_NOEXCEPT
            {
                return (this->captureBufferIndex == v.captureBufferIndex)
                    && (this->timestamp == v.timestamp);
            }
            bool operator!=(const LayerEncodingBuffer& v) const NN_NOEXCEPT
            {
                return !(*this == v);
            }

        public:
            int captureBufferIndex;
            android::sp<android::GraphicBuffer> captureBuffer;
            int64_t timestamp;
        };

        typedef util::TNullableContainer<LayerEncodingBuffer, VideoCaptureImageBufferCount, LayerEncodingBuffer::InvalidValue, util::ContainerThreadSafetyOption_MultiThread> LayerEncodingBufferList;
    }

    class Layer
    {
    public:
        Layer() NN_NOEXCEPT;

        nn::Result Initialize(
            uint64_t* pOutHandle,
            nn::applet::AppletResourceUserId rendererAruid,
            nn::capsrv::capture::CaptureModule* pCaptureModule
        ) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

    private:
        nn::Result InitializeRendererObject(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;
        void FinalizeRendererObject() NN_NOEXCEPT;

        nn::Result InitializeVideoBufferObject(int count, android::sp<android::GraphicBuffer>* list) NN_NOEXCEPT;
        void FinalizeVideoBufferObject() NN_NOEXCEPT;

    public:
        // @pre true
        uint64_t GetHandle() const NN_NOEXCEPT;

        // @pre true
        nn::Result GetErrorResult() const NN_NOEXCEPT;

        // @pre IsStarted
        nn::fs::FileHandle GetFileHandle() const NN_NOEXCEPT;

        // @pre true
        bool IsInitialized() const NN_NOEXCEPT;

        // @pre true
        bool IsRendererBound() const NN_NOEXCEPT;

        // @pre true
        bool IsStarted() const NN_NOEXCEPT;

        // @pre IsInitialized
        int GetVideoFrameCount() const NN_NOEXCEPT;

        // @pre IsInitialized
        int GetLastCaptureBufferIndex() const NN_NOEXCEPT;

        // @pre IsInitialized
        // State == Idle の場合デフォルト値を返す。
        nn::grc::OffscreenRecordingParameter GetRunningParameter() const NN_NOEXCEPT;

        // @pre true
        nn::applet::AppletResourceUserId GetRendererAruid() const NN_NOEXCEPT;

        // @pre IsRendererBound
        android::sp<android::IGraphicBufferProducer> GetRendererProducer() NN_NOEXCEPT;

        // @pre Initialized
        void GetFinishReadySystemEventHandle(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT;

        // @pre Initialized
        void GetAudioEncodeReadySystemEventHandle(nn::sf::NativeHandle& outHandle) NN_NOEXCEPT;

        // @pre !IsRendererBound
        // @post IsRendererBound
        nn::Result BindRenderer(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // @pre IsRendererBound
        // @post !IsStarted
        // @post !IsRendererBound
        void UnbindRenderer() NN_NOEXCEPT;

        // @pre !IsStarted
        // @pre IsRendererBound
        // @post IsStarted
        nn::Result Start(
            nn::fs::FileHandle hFile,
            const nn::grc::OffscreenRecordingParameter& param,
            int videoBufferCount,
            android::sp<android::GraphicBuffer>* videoBufferList,
            nn::os::ThreadType* pVideoEncodeThread,
            nn::os::ThreadType* pVideoMuxerThread,
            nn::os::ThreadType* pAudioEncodeThread,
            nn::os::ThreadType* pAudioMuxerThread
        ) NN_NOEXCEPT;

        // @pre IsStarted
        // @pst !IsStarted
        void Discard() NN_NOEXCEPT;

        nn::Result RequestFinishReady() NN_NOEXCEPT;
        nn::Result CheckFinishReady() const NN_NOEXCEPT;
        nn::Result Finish() NN_NOEXCEPT;

        nn::Result EncodeAudioSample(size_t* pOutSize, const void* buffer, size_t size) NN_NOEXCEPT;

    private:
        // Discard() と Finish() の内部実装。
        // エンコードで使用した状態をすべてリセットして Idle に戻す。
        // 全 Handler とワーカースレッドを停止させてから呼び出す。
        void CleanupAndChangeStateIdleImpl() NN_NOEXCEPT;

    private:
        void UnlinkExitObserverImpl() NN_NOEXCEPT;
        void LinkExitObserverAsErrorDetectorImpl() NN_NOEXCEPT;
        void LinkExitObserverAsFinishWaiterImpl() NN_NOEXCEPT;

        void ErrorDetectorExitWaiterImpl(nn::util::optional<nn::Result> exitResult) NN_NOEXCEPT;
        void ExitObserverFinishWaiterImpl(nn::util::optional<nn::Result> exitResult) NN_NOEXCEPT;
        void CheckFinishReadyCallbackImpl() NN_NOEXCEPT;

    public:
        static void VideoEncodeThreadFunction(void* pLayer) NN_NOEXCEPT;
        static void VideoMuxerThreadFunction(void* pLayer) NN_NOEXCEPT;
        static void AudioEncodeThreadFunction(void* pLayer) NN_NOEXCEPT;
        static void AudioMuxerThreadFunction(void* pLayer) NN_NOEXCEPT;

    private:
        void VideoEncodeThreadFunctionImpl() NN_NOEXCEPT;
        void VideoMuxerThreadFunctionImpl() NN_NOEXCEPT;
        void AudioEncodeThreadFunctionImpl() NN_NOEXCEPT;
        void AudioMuxerThreadFunctionImpl() NN_NOEXCEPT;

    private:
        enum class LayerState
        {
            Idle,
            Running,
            Finishing,
            FinishReady,
            Error,
        };

        LayerState m_State;
        uint64_t m_Handle;
        nn::util::optional<nn::Result> m_ErrorResult;
        nn::fs::FileHandle m_FileHandle;
        nn::grc::OffscreenRecordingParameter m_Parameter;

        // ワーカースレッド
        nn::os::ThreadType* m_pVideoEncodeThread;
        nn::os::ThreadType* m_pVideoMuxerThread;
        nn::os::ThreadType* m_pAudioEncodeThread;
        nn::os::ThreadType* m_pAudioMuxerThread;
        nn::util::optional<nn::Result> m_VideoEncodeThreadResult;
        nn::util::optional<nn::Result> m_VideoMuxerThreadResult;
        nn::util::optional<nn::Result> m_AudioEncodeThreadResult;
        nn::util::optional<nn::Result> m_AudioMuxerThreadResult;

        struct RendererState
        {
            bool m_IsBound;
            nn::applet::AppletResourceUserId m_Aruid;
            android::sp<android::IGraphicBufferProducer> m_pProducer;
            android::sp<android::IGraphicBufferConsumer> m_pConsumer;
            nn::os::SystemEventType m_AcquirableEvent;
        } m_Renderer;

        struct VideoBuffer
        {
            int m_Count;
            android::sp<android::GraphicBuffer>* m_List;
        } m_VideoBuffer;

        detail::LayerEncodingBufferList m_EncodingBufferList;
        detail::VideoRendererToVideoCaptureQueue m_VideoRendererToVideoCaptureQueue;
        detail::VideoCaptureToVideoRendererQueue m_VideoCaptureToVideoRendererQueue;
        detail::VideoEncoderToVideoCaptureQueue m_VideoEncoderToVideoCaptureQueue;
        detail::VideoCaptureToVideoEncoderQueue m_VideoCaptureToVideoEncoderQueue;
        detail::AudioRendererToAudioEncoderQueue m_AudioRendererToAudioEncoderQueue;
        detail::AudioEncoderToAudioRendererQueue m_AudioEncoderToAudioRendererQueue;

        detail::VideoRendererHandler m_VideoRenderer;
        detail::VideoCaptureHandler  m_VideoCapture;
        detail::VideoEncoderHandler  m_VideoEncoder;
        detail::AudioRendererHandler m_AudioRenderer;
        detail::AudioEncoderHandler  m_AudioEncoder;
        detail::MuxerHandler         m_Muxer;

        nn::os::SystemEventType m_FinishReadyEvent;

        // ExitObserver
        MultiWaitHandler m_VideoRendererExitWaitHandler;
        MultiWaitHandler m_VideoCaptureExitWaitHandler;
        MultiWaitHandler m_AudioRendererExitWaitHandler;
        MultiWaitHandler m_VideoEncodeThreadExitWaitHandler;
        MultiWaitHandler m_VideoMuxerThreadExitWaitHandler;
        MultiWaitHandler m_AudioEncodeThreadExitWaitHandler;
        MultiWaitHandler m_AudioMuxerThreadExitWaitHandler;
    };

}}}
