﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/os.h>
#include <nvnflinger_service.h>
#include "../native/grcsrv_SyncpointWaiter.h"
#include "../util/grcsrv_TQueue.h"
#include "../grcsrvOffscreen_Config.h"
#include "../grcsrvOffscreen_MultiWaitHandler.h"
#include "../../../capsrv/capture/capsrv_CaptureModule.h"
#include "grcsrvOffscreen_QueueType.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    class VideoCaptureHandler
    {
    public:
        enum class State
        {
            NotInitialized,
            Idle,
            WaitingForInputBuffer,
            WaitingForOutputBuffer,
            Exited,
        };

    public:
        VideoCaptureHandler() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;
        nn::util::optional<nn::Result> GetExitResult() const NN_NOEXCEPT;
        int GetCapturedFrameCount() const NN_NOEXCEPT;
        int GetLastDestinationBufferIndex() const NN_NOEXCEPT;

        nn::Result Initialize(
            nn::capsrv::capture::CaptureModule* pCaptureModule,
            detail::VideoRendererToVideoCaptureQueue* pAcquirableInputQueue,
            detail::VideoCaptureToVideoRendererQueue* pReleasedInputQueue,
            detail::VideoEncoderToVideoCaptureQueue*  pAcquirableOutputQueue,
            detail::VideoCaptureToVideoEncoderQueue*  pQueuedOutputQueue
        ) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        // State::Idle -> State::Running
        nn::Result Start() NN_NOEXCEPT;

        // State::* -> State::Idle
        void Abort() NN_NOEXCEPT;

        // State::Exited -> State::Idle
        void Reset() NN_NOEXCEPT;

        void InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        void FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    private:
        void ChangeStateWaitingForInputBufferImpl() NN_NOEXCEPT;
        void ChangeStateWaitingForOutputBufferImpl() NN_NOEXCEPT;
        void ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT;
        void ExitCurrentStateImpl() NN_NOEXCEPT;

        // コピーを実行する。
        // この関数は State を変更しない。
        // 終了する場合、有効な値を返す。
        nn::util::optional<nn::Result> ExecuteImpl() NN_NOEXCEPT;

        nn::Result CopyBufferImpl(
            android::sp<android::Fence>* pOutCopyFence,
            const android::sp<android::GraphicBuffer>& dstBuffer,
            const android::sp<android::GraphicBuffer>& srcBuffer,
            const android::IGraphicBufferConsumer::BufferItem& srcItem
        ) NN_NOEXCEPT;

    private:
        State m_State;

        nn::util::optional<nn::Result> m_ExitResult;
        nn::os::EventType m_ExitEvent;

        int m_CapturedFrameCount;
        int m_LastDestinationBufferIndex;

        nn::capsrv::capture::CaptureModule* m_pCaptureModule;

        detail::VideoRendererToVideoCaptureQueue* m_pAcquirableInputQueue;
        detail::VideoCaptureToVideoRendererQueue* m_pReleasedInputQueue;
        detail::VideoEncoderToVideoCaptureQueue* m_pAcquirableOutputQueue;
        detail::VideoCaptureToVideoEncoderQueue* m_pQueuedOutputQueue;

        MultiWaitHandler m_InputAcquirableWaitHandler;
        MultiWaitHandler m_OutputAcquirableWaitHandler;
    };

}}}}
