﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrvOffscreen_VideoCaptureHandler.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/grc/grc_Result.h>
#include <nn/grc/grc_ResultPrivate.h>

#include "../grcsrvOffscreen_Macro.h"
#include "../grcsrvOffscreen_ResourceId.h"
#include "../grcsrvOffscreen_MultiWaitHandler.h"
#include "../grcsrvOffscreen_VicCopyCaptureImageBuffer.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    VideoCaptureHandler::VideoCaptureHandler() NN_NOEXCEPT
        : m_State(State::NotInitialized)
    {
    }

    bool VideoCaptureHandler::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != State::NotInitialized;
    }

    nn::util::optional<nn::Result> VideoCaptureHandler::GetExitResult() const NN_NOEXCEPT
    {
        return m_ExitResult;
    }

    int VideoCaptureHandler::GetCapturedFrameCount() const NN_NOEXCEPT
    {
        return m_CapturedFrameCount;
    }

    int VideoCaptureHandler::GetLastDestinationBufferIndex() const NN_NOEXCEPT
    {
        return m_LastDestinationBufferIndex;
    }

    nn::Result VideoCaptureHandler::Initialize(
        nn::capsrv::capture::CaptureModule* pCaptureModule,
        detail::VideoRendererToVideoCaptureQueue* pAcquirableInputQueue,
        detail::VideoCaptureToVideoRendererQueue* pReleasedInputQueue,
        detail::VideoEncoderToVideoCaptureQueue*  pAcquirableOutputQueue,
        detail::VideoCaptureToVideoEncoderQueue*  pQueuedOutputQueue
    ) NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("initializing video-capture...\n");
        NN_SDK_REQUIRES(!IsInitialized());

        NN_GRCSRV_PROCESS_START();

        NN_GRCSRV_PROCESS_SUCCESS(); // 以降失敗しない

        nn::os::InitializeEvent(&m_ExitEvent, false, nn::os::EventClearMode_ManualClear);
        pAcquirableInputQueue->InitializeDequeueableMultiWaitHolder(m_InputAcquirableWaitHandler.GetHolder());
        pAcquirableOutputQueue->InitializeDequeueableMultiWaitHolder(m_OutputAcquirableWaitHandler.GetHolder());

        m_ExitResult = nn::util::nullopt;
        m_CapturedFrameCount = 0;
        m_LastDestinationBufferIndex = -1;
        m_pCaptureModule = pCaptureModule;
        m_pAcquirableInputQueue = pAcquirableInputQueue;
        m_pReleasedInputQueue = pReleasedInputQueue;
        m_pAcquirableOutputQueue = pAcquirableOutputQueue;
        m_pQueuedOutputQueue = pQueuedOutputQueue;
        m_State = State::Idle;
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("initialized video-capture\n");
        NN_RESULT_SUCCESS;
    }

    void VideoCaptureHandler::Finalize() NN_NOEXCEPT
    {
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("finalizing video-capture...\n");
        NN_SDK_REQUIRES(IsInitialized());
        NN_GRCSRV_ASSERT_ABORT(m_State == State::Idle);

        m_InputAcquirableWaitHandler.Unlink();
        m_OutputAcquirableWaitHandler.Unlink();
        m_pAcquirableInputQueue->FinalizeDequeueableMultiWaitHolder(m_InputAcquirableWaitHandler.GetHolder());
        m_pAcquirableOutputQueue->FinalizeDequeueableMultiWaitHolder(m_OutputAcquirableWaitHandler.GetHolder());

        nn::os::FinalizeEvent(&m_ExitEvent);

        m_ExitResult = nn::util::nullopt;
        m_CapturedFrameCount = 0;
        m_LastDestinationBufferIndex = -1;
        m_pCaptureModule = nullptr;
        m_pAcquirableInputQueue = nullptr;
        m_pReleasedInputQueue = nullptr;
        m_pAcquirableOutputQueue = nullptr;
        m_pQueuedOutputQueue = nullptr;
        m_State = State::NotInitialized;
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("finalized video-capture...\n");
    }

    nn::Result VideoCaptureHandler::Start() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(m_State == State::Idle);

        m_ExitResult = nn::util::nullopt;
        m_CapturedFrameCount = 0;
        m_LastDestinationBufferIndex = -1; // 初期値は -1
        ChangeStateWaitingForInputBufferImpl();
        NN_RESULT_SUCCESS;
    }

    void VideoCaptureHandler::Abort()  NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        m_InputAcquirableWaitHandler.Unlink();
        m_OutputAcquirableWaitHandler.Unlink();

        m_ExitResult = nn::util::nullopt;
        m_CapturedFrameCount = 0;
        m_LastDestinationBufferIndex = -1;
        m_State = State::Idle;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoCaptureHandler::Reset() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(m_State == State::Exited);

        // 事実上 Abort と同義
        Abort();
    }

    void VideoCaptureHandler::InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        nn::os::InitializeMultiWaitHolder(pHolder, &m_ExitEvent);
    }

    void VideoCaptureHandler::FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        nn::os::FinalizeMultiWaitHolder(pHolder);
    }

    //-------------------------------------------------------------------

    void VideoCaptureHandler::ChangeStateWaitingForInputBufferImpl() NN_NOEXCEPT
    {
        ExitCurrentStateImpl();
        m_InputAcquirableWaitHandler.Link(
            [](MultiWaitHandler*, void* usrPtr)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("input buffer acquirable\n");
                auto pSelf = reinterpret_cast<VideoCaptureHandler*>(usrPtr);
                NN_SDK_ASSERT_EQUAL(pSelf->m_State, State::WaitingForInputBuffer);

                pSelf->ChangeStateWaitingForOutputBufferImpl();
            },
            this
        );

        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("  ->waiting for input buffer\n");
        m_State = State::WaitingForInputBuffer;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoCaptureHandler::ChangeStateWaitingForOutputBufferImpl() NN_NOEXCEPT
    {
        ExitCurrentStateImpl();
        m_OutputAcquirableWaitHandler.Link(
            [](MultiWaitHandler*, void* usrPtr)->void
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("output buffer acquirable\n");
                auto pSelf = reinterpret_cast<VideoCaptureHandler*>(usrPtr);
                NN_SDK_ASSERT_EQUAL(pSelf->m_State, State::WaitingForOutputBuffer);

                auto exitResult = pSelf->ExecuteImpl();

                if(exitResult == nn::util::nullopt)
                {
                    pSelf->ChangeStateWaitingForInputBufferImpl();
                }
                else
                {
                    pSelf->ChangeStateExitedImpl(*exitResult);
                }
            },
            this
        );

        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("  ->waiting for output buffer\n");
        m_State = State::WaitingForOutputBuffer;
        nn::os::ClearEvent(&m_ExitEvent);
    }

    void VideoCaptureHandler::ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT
    {
        ExitCurrentStateImpl();

        m_ExitResult = exitResult;
        m_State = State::Exited;
        nn::os::SignalEvent(&m_ExitEvent);
    }

    void VideoCaptureHandler::ExitCurrentStateImpl() NN_NOEXCEPT
    {
        m_InputAcquirableWaitHandler.Unlink();
        m_OutputAcquirableWaitHandler.Unlink();
    }

    //-------------------------------------------------------------------

    nn::util::optional<nn::Result> VideoCaptureHandler::ExecuteImpl() NN_NOEXCEPT
    {
        nn::Result result = nn::ResultSuccess();

        detail::VideoRendererToVideoCaptureValue sourceBuffer = {};
        m_pAcquirableInputQueue->Dequeue(&sourceBuffer);
        NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("dequeued video-renderer buffer(index=%d)\n", sourceBuffer.index);

        if(sourceBuffer.isFinishRequested)
        {
            // 1 フレームもコピーしていなかったら不正
            if(m_CapturedFrameCount == 0 || m_LastDestinationBufferIndex < 0)
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("no captured buffer\n");
                return nn::util::make_optional<nn::Result>(nn::grc::ResultInternalOffscreenVideoCaptureError());
            }

            detail::VideoCaptureToVideoEncoderValue e = {};
            e.isFinishRequested = true;
            m_pQueuedOutputQueue->Enqueue(e);

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("exit successfully\n");
            return nn::util::make_optional<nn::Result>(nn::ResultSuccess());
        }
        else
        {
            detail::VideoEncoderToVideoCaptureValue destinationBuffer = {};
            m_pAcquirableOutputQueue->Dequeue(&destinationBuffer);
            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("acquired video-capture buffer(index=%d)\n", destinationBuffer.index);

            NN_SDK_REQUIRES_NOT_NULL(sourceBuffer.buffer);
            NN_SDK_REQUIRES_NOT_NULL(destinationBuffer.buffer);

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("waiting for source fence\n");
            sourceBuffer.item.mFence->wait(android::Fence::TIMEOUT_NEVER);

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("copying video-renderer buffer to video-capture buffer\n");
            android::sp<android::Fence> fence = android::Fence::NO_FENCE;
            result = CopyBufferImpl(
                &fence,
                destinationBuffer.buffer,
                sourceBuffer.buffer,
                sourceBuffer.item
            );
            if(result.IsFailure())
            {
                NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("failed to copy buffer\n");
                return nn::util::make_optional<nn::Result>(nn::grc::ResultInternalOffscreenVideoCaptureError());
            }

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("waiting for copy fence\n");
            fence->wait(android::Fence::TIMEOUT_NEVER);

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("sending input buffer to video-renderer\n");
            {
                detail::VideoCaptureToVideoRendererValue e = {};
                e.index = sourceBuffer.index;
                m_pReleasedInputQueue->Enqueue(e);
            }

            NN_GRCSRV_OFFSCRN_LOG_DEV_VIDEOCAPTURE("sending output buffer to video-encoder\n");
            {
                detail::VideoCaptureToVideoEncoderValue e = {};
                e.index             = destinationBuffer.index;
                e.buffer            = destinationBuffer.buffer;
                e.timestampUs       = sourceBuffer.timestampUs;
                m_pQueuedOutputQueue->Enqueue(e);
            }

            // フレーム数とコピー先バッファを更新
            m_CapturedFrameCount++;
            m_LastDestinationBufferIndex = destinationBuffer.index;
        }

        return nn::util::nullopt;
    }


    //-------------------------------------------------------------------

    nn::Result VideoCaptureHandler::CopyBufferImpl(
        android::sp<android::Fence>* pOutCopyFence,
        const android::sp<android::GraphicBuffer>& dstBuffer,
        const android::sp<android::GraphicBuffer>& srcBuffer,
        const android::IGraphicBufferConsumer::BufferItem& srcItem
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(dstBuffer);
        NN_SDK_REQUIRES_NOT_NULL(srcBuffer);

        android::sp<android::Fence> outFence = android::Fence::NO_FENCE;
        NN_RESULT_DO(VicCopyCaptureImageBuffer(
            &outFence,
            dstBuffer,
            srcBuffer,
            srcItem,
            m_pCaptureModule
        ));

        *pOutCopyFence = outFence;
        NN_RESULT_SUCCESS;
    }

}}}}

