﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>

NN_PRAGMA_PUSH_WARNINGS
#pragma warning(disable:4091) // 宣言されていない typedef
#include <dbghelp.h>
NN_PRAGMA_POP_WARNINGS

#include <sstream>

#include <nn/gfxTool/gfxTool_BackTrace.h>
#include <nn/gfxTool/gfxTool_DllBase.h>

namespace nn {
namespace gfxTool {

class BackTraceDll
    : public DllBase
{
public:
    static const BackTraceDll& GetInstance();

    BackTraceDll();

    bool Initialize();

    virtual void Finalize() override;

    HANDLE GetProcess() const
    {
        return m_hProcess;
    }

    decltype( &SymInitialize ) SymbolInitialize;
    decltype( &SymCleanup ) SymbolCleanup;
    decltype( &SymSetOptions ) SymbolSetOptions;

    decltype( &SymFromAddr ) SymbolFromAddr;
    decltype( &SymGetLineFromAddr ) SymbolGetLineFromAddr;
    decltype( &SymGetModuleInfo ) SymbolGetModuleInfo;

private:
    HANDLE m_hProcess;
};

const BackTraceDll& BackTraceDll::GetInstance()
{
    static BackTraceDll s_BackTrace;
    if( !s_BackTrace.IsInitialized() )
    {
        s_BackTrace.Initialize();
    }
    return s_BackTrace;
}

BackTraceDll::BackTraceDll()
    : DllBase()
    , m_hProcess()
{
}

bool BackTraceDll::Initialize()
{
    if( !DllBase::Initialize( "Dbghelp.dll" ) )
    {
        return false;
    }

    SymbolInitialize = GetRequiredFunction< decltype( SymbolInitialize ) >( "SymInitialize" );
    SymbolCleanup = GetRequiredFunction< decltype( SymbolCleanup ) >( "SymCleanup" );
    SymbolSetOptions = GetRequiredFunction< decltype( SymbolSetOptions ) >( "SymSetOptions" );
    SymbolFromAddr = GetRequiredFunction< decltype( SymbolFromAddr ) >( "SymFromAddr" );
#if defined( _M_IX86 )
    SymbolGetLineFromAddr = GetRequiredFunction< decltype( SymbolGetLineFromAddr ) >( "SymGetLineFromAddr" );
    SymbolGetModuleInfo = GetRequiredFunction< decltype( SymbolGetModuleInfo ) >( "SymGetModuleInfo" );
#else
    SymbolGetLineFromAddr = GetRequiredFunction< decltype( SymbolGetLineFromAddr ) >( "SymGetLineFromAddr64" );
    SymbolGetModuleInfo = GetRequiredFunction< decltype( SymbolGetModuleInfo ) >( "SymGetModuleInfo64" );
#endif

    m_hProcess = GetCurrentProcess();

    SymbolSetOptions( SYMOPT_DEFERRED_LOADS | SYMOPT_LOAD_LINES | SYMOPT_UNDNAME );
    if( SymbolInitialize( m_hProcess, nullptr, TRUE ) )
    {
        return false;
    }

    return true;
}

void BackTraceDll::Finalize()
{
    SymbolCleanup( m_hProcess );

    DllBase::Finalize();
}

Custom< std::string >::Type GetNameFromAddress( void* address )
{
    DWORD64 displacement;
    char buffer[ sizeof( SYMBOL_INFO ) + MAX_SYM_NAME ];
    auto pSymbol = reinterpret_cast< PSYMBOL_INFO >( buffer );
    pSymbol->SizeOfStruct = sizeof( SYMBOL_INFO );
    pSymbol->MaxNameLen = MAX_SYM_NAME;

    auto& backTraceDll = BackTraceDll::GetInstance();
    if( backTraceDll.SymbolFromAddr( backTraceDll.GetProcess(),
        reinterpret_cast< DWORD64 >( address ), &displacement, pSymbol ) )
    {
        return Custom< std::string >::Type( pSymbol->Name );
    }

    return "";
}

Custom< std::string >::Type GetLineFromAddress( void* address )
{
    DWORD displacement;
    auto& backTraceDll = BackTraceDll::GetInstance();
    IMAGEHLP_LINE line;
    if( backTraceDll.SymbolGetLineFromAddr( backTraceDll.GetProcess(),
        reinterpret_cast< DWORD64 >( address ), &displacement, &line ) )
    {
        Custom< std::stringstream >::Type ss;
        ss << line.FileName;
        ss << "(" << line.LineNumber << ")";
        return ss.str();
    }

    return "";
}

Custom< std::string >::Type GetModuleNameFromAddress( void* address )
{
    auto& backTraceDll = BackTraceDll::GetInstance();
    IMAGEHLP_MODULE moduleInfo = { sizeof( IMAGEHLP_MODULE ) };
    if( backTraceDll.SymbolGetModuleInfo( backTraceDll.GetProcess(),
        reinterpret_cast<DWORD64>( address ), &moduleInfo ) )
    {
        return moduleInfo.ModuleName;
    }
    return "";
}

}
}
