﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_TransientBufferAllocator.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn{ namespace gfx{ namespace util{

    TransientBufferAllocator::TransientBufferAllocator() NN_NOEXCEPT
        : m_pBuffer(NULL)
        , m_OffsetRingBuffer()
        , m_AllocatableAlignmentMax(0)
        , m_pHistory(NULL)
        , m_HistoryMemorySize(0)
        , m_HistoryRingSize(0)
        , m_HistoryHead(0)
        , m_HistoryTail(0)
        , m_IsInitialized(false)
    {
    }

    void TransientBufferAllocator::Initialize(
        nn::gfx::Buffer* pBuffer,
        ptrdiff_t baseOffset,
        size_t size,
        size_t allocatableAlignmentMax,
        int historyLengthMax
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(baseOffset, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, static_cast<size_t>(1));
        NN_SDK_REQUIRES_LESS_EQUAL(size, static_cast<size_t>(SizeMax));
        NN_SDK_REQUIRES_GREATER(allocatableAlignmentMax, static_cast<size_t>(0));
        NN_SDK_REQUIRES_LESS_EQUAL(allocatableAlignmentMax, static_cast<size_t>(AlignmentMax));
        NN_SDK_REQUIRES((allocatableAlignmentMax & (allocatableAlignmentMax - 1)) == 0);
        NN_SDK_REQUIRES(baseOffset % allocatableAlignmentMax == 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(historyLengthMax, 1);
        NN_SDK_REQUIRES_NOT_NULL(m_pHistory);
        NN_SDK_REQUIRES_ALIGNED(m_pHistory, GetRequiredMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(m_HistoryMemorySize, GetRequiredMemorySize(historyLengthMax));

        m_pBuffer = pBuffer;
        m_OffsetRingBuffer.Initialize(baseOffset, static_cast<ptrdiff_t>(size));
        m_AllocatableAlignmentMax = static_cast<ptrdiff_t>(allocatableAlignmentMax);
        m_HistoryRingSize = historyLengthMax + 1;
        m_HistoryHead = 0;
        m_HistoryTail = m_HistoryRingSize - 1;

        m_IsInitialized = true;
    }
    void TransientBufferAllocator::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        m_pBuffer = NULL;
        m_OffsetRingBuffer.Finalize();
        m_IsInitialized = false;
    }

    bool TransientBufferAllocator::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    bool TransientBufferAllocator::IsRecording() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_OffsetRingBuffer.IsRecording();
    }

    size_t TransientBufferAllocator::GetRequiredMemorySize(int historyLengthMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(historyLengthMax, 1);
        return sizeof(detail::OffsetRange) * (static_cast<size_t>(historyLengthMax) + 1);
    }

    size_t TransientBufferAllocator::GetRequiredMemoryAlignment() NN_NOEXCEPT
    {
        return NN_ALIGNOF(detail::OffsetRange);
    }

    void TransientBufferAllocator::SetMemory(void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        m_pHistory = static_cast<detail::OffsetRange*>(pMemory);
        m_HistoryMemorySize = size;
    }

    nn::gfx::Buffer* TransientBufferAllocator::GetBuffer() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_pBuffer;
    }
    ptrdiff_t TransientBufferAllocator::GetBaseOffset() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_OffsetRingBuffer.GetBaseOffset();
    }
    size_t TransientBufferAllocator::GetSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return static_cast<size_t>(m_OffsetRingBuffer.GetSize());
    }

    size_t TransientBufferAllocator::GetAllocatableAlignmentMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return static_cast<size_t>(m_AllocatableAlignmentMax);
    }

    void TransientBufferAllocator::Begin() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_LESS(GetHistoryLength(), GetHistoryLengthMax());
        m_OffsetRingBuffer.Begin();
    }

    void TransientBufferAllocator::End() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_ASSERT_LESS(GetHistoryLength(), GetHistoryLengthMax());
        m_OffsetRingBuffer.End(&m_pHistory[m_HistoryHead]);
        m_HistoryHead = (m_HistoryHead + 1) % m_HistoryRingSize;
    }

    void TransientBufferAllocator::Free() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_GREATER(GetHistoryLength(), 0);
        m_HistoryTail = (m_HistoryTail + 1) % m_HistoryRingSize;
        m_OffsetRingBuffer.ReleaseOffsetRange(&m_pHistory[m_HistoryTail]);
    }

    ptrdiff_t TransientBufferAllocator::Allocate(size_t size, size_t alignment) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_LESS_EQUAL(alignment, static_cast<size_t>(m_AllocatableAlignmentMax));
        NN_SDK_REQUIRES((alignment & (alignment - 1)) == 0);
        NN_SDK_REQUIRES(m_AllocatableAlignmentMax == 0 || alignment <= static_cast<size_t>(m_AllocatableAlignmentMax));
        if(size > SizeMax)
        {
            return InvalidOffset;
        }
        return m_OffsetRingBuffer.Allocate(static_cast<ptrdiff_t>(size), static_cast<ptrdiff_t>(alignment));
    }

    int TransientBufferAllocator::GetHistoryLengthMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_HistoryRingSize - 1;
    }

    int TransientBufferAllocator::GetHistoryLength() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        if(m_HistoryTail < m_HistoryHead)
        {
            return m_HistoryHead - m_HistoryTail - 1;
        }
        else
        {
            return m_HistoryRingSize - 1 - m_HistoryTail + m_HistoryHead;
        }
    }

    void TransientBufferAllocator::FillHistory() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(!m_OffsetRingBuffer.IsRecording());
        while(GetHistoryLength() < GetHistoryLengthMax())
        {
            Begin();
            End();
        }
    }

}}}
