﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/nn_SdkAssert.h>


namespace nn {
namespace gfx {
namespace util {


DescriptorPoolAllocator::DescriptorPoolAllocator() NN_NOEXCEPT :
m_Impl(),
m_pDescriptorPool(nullptr),
m_BaseSlotIndex(0),
m_SlotCount(0),
m_IsInitialized(false)
{}

void DescriptorPoolAllocator::Initialize(
    nn::mem::MallocCallback pAllocateFunction,
    void* pAllocateFunctionUserData,
    nn::mem::FreeCallback pFreeFunction,
    void* pFreeFunctionUserData,
    nn::gfx::DescriptorPool* pDescriptorPool,
    int baseSlotIndex,
    int slotCount,
    bool isThreadSafe
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_IsInitialized);
    NN_SDK_REQUIRES_NOT_NULL(pAllocateFunction);
    NN_SDK_REQUIRES_NOT_NULL(pFreeFunction);
    NN_SDK_REQUIRES_NOT_NULL(pDescriptorPool);
    NN_SDK_REQUIRES_GREATER_EQUAL(baseSlotIndex, 0);
    NN_SDK_REQUIRES_MINMAX(slotCount, 1, static_cast<int>(SlotCountMax));

    m_pDescriptorPool = pDescriptorPool;
    m_BaseSlotIndex = baseSlotIndex;
    m_SlotCount = slotCount;

    m_Impl.Initialize(pAllocateFunction, pAllocateFunctionUserData, pFreeFunction, pFreeFunctionUserData, isThreadSafe);
    m_Impl.AddRange(0, static_cast<int>(slotCount));
    m_IsInitialized = true;
}

void DescriptorPoolAllocator::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    m_Impl.Finalize();
    m_pDescriptorPool = nullptr;
    m_IsInitialized = false;
}

bool DescriptorPoolAllocator::IsInitialized() const NN_NOEXCEPT
{
    return m_IsInitialized;
}

int DescriptorPoolAllocator::Allocate(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    int indexUnit;
    if(!m_Impl.Allocate(&indexUnit, count)){
        return InvalidIndex;
    }
    return m_BaseSlotIndex + static_cast<int>(indexUnit);
}

void DescriptorPoolAllocator::Free(int indexSlot) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    if (indexSlot == InvalidIndex) {
        return;
    }

    int indexUnit = static_cast<int>(indexSlot - m_BaseSlotIndex);
    m_Impl.Free(indexUnit);
}

nn::gfx::DescriptorPool* DescriptorPoolAllocator::GetDescriptorPool() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_pDescriptorPool;
}

int DescriptorPoolAllocator::GetBaseSlotIndex() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_BaseSlotIndex;
}

int DescriptorPoolAllocator::GetSlotCount() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_SlotCount;
}

int DescriptorPoolAllocator::GetSizeOf(int indexSlot) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    if (indexSlot == InvalidIndex) {
        return 0;
    }

    int indexUnit = static_cast<int>(indexSlot - m_BaseSlotIndex);
    return m_Impl.GetSizeOf(indexUnit);
}

int DescriptorPoolAllocator::GetTotalFreeSize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_Impl.GetTotalFreeSize();
}

int DescriptorPoolAllocator::GetAllocatableSize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    return m_Impl.GetAllocatableSize();
}

void DescriptorPoolAllocator::Dump() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);

    m_Impl.Dump(m_BaseSlotIndex, 1);
}


} // namespace util
} // namespace gfx
} // namespace nn
